package com.floreantpos.model.dao;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Patient;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginationSupport;

public class PatientDAO extends BasePatientDAO {

	public PatientDAO() {
	}

	@Override
	protected void delete(Object obj, Session session) {
		Patient patient = (Patient) obj;
		if (patient == null) {
			throw new PosException("Patient not found");
		}

		patient.setDeleted(Boolean.TRUE);
		saveOrUpdate(patient, session);
	}

	public void findByNameOrPhoneOrDob(String patientName, Date patientDOB, String patientMobile, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotBlank(patientName)) {
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, patientName, MatchMode.ANYWHERE));
			}
			if (StringUtils.isNotBlank(patientMobile)) {
				criteria.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, patientMobile, MatchMode.ANYWHERE));
			}
			if (patientDOB != null) {
				criteria.add(Restrictions.between(Customer.PROP_DATE_OF_BIRTH, DateUtil.startOfDay(patientDOB), DateUtil.endOfDay(patientDOB)));
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public void findByPhoneOrName(String searchString, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();

				Criterion dobCriteria = Restrictions.ilike(Customer.PROP_PROPERTIES, "\"paitent.dob.date\":\"%" + searchString + "%\"", MatchMode.ANYWHERE); //$NON-NLS-1$ //$NON-NLS-2$
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_LOYALTY_NO, searchString, MatchMode.ANYWHERE), dobCriteria));
				criteria.add(disjunction);
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public List<Patient> findByPhoneOrName(String searchString) {
		try (Session session = createNewSession()) {
			return findByPhoneOrName(searchString, session);
		}
	}

	public List<Patient> findByPhoneOrName(String searchString, Session session) {
		Criteria criteria = session.createCriteria(Patient.class);

		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

		if (StringUtils.isNotEmpty(searchString)) {
			Disjunction disjunction = Restrictions.disjunction();
			Criterion dobCriteria = Restrictions.ilike(Customer.PROP_PROPERTIES, "\"paitent.dob.date\":\"%" + searchString + "%\"", MatchMode.ANYWHERE); //$NON-NLS-1$ //$NON-NLS-2$
			disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
					Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE),
					Restrictions.ilike(Customer.PROP_MEMBER_ID, searchString, MatchMode.ANYWHERE), dobCriteria));
			criteria.add(disjunction);
		}
		criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());

		List<Patient> result = criteria.list();
		if (result == null || result.isEmpty()) {
			return null;
		}
		return result;
	}

}