package com.floreantpos.report;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang.WordUtils;
import org.apache.commons.lang3.StringUtils;

import com.floreantpos.model.BloodGroupType;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Department;
import com.floreantpos.model.GlobalConfig;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.GlobalConfigDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.PaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;
import pl.allegro.finance.tradukisto.MoneyConverters;

public class LabTestInvoiceUtil {

	public static JasperPrint createJasperPrint(Ticket ticket) throws JRException {
		return createJasperPrint(ticket, null);
	}

	public static JasperPrint createJasperPrint(Ticket ticket, Map<String, Object> mapForPreview) throws JRException {
		LabTestReportModel reportModel = createModels(ticket);

		HashMap<String, Object> map = new HashMap<String, Object>();
		Outlet outlet = DataProvider.get().getOutlet();

		PaperSize paperSize;
		if (mapForPreview == null) {
			String receiptPaperSize = GlobalConfigDAO.getInstance().getProperty(GlobalConfig.RECEIPT_PAPER_SIZE, PaperSize.A4.name());
			paperSize = PaperSize.fromString(receiptPaperSize);
		}
		else {
			paperSize = (PaperSize) mapForPreview.get("paperSize");
		}

		reportModel.setPaperSize(paperSize);
		ReportUtil.populateMedlogicsProperties(map, outlet, paperSize);

		JasperReport header;
		if (paperSize == PaperSize.A5) {
			header = ReportUtil.getReport("medlogics_report_customer_header_order_info_A5"); //$NON-NLS-1$	
		}
		else {
			header = ReportUtil.getReport("medlogics_report_customer_header_order_info"); //$NON-NLS-1$	
		}

		map.put("head", header); //$NON-NLS-1$
		map.put("reportTitle", "Lab test report"); //$NON-NLS-1$
		populateReportParameter(map, ticket);

		Store store = DataProvider.get().getStore();
		int receiptReportHeaderMargin = store.getReceiptReportHeaderMargin();
		int receiptReportFooterMargin = store.getReceiptReportFooterMargin();
		JasperReport masterReport;
		if (mapForPreview != null) {
			map.putAll(mapForPreview);
			receiptReportHeaderMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptHeaderMarginValue"));
			receiptReportFooterMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptFooterMarginValue"));
		}
		if (paperSize == PaperSize.A5) {
			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-report_A5", receiptReportFooterMargin, receiptReportHeaderMargin);
		}
		else {
			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-report", receiptReportFooterMargin, receiptReportHeaderMargin);
		}

		JasperPrint jasperPrint = JasperFillManager.fillReport(masterReport, map, new JRTableModelDataSource(reportModel));
		return jasperPrint;
	}

	public static String getMoneyIntoWords(String input) {
		MoneyConverters converter = MoneyConverters.ENGLISH_BANKING_MONEY_VALUE;
		return WordUtils.capitalize(converter.asWords(new BigDecimal(input)));
	}

	private static void populateReportParameter(HashMap<String, Object> map, Ticket ticket) {
		Customer patient = ticket.getCustomer();
		Store store = DataProvider.get().getStore();
		String departmentName = ""; //$NON-NLS-1$
		String doctorId = ticket.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			String departmentId = DoctorDAO.getInstance().get(doctorId).getDoctorDepartmentId();
			Department department = DataProvider.get().getDepartmentById(departmentId);
			if (department != null) {
				departmentName = department.getName();
			}
		}

		String refName = "";
		String referrerId = ticket.getReferrerId();
		if (StringUtils.isNotBlank(referrerId)) {
			Customer referrer = CustomerDAO.getInstance().get(referrerId);
			if (referrer != null) {
				refName = referrer.getName();
			}
		}

		Boolean doVisibleReceipttHeader = POSUtil.getBoolean(GlobalConfigDAO.getInstance().getProperty(GlobalConfig.RECEIPT_PRINT_HEADER), true); //$NON-NLS-1$
		map.put("visibleReportHeader", doVisibleReceipttHeader); //$NON-NLS-1$

		map.put("bill", "Bill"); //$NON-NLS-1$
		map.put("barcode", ticket.getId()); //$NON-NLS-1$
		map.put("billNo", ticket.getId()); //$NON-NLS-1$
		map.put("date", DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(ticket.getCreateDate()), "dd MMM yy hh:mm a")); //$NON-NLS-1$
		map.put("dateReceived", DateUtil.simplifyDateAndTime(ticket.getDeliveryDate(), "dd MMM yy hh:mm a")); //$NON-NLS-1$
		map.put("labId", store.getUuid()); //$NON-NLS-1$

		map.put("patientId", patient.getId()); //$NON-NLS-1$
		map.put("ptName", patient.getName()); //$NON-NLS-1$
		map.put("phone", patient.getMobileNo()); //$NON-NLS-1$
		map.put("sex", patient.getPatientGender()); //$NON-NLS-1$
		Date dateOfBirth = patient.getDateOfBirth();
		if (dateOfBirth != null) {
			map.put("age", DateUtil.calculateAge(dateOfBirth)); //$NON-NLS-1$
		}

		map.put("consultant", ticket.getDoctorName()); //$NON-NLS-1$
		map.put("department", departmentName); //$NON-NLS-1$
		String bloodGroup = patient.getBloodGroup();
		if (StringUtils.isNotBlank(bloodGroup)) {
			BloodGroupType bloodGroupType = BloodGroupType.fromNameString(bloodGroup);
			if (bloodGroupType != null) {
				bloodGroup = bloodGroupType.getDisplayString();
			}
			else {
				bloodGroup = "";
			}
		}
		map.put("bloodGroup", bloodGroup); //$NON-NLS-1$
		map.put("refBy", refName); //$NON-NLS-1$

		map.put("colNameOfTest", "<b>Item</b>"); //$NON-NLS-1$
		map.put("colQty", "<b>Quantity</b>"); //$NON-NLS-1$
		map.put("colAmount", "<b>Amount</b>"); //$NON-NLS-1$

		map.put("grandTotal", "Grand total"); //$NON-NLS-1$
		map.put("colBillAmount", "Bill amount:");
		map.put("billAmount", NumberUtil.formatAmount(ticket.getSubtotalAmount()));

		map.put("colTax", "Tax:");
		map.put("tax", NumberUtil.formatAmount(ticket.getTaxAmount()));

		map.put("colSpecialDiscount", "Discount:");
		map.put("specialDiscount", NumberUtil.formatAmount(ticket.getDiscountAmount()));

		map.put("colTotalAmount", "Total Amount:");
		map.put("totalAmount", NumberUtil.formatAmount(ticket.getTotalAmount()));

		//		map.put("colReceviedAmount", "Received Amount  :");
		//		map.put("receviedAmount", NumberUtil.formatAmount(ticket.getPaidAmount()));

		map.put("colDueAmount", "Due Amount:");
		map.put("dueAmount", NumberUtil.formatAmount(ticket.getDueAmount()));

		//map.put("colCancelAmount", "Cancel Amount  :");
		//map.put("cancelAmount", NumberUtil.formatAmount(ticket.getc())); 

		String serverName = ticket.getOwner().getFullName();
		if (StringUtils.isBlank(serverName)) {
			serverName = "";
		}
		map.put("colBillBy",
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd MMM yy hh:mm a") + " "
						+ "by" + " " + serverName);

		String payModeDtl = null;
		Set<PosTransaction> transactionList = ticket.getTransactions();
		if (transactionList != null) {
			List<PosTransaction> transactions = new ArrayList<>(transactionList.stream().filter(t -> !t.isVoided()).collect(Collectors.toList()));
			if (transactions != null && !transactions.isEmpty()) {
				payModeDtl = "<div style='text-align: right;'>"; //$NON-NLS-1$
				for (PosTransaction posTransaction : transactions) {
					payModeDtl += (posTransaction.buildPaymentTypeDisplayName() + ": " + NumberUtil.getCurrencyFormat(posTransaction.getAmount()) + "<br>");
				}
				payModeDtl += "</div>";
			}
		}

		map.put("colPayModeDtl", payModeDtl);

		//		map.put("colRemarks", "Remarks:");
		//		map.put("remarks", ticket.getNote());
		map.put("colPrint", "Print: " + DateUtil.formatDateWithTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp())));

		Double totalAmount = ticket.getTotalAmount();
		String pAmount = totalAmount.toString();
		if (!NumberUtil.isFractional(totalAmount)) {
			pAmount = String.valueOf(totalAmount.intValue());
		}

		String words = getMoneyIntoWords(pAmount);
		map.put("colInWord", "In word: " + words);
		map.put("qrCode", store.getCustomerLoginURL()); //$NON-NLS-1$
		map.put("qrCodeBottomMsg", "Scan the QR code to access the report");

		Boolean paid = ticket.isPaid();
		if (paid) {
			map.put("paidMsg", "Paid");
		}
	}

	private static LabTestReportModel createModels(Ticket ticket) {
		LabTestReportModel reportModel = new LabTestReportModel();
		List<TicketItem> ticketItems = ticket.getTicketItems();
		Comparator<TicketItem> productTypeComparator = Comparator.comparing(TicketItem::getProductType, Comparator.nullsLast(Comparator.reverseOrder()));
		Comparator<TicketItem> comparator = Comparator.comparing(TicketItem::getCategoryName, Comparator.nullsFirst(Comparator.naturalOrder()));
		ticketItems.sort(productTypeComparator.thenComparing(comparator));
		//ticketItems.sort(comparator.thenComparing(productTypeComparator));
		reportModel.setRows(ticketItems);
		return reportModel;
	}
}
