package com.floreantpos.util;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.text.SimpleDateFormat;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.id.IdentifierGenerator;

import com.floreantpos.PosLog;
import com.floreantpos.model.SequenceNumber;
import com.floreantpos.model.TicketIdSequence;
import com.floreantpos.model.dao.SequenceNumberDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.util.DataProvider;

public class TicketIdGenerator implements IdentifierGenerator {
	private static SimpleDateFormat yearMonthDay = new SimpleDateFormat("yyMMdd"); //$NON-NLS-1$
	private static SimpleDateFormat yearWeek = new SimpleDateFormat("yyww"); //$NON-NLS-1$

	@Override
	public Serializable generate(SessionImplementor session, Object object) throws HibernateException {
		Class<? extends Object> clazz = object.getClass();
		Serializable generatedId = null;
		try {
			Method method = clazz.getMethod("getId", (Class<?>[]) null); //$NON-NLS-1$
			if (method != null) {
				Object id = method.invoke(object, (Object[]) null);
				if (id != null) {
					generatedId = (Serializable) id;
				}
			}

			if (generatedId == null) {
				generatedId = generateTicketId((Session) session);
			}
		} catch (Exception e) {
			PosLog.error(TicketIdGenerator.class, e);
		}

		return generatedId;
	}

	public static String generateTicketId(Session session) {
		return generateTicketId(session, DataProvider.get().getOutletId());
	}

	public static String generateTicketId(String outletId) {
		try (Session session = TicketDAO.getInstance().createNewSession()) {
			return generateTicketId(session, outletId);
		}
	}

	public static String generateTicketId(Session session, String outletId) {
		TicketIdSequence i = TicketIdSequence.DateWithNumber;
		String id = generateTicketId(i, session);
		if (StringUtils.isBlank(outletId)) {
			outletId = DataProvider.get().getOutletId();
		}

		while (TicketDAO.getInstance().get(id, outletId, session) != null) {
			id = generateTicketId(i, session);
		}
		return id;
	}

	public static String generateTicketId(TicketIdSequence ticketIdSequence, Session session) {
		if (ticketIdSequence == TicketIdSequence.SequenceNumber) {
			return getSequenceNumber(session);
		}
		else if (ticketIdSequence == TicketIdSequence.DateWithNumber) {
			return yearMonthDay.format(StoreDAO.getServerTimestamp()) + getSequenceNumber(session);
		}
		else if (ticketIdSequence == TicketIdSequence.WeekWithNumber) {
			return yearWeek.format(StoreDAO.getServerTimestamp()) + getSequenceNumber(session);
		}
		else {
			return NumericGlobalIdGenerator.generateGlobalId();
		}
	}

	private static String getSequenceNumber(Session session) {
		if (session == null) {
			return ""; //$NON-NLS-1$
		}
		int nextSequenceNumber = SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.TICKET_SEQUENCE_NUMBER, session);
		//		String formatNumber = DataProvider.get().getTicketIdSequenceFormat().format(nextSequenceNumber);
		return nextSequenceNumber + ""; //$NON-NLS-1$
	}

}
