/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.ui.views.payment;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.POSUtil;

import net.authorize.Environment;
import net.authorize.Merchant;
import net.authorize.TransactionType;
import net.authorize.aim.Transaction;
import net.authorize.aim.cardpresent.Result;
import net.authorize.api.contract.v1.CreateTransactionRequest;
import net.authorize.api.contract.v1.CreateTransactionResponse;
import net.authorize.api.contract.v1.CreditCardType;
import net.authorize.api.contract.v1.MerchantAuthenticationType;
import net.authorize.api.contract.v1.MessageTypeEnum;
import net.authorize.api.contract.v1.PaymentType;
import net.authorize.api.contract.v1.TransactionRequestType;
import net.authorize.api.contract.v1.TransactionResponse;
import net.authorize.api.contract.v1.TransactionTypeEnum;
import net.authorize.api.controller.CreateTransactionController;
import net.authorize.api.controller.base.ApiOperationBase;
import net.authorize.data.creditcard.CardType;
import net.authorize.data.creditcard.CreditCard;
import us.fatehi.magnetictrack.BankCardMagneticTrack;

public class AuthorizeDotNetProcessor implements CardProcessor {
	private static final String AUTHORIZE_NET_TRANS_KEY = "authorize.net.trans.key"; //$NON-NLS-1$
	private static final String AUTHORIZE_NET_LOGIN_ID = "authorize.net.login.id"; //$NON-NLS-1$

	public void preAuth(PosTransaction transaction) throws Exception {
		Merchant merchant = createMerchant(transaction);

		CreditCard creditCard = createCard(transaction);

		// Create transaction
		Transaction authCaptureTransaction = merchant.createAIMTransaction(TransactionType.AUTH_ONLY, new BigDecimal(transaction.calculateAuthorizeAmount()));
		authCaptureTransaction.setCreditCard(creditCard);

		Result<Transaction> result = (Result<Transaction>) merchant.postTransaction(authCaptureTransaction);

		if (result.isApproved()) {
			transaction.setCardTransactionId(result.getTransId());
			transaction.setCardAuthCode(result.getAuthCode());
			transaction.setAuthorizable(true);

			populateCardInformation(transaction, result);
		}
		else if (result.isDeclined()) {
			throw new PosException(/*Messages.getString("AuthorizeDotNetProcessor.2") +*/ result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
		else {
			throw new PosException(/*Messages.getString("AuthorizeDotNetProcessor.3") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
	}

	@Override
	public void chargeAmount(PosTransaction transaction) throws Exception {
		Merchant merchant = createMerchant(transaction);

		CreditCard creditCard = createCard(transaction);

		// Create transaction
		Transaction authCaptureTransaction = merchant.createAIMTransaction(TransactionType.AUTH_CAPTURE, new BigDecimal(transaction.getAmount()));
		authCaptureTransaction.setCreditCard(creditCard);

		Result<Transaction> result = (Result<Transaction>) merchant.postTransaction(authCaptureTransaction);
		if (result.isApproved()) {
			transaction.setCaptured(false);
			transaction.setAuthorizable(false);
			transaction.setCardTransactionId(result.getTransId());
			transaction.setCardAuthCode(result.getAuthCode());

			populateCardInformation(transaction, result);
		}
		else if (result.isDeclined()) {
			throw new PosException(/*Messages.getString("AuthorizeDotNetProcessor.2") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
		else {
			throw new PosException(/*Messages.getString("AuthorizeDotNetProcessor.3") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
	}

	public void captureAuthAmount(PosTransaction transaction) throws Exception {
		Merchant merchant = createMerchant(transaction);

		// Create transaction
		Transaction authCaptureTransaction = merchant.createAIMTransaction(TransactionType.PRIOR_AUTH_CAPTURE, new BigDecimal(transaction.getAmount()));
		authCaptureTransaction.setTransactionId(transaction.getCardTransactionId());

		Result<Transaction> result = (Result<Transaction>) merchant.postTransaction(authCaptureTransaction);
		if (result.isApproved()) {
			transaction.setCardTransactionId(result.getTransId());
			transaction.setCardAuthCode(result.getAuthCode());
			transaction.setCaptured(true);

			populateCardInformation(transaction, result);
		}
		else if (result.isDeclined()) {
			throw new PosException(/*Messages.getString("AuthorizeDotNetProcessor.6") +*/ result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
		else {
			throw new PosException(/*Messages.getString("AuthorizeDotNetProcessor.7") +*/ result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
	}

	public void captureNewAmount(PosTransaction transaction) throws Exception {
		Merchant merchant = createMerchant(transaction);

		CreditCard creditCard = createCard(transaction);

		// Create transaction
		Transaction authCaptureTransaction = merchant.createAIMTransaction(TransactionType.AUTH_CAPTURE, new BigDecimal(transaction.getAmount()));
		authCaptureTransaction.setCreditCard(creditCard);

		Result<Transaction> result = (Result<Transaction>) merchant.postTransaction(authCaptureTransaction);
		if (result.isApproved()) {
			transaction.setCardTransactionId(result.getTransId());
			transaction.setCardAuthCode(result.getAuthCode());
			transaction.setCaptured(true);

			populateCardInformation(transaction, result);
		}
		else if (result.isDeclined()) {
			throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.8") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
		else {
			throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.9") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
	}

	public void voidAmount(PosTransaction transaction) throws Exception {
		Merchant merchant = createMerchant(transaction);

		// Create transaction
		Transaction authCaptureTransaction = merchant.createAIMTransaction(TransactionType.VOID, new BigDecimal(transaction.getAmount()));
		authCaptureTransaction.setTransactionId(transaction.getCardTransactionId());

		Result<Transaction> result = (Result<Transaction>) merchant.postTransaction(authCaptureTransaction);
		if (result.isApproved()) {
			transaction.setCardTransactionId(result.getTransId());
			transaction.setCardAuthCode(result.getAuthCode());
			transaction.setVoided(true);

			populateCardInformation(transaction, result);
		}
		else if (result.isDeclined()) {
			throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.10") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
		else {
			throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.11") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
	}

	private CreditCard createCard(PosTransaction transaction) {
		CreditCard creditCard = CreditCard.createCreditCard();
		String cardType = transaction.getCardType();
		creditCard.setCardType(CardType.findByValue(cardType));

		if (StringUtils.isNotEmpty(transaction.getCardTrack())) {
			return createCard(transaction.getCardTrack(), cardType);

		}
		else {
			String cardNumber = transaction.getCardNumber();
			String cardExpMonth = transaction.getCardExpMonth();
			String cardExpYear = transaction.getCardExpYear();

			validateCreditCardInfo(cardNumber, cardExpMonth, cardExpYear);

			return createCard(cardNumber, cardExpMonth, cardExpYear, cardType);
		}
	}

	private void validateCreditCardInfo(String cardNumber, String cardExpMonth, String cardExpYear) {
		List<String> inputList = Arrays.asList(cardNumber, cardExpMonth, cardExpYear);
		for (String input : inputList) {
			if (StringUtils.isEmpty(input)) {
				throw new PosException(Messages.getString("AuthorizeDotNetProcessor.16")); //$NON-NLS-1$
			}
		}
	}

	private CreditCard createCard(String cardTrack, String cardType) {
		CreditCard creditCard = CreditCard.createCreditCard();
		creditCard.setCardType(CardType.findByValue(cardType));

		//%B4111111111111111^SHAH/RIAR^1803101000000000020000831000000?;4111111111111111=1803101000020000831?
		String[] tracks = cardTrack.split(";"); //$NON-NLS-1$

		creditCard.setTrack1(tracks[0]);
		if (tracks.length > 1) {
			creditCard.setTrack2(";" + tracks[1]); //$NON-NLS-1$
		}

		return creditCard;
	}

	private CreditCard createCard(String cardNumber, String expMonth, String expYear, String cardType) {
		CreditCard creditCard = CreditCard.createCreditCard();
		creditCard.setCardType(CardType.findByValue(cardType));

		creditCard.setExpirationYear(expYear);
		creditCard.setExpirationMonth(expMonth);
		creditCard.setCreditCardNumber(cardNumber);

		return creditCard;
	}

	private Merchant createMerchant(PosTransaction transaction) throws Exception {
		String apiLoginID = transaction.getProperty(AUTHORIZE_NET_LOGIN_ID, "");
		String transactionKey = transaction.getProperty(AUTHORIZE_NET_TRANS_KEY, "");
		Environment environment = createEnvironment();

		if (POSUtil.getBoolean(transaction.getProperty(AppConstants.ONLINE_TRANSACTION))) {
			Outlet outlet = DataProvider.get().getOutlet();
			apiLoginID = outlet.getProperty(Store.PROP_ONLINE_ORDER_AUTHNET_API_LOGIN_ID);
			transactionKey = outlet.getProperty(Store.PROP_ONLINE_ORDER_AUTHNET_MERCHANTE_ID);
			environment = createEnvironmentFromOutlet(outlet);
		}
		Merchant merchant = Merchant.createMerchant(environment, apiLoginID, transactionKey);
		merchant.setDeviceType(net.authorize.DeviceType.VIRTUAL_TERMINAL);
		merchant.setMarketType(net.authorize.MarketType.RETAIL);

		transaction.addProperty(AUTHORIZE_NET_LOGIN_ID, apiLoginID);
		transaction.addProperty(AUTHORIZE_NET_TRANS_KEY, transactionKey);

		//new impl
		ApiOperationBase.setEnvironment(environment);
		MerchantAuthenticationType merchantAuthenticationType = new MerchantAuthenticationType();
		merchantAuthenticationType.setName(apiLoginID);
		merchantAuthenticationType.setTransactionKey(transactionKey);
		ApiOperationBase.setMerchantAuthentication(merchantAuthenticationType);

		return merchant;
	}

	private Environment createEnvironmentFromOutlet(Outlet outlet) {
		boolean isSandboxMode = POSUtil.getBoolean(outlet.getProperty(Store.PROP_ONLINE_ORDER_AUTHNET_SANDBOX_MODE));
		Environment environment = Environment.PRODUCTION;
		if (isSandboxMode) {
			environment = Environment.SANDBOX;
		}
		return environment;
	}

	private Environment createEnvironment() {
		Environment environment = Environment.PRODUCTION;
		//		if (CardConfig.isSandboxMode()) {
		//			environment = Environment.SANDBOX;
		//		}
		return environment;
	}

	public static void main(String[] args) throws Exception {
		//		String apiLoginID = "6tuU4N3H";
		//		String transactionKey = "4k6955x3T8bCVPVm";
		//		//String MD5Value = "paltalk123";
		//
		//		Environment environment = Environment.SANDBOX;
		//
		//		Merchant merchant = Merchant.createMerchant(environment, apiLoginID, transactionKey);
		//		merchant.setDeviceType(net.authorize.DeviceType.VIRTUAL_TERMINAL);
		//		merchant.setMarketType(net.authorize.MarketType.RETAIL);
		//		//merchant.setMD5Value(MD5Value);
		//
		//		// Create credit card
		//		CreditCard creditCard = createCard("%B4111111111111111^SHAH/RIAR^1803101000000000020000831000000?;4111111111111111=1803101000020000831?", CardType.VISA.name());
		//		
		//		Transaction authTransaction = merchant.createAIMTransaction(TransactionType.AUTH_ONLY, new BigDecimal(100));
		//		authTransaction.setCreditCard(creditCard);
		//		
		//		Result<Transaction> result = (Result<Transaction>) merchant.postTransaction(authTransaction);
		//		
		//		if (result.isApproved()) {
		//			
		//			PosLog.info(getClass(),"authorization successful");
		//			
		//			Thread.sleep(1000);
		//			
		//			Transaction authCaptureTransaction = merchant.createAIMTransaction(TransactionType.PRIOR_AUTH_CAPTURE, new BigDecimal(100));
		//			authCaptureTransaction.setTransactionId(result.getTransId());
		//			authCaptureTransaction.setCreditCard(creditCard);
		//
		//			Result<Transaction> result2 = (Result<Transaction>) merchant.postTransaction(authCaptureTransaction);
		//			
		//			if (result2.isApproved()) {
		//				PosLog.info(getClass(),"capture successful");
		//				
		//				Thread.sleep(1000);
		//				
		//				Transaction voidTransaction = merchant.createAIMTransaction(TransactionType.VOID, new BigDecimal(100));
		//				voidTransaction.setTransactionId(result.getTransId());
		//				voidTransaction.setCreditCard(creditCard);
		//				
		//				Result<Transaction> result3 = (Result<Transaction>) merchant.postTransaction(authCaptureTransaction);
		//				
		//				if(result3.isApproved()) {
		//					PosLog.info(getClass(),"void successful");
		//				}
		//				else {
		//					PosLog.info(getClass(),"void declined");
		//				}
		//			}
		//		}
		//		else {
		//			PosLog.info(getClass(),result.getXmlResponseDocument());
		//		}

		final BankCardMagneticTrack track = BankCardMagneticTrack
				.from("%B4111111111111111^SHAH/RIAR^1803101000000000020000831000000?;4111111111111111=1803101000020000831?"); //$NON-NLS-1$

	}

	@Override
	public void voidTransaction(PosTransaction transaction) throws Exception {
		Merchant merchant = createMerchant(transaction);

		// Create transaction
		Transaction voidTransaction = merchant.createAIMTransaction(TransactionType.VOID, new BigDecimal(transaction.getAmount()));
		voidTransaction.setTransactionId(transaction.getCardTransactionId());

		Result<Transaction> result = (Result<Transaction>) merchant.postTransaction(voidTransaction);
		if (result.isApproved()) {
			transaction.setCardTransactionId(result.getTransId());
			transaction.setCardAuthCode(result.getAuthCode());
			transaction.setVoided(true);
			populateCardInformation(transaction, result);
		}
		else if (result.isDeclined()) {
			throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.10") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
		else {
			throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.11") + */result.getResponseReasonCodes().get(0).getReasonText()); //$NON-NLS-1$
		}
	}

	@Override
	public String getCardInformationForReceipt(PosTransaction transaction) {
		return null;
	}

	private void populateCardInformation(PosTransaction transaction, Result<Transaction> result) {
		if (result.getTarget() != null) {
			CreditCard card = result.getTarget().getCreditCard();
			if (card != null) {
				transaction.setCardType(card.getCardType().name());
				transaction.setCardNumber(card.getCreditCardNumber());

				if (StringUtils.isNotEmpty(card.getTrack1())) {
					String rawTrackData = "%" + card.getTrack1() + "?;" + card.getTrack2() + "?"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					BankCardMagneticTrack track = BankCardMagneticTrack.from(rawTrackData);
					transaction.setCardHolderName(track.getTrack1().getName().toString());
					transaction.setCardExpYear(card.getExpirationYear());
					transaction.setCardExpMonth(card.getExpirationMonth());
				}
			}
		}
	}

	@Override
	public void cancelTransaction() {
	}

	@Override
	public void refundTransaction(PosTransaction transaction, double refundAmount) throws Exception {
		createMerchant(transaction);

		PaymentType paymentType = new PaymentType();
		CreditCardType creditCard = new CreditCardType();
		creditCard.setCardNumber(transaction.getCardNumber().replaceAll("X", "")); //$NON-NLS-1$ //$NON-NLS-2$
		creditCard.setExpirationDate("XXXX"); //$NON-NLS-1$
		paymentType.setCreditCard(creditCard);

		TransactionRequestType txnRequest = new TransactionRequestType();
		txnRequest.setTransactionType(TransactionTypeEnum.REFUND_TRANSACTION.value());
		txnRequest.setRefTransId(transaction.getCardTransactionId());
		txnRequest.setAmount(new BigDecimal(String.valueOf(refundAmount)));
		txnRequest.setPayment(paymentType);

		// Make the API Request
		CreateTransactionRequest apiRequest = new CreateTransactionRequest();
		apiRequest.setTransactionRequest(txnRequest);
		CreateTransactionController controller = new CreateTransactionController(apiRequest);
		controller.execute();

		CreateTransactionResponse response = controller.getApiResponse();

		if (response != null) {
			// If API Response is ok, go ahead and check the transaction response
			String errMsg = ""; //$NON-NLS-1$
			if (response.getMessages().getResultCode() == MessageTypeEnum.OK) {
				TransactionResponse result = response.getTransactionResponse();
				if (result.getMessages() != null) {
					transaction.setRefunded(true);
					PosLog.info(getClass(), "Successfully created transaction with Transaction ID: " + result.getTransId()); //$NON-NLS-1$
					PosLog.info(getClass(), "Response Code: " + result.getResponseCode()); //$NON-NLS-1$
					PosLog.info(getClass(), "Message Code: " + result.getMessages().getMessage().get(0).getCode()); //$NON-NLS-1$
					PosLog.info(getClass(), "Description: " + result.getMessages().getMessage().get(0).getDescription()); //$NON-NLS-1$
					PosLog.info(getClass(), "Auth Code: " + result.getAuthCode()); //$NON-NLS-1$
				}
				else {
					PosLog.info(getClass(), "Failed Transaction."); //$NON-NLS-1$
					if (response.getTransactionResponse().getErrors() != null) {
						PosLog.info(getClass(), "Error Code: " + response.getTransactionResponse().getErrors().getError().get(0).getErrorCode()); //$NON-NLS-1$
						errMsg = response.getTransactionResponse().getErrors().getError().get(0).getErrorText();
						PosLog.info(getClass(), "Error message: " + errMsg); //$NON-NLS-1$
						throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.11") + */errMsg); //$NON-NLS-1$
					}
				}
			}
			else {
				PosLog.info(getClass(), "Failed Transaction."); //$NON-NLS-1$
				if (response.getTransactionResponse() != null && response.getTransactionResponse().getErrors() != null) {
					errMsg = response.getTransactionResponse().getErrors().getError().get(0).getErrorText();
					PosLog.info(getClass(), "Error Code: " + response.getTransactionResponse().getErrors().getError().get(0).getErrorCode()); //$NON-NLS-1$
					PosLog.info(getClass(), "Error message: " + errMsg); //$NON-NLS-1$
					throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.11") + */errMsg); //$NON-NLS-1$
				}
				else {
					errMsg = response.getMessages().getMessage().get(0).getText();
					PosLog.info(getClass(), "Error Code: " + response.getMessages().getMessage().get(0).getCode()); //$NON-NLS-1$
					PosLog.info(getClass(), "Error message: " + errMsg); //$NON-NLS-1$
					throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.11") + */errMsg); //$NON-NLS-1$
				}
			}
		}
		else {
			throw new Exception(/*Messages.getString("AuthorizeDotNetProcessor.11") +*/ Messages.getString("AuthorizeDotNetProcessor.38")); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}
}
