package com.floreantpos.model.dao;

import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;
import org.hibernate.type.DoubleType;
import org.hibernate.type.Type;

import com.floreantpos.PosLog;
import com.floreantpos.model.COAAccountType;
import com.floreantpos.model.COAAccountTypeGroup;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.DirectionType;
import com.floreantpos.model.LedgerEntry;
import com.floreantpos.model.LedgerEntryType;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.report.model.BalanceSheetLedgerEntryModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;

public class LedgerEntryDAO extends BaseLedgerEntryDAO {

	public List<LedgerEntry> findByChartOfAcountName(Date fromDate, Date toDate, String accountId, String orderId, String refNo) {
		Pagination<LedgerEntry> pagination = new Pagination<>(0, -1);
		findByChartOfAcountName(pagination, true, fromDate, toDate, accountId, orderId, refNo);
		return pagination.getDataList();
	}

	public void findByChartOfAcountName(PaginationSupport paginationSupport, Date fromDate, Date toDate, String accountId, String orderId, String refNo) {
		findByChartOfAcountName(paginationSupport, false, fromDate, toDate, accountId, orderId, refNo);
	}

	private void findByChartOfAcountName(PaginationSupport paginationSupport, boolean ignorePagination, Date fromDate, Date toDate, String accountId, String orderId, String refNo) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(LedgerEntry.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, orderId));
			}

			if (fromDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, fromDate));
			}
			if (toDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, toDate));
			}

			if (StringUtils.isNotBlank(accountId)) {
				criteria.add(Restrictions.eq(LedgerEntry.PROP_ACCOUNT_ID, accountId));
			}

			if (StringUtils.isNotBlank(refNo)) {
				criteria.add(Restrictions.ilike(LedgerEntry.PROP_GROUP_ID, refNo, MatchMode.ANYWHERE));
			}

			//			if (StringUtils.isNotBlank(searchString)) {
			//				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			//				detachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			//				detachedCriteria.add(Restrictions.ilike(COAAccountType.PROP_NAME, searchString, MatchMode.ANYWHERE));
			//
			//				criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(detachedCriteria)); //$NON-NLS-1$
			//			}
			criteria.setProjection(Projections.rowCount());

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.desc(LedgerEntry.PROP_CREATE_DATE));
			criteria.addOrder(Order.asc(LedgerEntry.PROP_SORT_ORDER));
			if (!ignorePagination) {
				criteria.setFirstResult(paginationSupport.getCurrentRowIndex());
				criteria.setMaxResults(paginationSupport.getPageSize());
			}
			paginationSupport.setRows(criteria.list());
		}
	}

	public void saveTransactionLedgerEntry(Ticket ticket, PosTransaction transaction) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);
			ChartOfAccounts receivableChartOfAccounts = instance.findByAcountCode("5000", session);
			Date now = StoreDAO.getServerTimestamp();
			String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("TRANS");
			Double transAmount = transaction.isVoided() ? (-1) * transaction.getAmount() : transaction.getAmount();
			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.DEBIT, transAmount,
					"TRANS/" + transaction.getId(), now);
			createDebitLedgerEntry.setTransactionId(transaction.getId());
			createDebitLedgerEntry.setGroupId(groupId);
			createDebitLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createDebitLedgerEntry.setSortOrder(1);
			LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);

			LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableChartOfAccounts, DirectionType.CREDIT, transAmount,
					"TRANS/" + transaction.getId(), now);
			createCreditLedgerEntry.setTransactionId(transaction.getId());
			createCreditLedgerEntry.setGroupId(groupId);
			createCreditLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createCreditLedgerEntry.setSortOrder(2);
			LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void savePurchaseOrderLedgerEntry(PurchaseOrder purchaseOrder) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts accountPayableCOA = instance.findByAcountCode("800", session);
			ChartOfAccounts inventoryCOA = instance.findByAcountCode("630", session);
			Double purchaseAmount = purchaseOrder.getTotalAmount();
			if (!NumberUtil.isZero(purchaseAmount)) {
				String purchaseGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("PURCHASE");
				String orderId = purchaseOrder.getOrderId();
				String entityId = purchaseOrder.getEntityId();
				Date now = StoreDAO.getServerTimestamp();

				LedgerEntry createCRPayableLedgerEntry = LedgerEntry.createLedgerEntry(null, accountPayableCOA, DirectionType.CREDIT, purchaseAmount,
						"PURCHASE/" + orderId, now);
				createCRPayableLedgerEntry.setGroupId(purchaseGroupId);
				createCRPayableLedgerEntry.setLedgerEntryType(LedgerEntryType.PURCHASE.name());
				createCRPayableLedgerEntry.setSortOrder(1);
				createCRPayableLedgerEntry.setPurchaseOrderId(entityId);
				LedgerEntryDAO.getInstance().save(createCRPayableLedgerEntry, session);

				if (!NumberUtil.isZero(purchaseAmount)) {
					LedgerEntry createInvLedgerEntry = LedgerEntry.createLedgerEntry(null, inventoryCOA, DirectionType.DEBIT, purchaseAmount,
							"PURCHASE/" + orderId, now);
					createInvLedgerEntry.setGroupId(purchaseGroupId);
					createInvLedgerEntry.setLedgerEntryType(LedgerEntryType.PURCHASE.name());
					createInvLedgerEntry.setSortOrder(2);
					createInvLedgerEntry.setPurchaseOrderId(entityId);
					LedgerEntryDAO.getInstance().save(createInvLedgerEntry, session);
				}

				if (purchaseOrder.isFullPaid()) {
					purchaseOrder.setAccountProcessed(true);
					PurchaseOrderDAO.getInstance().saveOrUpdate(purchaseOrder, session);
				}
			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void savePurchaseBillLedgerEntry(PurchaseOrder purchaseOrder, PosTransaction transaction) {

		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts accountPayableCOA = instance.findByAcountCode("800", session);
			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);

			String entityId = purchaseOrder.getEntityId();
			String orderId = purchaseOrder.getOrderId();
			String transactionId = transaction.getId();
			Date now = StoreDAO.getServerTimestamp();
			Double transAmount = transaction.getAmount();
			if (!NumberUtil.isZero(transAmount)) {
				String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("PURCHASE_BILL");
				LedgerEntry createDBTPayableLedgerEntry = LedgerEntry.createLedgerEntry(null, accountPayableCOA, DirectionType.DEBIT, transAmount,
						"PURCHASE_BILL/" + orderId, now);
				createDBTPayableLedgerEntry.setGroupId(groupId);
				createDBTPayableLedgerEntry.setLedgerEntryType(LedgerEntryType.PURCHASE.name());
				createDBTPayableLedgerEntry.setSortOrder(1);
				createDBTPayableLedgerEntry.setPurchaseOrderId(entityId);
				createDBTPayableLedgerEntry.setTransactionId(transactionId);
				LedgerEntryDAO.getInstance().save(createDBTPayableLedgerEntry, session);

				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(null, cashChartOfAccounts, DirectionType.CREDIT, transAmount,
						"PURCHASE_BILL/" + orderId, now);
				createCashLedgerEntry.setGroupId(groupId);
				createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.PURCHASE.name());
				createCashLedgerEntry.setSortOrder(2);
				createCashLedgerEntry.setPurchaseOrderId(entityId);
				createCashLedgerEntry.setTransactionId(transactionId);
				LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveOrderLedgerEntry(Ticket ticket, PosTransaction transaction) {
		double maxRFFee = ticket.getReferrerFeeOnNetSales();
		double totalCost = 0;
		List<TicketItem> ticketItems = ticket.getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				maxRFFee += ticketItem.getReferrerFeeOnReport();
				if (ticketItem.isVoided() || ticketItem.isReturned() || ticketItem.isReturnedSource()) {
					continue;
				}
				totalCost += ticketItem.getUnitCost();
			}
		}

		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts productsSalesCOA = instance.findByAcountCode("200", session);
			ChartOfAccounts taxCOA = instance.findByAcountCode("831", session);
			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("832", session);
			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("833", session);
			//			ChartOfAccounts taxCOA = instance.findByAcountCode("506", session);
			//			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("507", session);
			//			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("508", session);
			ChartOfAccounts receivableCOA = instance.findByAcountCode("5000", session);
			//ChartOfAccounts ownerADrawings = instance.findByAcountCode("880", session);
			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);

			String ticketId = ticket.getId();
			double leProductSalesAmount = getLedgerEntryAmount(productsSalesCOA.getId(), ticketId);
			double leTaxAmount = getLedgerEntryAmount(taxCOA.getId(), ticketId);
			double leRFAmount = getLedgerEntryAmount(rfPayableCOA.getId(), ticketId);
			double leLDFAmount = getLedgerEntryAmount(ldfPayableCOA.getId(), ticketId);

			Date now = StoreDAO.getServerTimestamp();
			int sortOrder = 0;
			double previousProcessAmount = leProductSalesAmount + leTaxAmount + leRFAmount + leLDFAmount;
			double remainLedgerAmount = ticket.getTotalAmount() + previousProcessAmount;
			if (!NumberUtil.isZero(remainLedgerAmount)) {
				double maxTaxAmount = ticket.getTaxAmount();
				double maxLabDoctorFee = ticket.getLabDoctorFee();
				double otherAmount = maxTaxAmount + maxRFFee + maxLabDoctorFee;
				double maxSalesAmount = ticket.getTotalAmount() - otherAmount;
				//remainLedgerAmount = remainLedgerAmount - otherAmount;
				//double maxSalesAmount = ticket.getTotalAmount();
				double remainProductSalesAmount = maxSalesAmount + leProductSalesAmount;
				double remainTaxAmount = maxTaxAmount + leTaxAmount;
				double remainRFAmount = maxRFFee + leRFAmount;
				double remainLDFAmount = maxLabDoctorFee + leLDFAmount;

				String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("ORDER");

				if (!NumberUtil.isZero(remainProductSalesAmount)) {
					LedgerEntry createPSLedgerEntry = LedgerEntry.createLedgerEntry(ticket, productsSalesCOA, DirectionType.CREDIT, remainProductSalesAmount,
							"ORDER/" + ticket.getId(), now);
					createPSLedgerEntry.setGroupId(groupId);
					createPSLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
					createPSLedgerEntry.setSortOrder(++sortOrder);
					LedgerEntryDAO.getInstance().save(createPSLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainTaxAmount)) {
					LedgerEntry createTaxLedgerEntry = LedgerEntry.createLedgerEntry(ticket, taxCOA, DirectionType.CREDIT, remainTaxAmount,
							"ORDER/" + ticket.getId(), now);
					createTaxLedgerEntry.setGroupId(groupId);
					createTaxLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
					createTaxLedgerEntry.setSortOrder(++sortOrder);
					LedgerEntryDAO.getInstance().save(createTaxLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainRFAmount)) {
					LedgerEntry createRFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, rfPayableCOA, DirectionType.CREDIT, remainRFAmount,
							"ORDER/" + ticket.getId(), now);
					createRFLedgerEntry.setGroupId(groupId);
					createRFLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
					createRFLedgerEntry.setSortOrder(++sortOrder);
					LedgerEntryDAO.getInstance().save(createRFLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainLDFAmount)) {
					LedgerEntry createLDFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ldfPayableCOA, DirectionType.CREDIT, remainLDFAmount,
							"ORDER/" + ticket.getId(), now);
					createLDFLedgerEntry.setGroupId(groupId);
					createLDFLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
					createLDFLedgerEntry.setSortOrder(++sortOrder);
					LedgerEntryDAO.getInstance().save(createLDFLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainLedgerAmount)) {
					LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableCOA, DirectionType.DEBIT, remainLedgerAmount,
							"ORDER/" + ticket.getId(), now);
					createDebitLedgerEntry.setGroupId(groupId);
					createDebitLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
					createDebitLedgerEntry.setSortOrder(++sortOrder);
					LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
				}

				saveInventoryAdjLedgerEntry(instance, session, groupId, sortOrder, now, ticket, totalCost);
				sortOrder = sortOrder + 2;
			}

			Double transAmount = transaction.getAmount();
			if (transaction instanceof RefundTransaction || transaction.isVoided()) {
				transAmount = (-1) * transAmount;
			}
			if (!NumberUtil.isZero(transAmount)) {
				String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("TRANS");

				LedgerEntry createReceivableTransLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableCOA, DirectionType.CREDIT, transAmount,
						"TRANS/" + transaction.getId(), now);
				createReceivableTransLedgerEntry.setTransactionId(transaction.getId());
				createReceivableTransLedgerEntry.setGroupId(groupId);
				createReceivableTransLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
				createReceivableTransLedgerEntry.setSortOrder(++sortOrder);
				createReceivableTransLedgerEntry.setCashDrawerId(transaction.getCashDrawerId());
				LedgerEntryDAO.getInstance().save(createReceivableTransLedgerEntry, session);

				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.DEBIT, transAmount,
						"TRANS/" + transaction.getId(), now);
				createCashLedgerEntry.setTransactionId(transaction.getId());
				createCashLedgerEntry.setGroupId(groupId);
				createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
				createCashLedgerEntry.setSortOrder(++sortOrder);
				createCashLedgerEntry.setCashDrawerId(transaction.getCashDrawerId());
				LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);
			}

			if (ticket.isPaid()) {
				ticket.setAccountProcessed(true);
			}
			Set<PosTransaction> transactions = ticket.getTransactions();
			for (PosTransaction posTransaction : transactions) {
				if (posTransaction.getId().equals(transaction.getId())) {
					posTransaction.setAccountProcessed(true);
				}
			}
			TicketDAO.getInstance().saveOrUpdate(ticket, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveInventoryAdjLedgerEntry(ChartOfAccountsDAO instance, Session session, String groupId, int sortOrder, Date now, Ticket ticket,
			double totalCost) {
		ChartOfAccounts costOfGoodsSoldCOA = instance.findByAcountCode("310", session);
		ChartOfAccounts inventoryCOA = instance.findByAcountCode("630", session);

		double leCostOfGoodsSoldAmount = getLedgerEntryAmount(costOfGoodsSoldCOA.getId(), ticket.getId());
		double remainCostOfGoodsSoldAmount = totalCost - leCostOfGoodsSoldAmount;
		if (!NumberUtil.isZero(remainCostOfGoodsSoldAmount)) {
			LedgerEntry createInventoryLedgerEntry = LedgerEntry.createLedgerEntry(ticket, inventoryCOA, DirectionType.CREDIT, remainCostOfGoodsSoldAmount,
					"ORDER/" + ticket.getId(), now);
			createInventoryLedgerEntry.setGroupId(groupId);
			createInventoryLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createInventoryLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createInventoryLedgerEntry, session);

			LedgerEntry createCostOfGoodsSoldLedgerEntry = LedgerEntry.createLedgerEntry(ticket, costOfGoodsSoldCOA, DirectionType.DEBIT,
					remainCostOfGoodsSoldAmount, "ORDER/" + ticket.getId(), now);
			createCostOfGoodsSoldLedgerEntry.setGroupId(groupId);
			createCostOfGoodsSoldLedgerEntry.setLedgerEntryType(LedgerEntryType.ORDER_ENTRY.name());
			createCostOfGoodsSoldLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createCostOfGoodsSoldLedgerEntry, session);
		}
	}

	public double getLedgerEntryAmount(String accountId, String ticketId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			criteria.add(Restrictions.eq(LedgerEntry.PROP_ACCOUNT_ID, accountId));
			criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, ticketId));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + LedgerEntry.PROP_AMOUNT, new String[] { LedgerEntry.PROP_AMOUNT },
					new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);
			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				return number.doubleValue();
			}
		}
		return 0;
	}

	public void saveRFPayLedgerEntry(Map<Ticket, Double> ticketsMap, String transactionId, Session session) {
		Set<Ticket> keySet = ticketsMap.keySet();
		Date now = StoreDAO.getServerTimestamp();
		int sortOrder = 0;
		for (Ticket ticket : keySet) {
			ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();

			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);
			//ChartOfAccounts rfPayableCOA = instance.findByAcountCode("507", session);
			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("832", session);

			double transAmount = ticketsMap.get(ticket);
			String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("RF");

			LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.CREDIT, transAmount,
					"TRANS/" + transactionId, now);
			createCashLedgerEntry.setTransactionId(transactionId);
			createCashLedgerEntry.setGroupId(groupId);
			createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.RF_PAY.name());
			createCashLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

			LedgerEntry createRFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, rfPayableCOA, DirectionType.DEBIT, transAmount, "TRANS/" + transactionId,
					now);
			createRFLedgerEntry.setTransactionId(transactionId);
			createRFLedgerEntry.setGroupId(groupId);
			createRFLedgerEntry.setLedgerEntryType(LedgerEntryType.RF_PAY.name());
			createRFLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createRFLedgerEntry, session);
		}

	}

	public void saveLDFPayLedgerEntry(Map<Ticket, Double> ticketsMap, String transactionId, Session session) {
		Set<Ticket> keySet = ticketsMap.keySet();
		Date now = StoreDAO.getServerTimestamp();
		int sortOrder = 0;
		for (Ticket ticket : keySet) {
			ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();

			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);
			//ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("508", session);
			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("833", session);

			double transAmount = ticketsMap.get(ticket);
			String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("LDF");

			LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.CREDIT, transAmount,
					"TRANS/" + transactionId, now);
			createCashLedgerEntry.setTransactionId(transactionId);
			createCashLedgerEntry.setGroupId(groupId);
			createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.LDF_PAY.name());
			createCashLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

			LedgerEntry createLDFPayLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ldfPayableCOA, DirectionType.DEBIT, transAmount,
					"TRANS/" + transactionId, now);
			createLDFPayLedgerEntry.setTransactionId(transactionId);
			createLDFPayLedgerEntry.setGroupId(groupId);
			createLDFPayLedgerEntry.setLedgerEntryType(LedgerEntryType.LDF_PAY.name());
			createLDFPayLedgerEntry.setSortOrder(++sortOrder);
			LedgerEntryDAO.getInstance().save(createLDFPayLedgerEntry, session);
		}
	}

	public List getLedgerEntries(List<String> coaAccountTypeGroupIds) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			if (!coaAccountTypeGroupIds.isEmpty()) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, coaAccountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(LedgerEntry.class);

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			return criteria.list();

		}
	}

	public BalanceSheetLedgerEntryModel getRetainedEarning(Date startDate, Date endDate) {
		try (Session session = createNewSession()) {
			List<String> coaAccountTypeGroupIds = Arrays.asList("revenue", "expense");

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			if (!coaAccountTypeGroupIds.isEmpty()) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, coaAccountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(LedgerEntry.class);
			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$
			BalanceSheetLedgerEntryModel retainedEarning = new BalanceSheetLedgerEntryModel();
			retainedEarning.setName("Retained earning");
			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				retainedEarning.setAmount(number.doubleValue());
			}
			return retainedEarning;

		}
	}

	public List getSumOfLedgerEntries(Date startDate, Date endDate) {
		try (Session session = createNewSession()) {
			Map<String, BalanceSheetLedgerEntryModel> map = new HashMap<String, BalanceSheetLedgerEntryModel>();
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.property(LedgerEntry.PROP_DIRECTION), LedgerEntry.PROP_DIRECTION);
			projectionList.add(Projections.sqlProjection("sum(amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_DIRECTION));
			criteria.setProjection(projectionList);

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));

			List<BalanceSheetLedgerEntryModel> list = criteria.list();
			if (list != null) {

				for (BalanceSheetLedgerEntryModel entryModel : list) {
					String coaId = entryModel.getCoaId();

					BalanceSheetLedgerEntryModel model = map.get(coaId);
					if (model == null) {
						model = new BalanceSheetLedgerEntryModel();
						model.setCoaId(coaId);
						model.setDirection(null);
						map.put(coaId, model);
					}

					double amount = model.getAmount();
					amount += entryModel.getAmount() * entryModel.getDirection();

					ChartOfAccounts chartOfAccounts = ChartOfAccountsDAO.getInstance().get(coaId);
					COAAccountType coaAccountType = COAAccountTypeDAO.getInstance().get(chartOfAccounts.getCoaAccountTypeId());
					COAAccountTypeGroup group = COAAccountTypeGroupDAO.getInstance().get(coaAccountType.getCoaAccountTypeGroupId());
					DirectionType type = DirectionType.getByTypeNo(group.getAccountDirection());
					if (DirectionType.DEBIT == type) {
						model.setDebitAmount(amount);
					}
					else {
						model.setCreditAmount(amount * (-1));
					}
					model.setAmount(amount);
				}
			}
			return map.values().stream().filter(t -> !NumberUtil.isZero(t.getAmount())).collect(Collectors.toList());
		}
	}

	public double getTaxLedgerEntryAmount(Date startDate, Date endDate) {
		return getLedgerEntriesAmountByTypeId("expense", "506", startDate, endDate);
	}

	public double getLedgerEntriesAmountByTypeId(String coaAccountTypeId, Date startDate, Date endDate) {
		return getLedgerEntriesAmountByTypeId(coaAccountTypeId, "", startDate, endDate);
	}

	private double getLedgerEntriesAmountByTypeId(String coaAccountTypeId, String coaCode, Date startDate, Date endDate) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			if (StringUtils.isNotBlank(coaAccountTypeId)) {
				coaDetachedCriteria.add(Restrictions.eq(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID, coaAccountTypeId));
			}
			if (StringUtils.isNotBlank(coaCode)) {
				coaDetachedCriteria.add(Restrictions.eq(ChartOfAccounts.PROP_ACCOUNT_CODE, coaCode));
			}
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				return number.doubleValue();
			}

		}
		return 0;
	}

	/**
	 * @param startDate
	 * @param endDate
	 *  find all account type IDs with group id "expense"
	 *  find all ChartOfAccounts IDs with COAAccountType IDs without "direct_costs" account type
	 * 
	 * @return Expense amount by account type without cost
	 */
	public List getExpenseLedgerEntries(Date startDate, Date endDate) {
		return getCustomLedgerEntries(startDate, endDate, Arrays.asList("expense"), "direct_costs", "506");
	}

	public List getCustomLedgerEntries(Date startDate, Date endDate, List<String> accountTypeGroupIds, String withoutAccountTypeId, String withoutAccountCode) {
		return getCustomLedgerEntries(startDate, endDate, accountTypeGroupIds, Arrays.asList(withoutAccountTypeId), withoutAccountCode);
	}

	public List getCustomLedgerEntries(Date startDate, Date endDate, List<String> accountTypeGroupIds, List<String> withoutAccountTypeIds,
			String withoutAccountCode) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));

			if (accountTypeGroupIds != null) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, accountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$
			if (!withoutAccountTypeIds.isEmpty()) {
				coaDetachedCriteria.add(Restrictions.not(Restrictions.in(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID, withoutAccountTypeIds)));
			}
			if (StringUtils.isNotBlank(withoutAccountCode)) {
				coaDetachedCriteria.add(Restrictions.ne(ChartOfAccounts.PROP_ACCOUNT_CODE, withoutAccountCode));
			}
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			return criteria.list();

		}
	}

	public List getLedgerEntriesByTypeId(String coaAccountTypeId, Date startDate, Date endDate) {
		return getLedgerEntriesByTypeId(Arrays.asList(coaAccountTypeId), startDate, endDate);
	}

	public List getLedgerEntriesByTypeId(List<String> coaAccountTypeId, Date startDate, Date endDate) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaAccountTypeId)); //$NON-NLS-1$
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			criteria.addOrder(Order.desc("coaId"));
			return criteria.list();

		}
	}

	public void savePayoutLedgerEntry(PayOutTransaction payOutTransaction) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts generalExpensesCOA = instance.findByAcountCode("429", session);
			ChartOfAccounts cashCOA = instance.findByAcountCode("10100", session);
			Date now = StoreDAO.getServerTimestamp();
			Double payoutAmount = payOutTransaction.getAmount();
			if (!NumberUtil.isZero(payoutAmount)) {
				String payoutGroupId = SequenceNumberDAO.getInstance().getNextLedgerRef("PAYOUT");
				String transId = payOutTransaction.getId();

				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(null, cashCOA, DirectionType.CREDIT, payoutAmount, "PAYOUT/" + transId, now);
				createCashLedgerEntry.setGroupId(payoutGroupId);
				createCashLedgerEntry.setLedgerEntryType(LedgerEntryType.SALARY.name());
				createCashLedgerEntry.setTransactionId(transId);
				createCashLedgerEntry.setCashDrawerId(payOutTransaction.getCashDrawerId());
				createCashLedgerEntry.setPayoutRecepientId(payOutTransaction.getRecepientId());
				createCashLedgerEntry.setSortOrder(2);
				LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

				LedgerEntry createCRPayableLedgerEntry = LedgerEntry.createLedgerEntry(null, generalExpensesCOA, DirectionType.DEBIT, payoutAmount,
						"PAYOUT/" + transId, now);
				createCRPayableLedgerEntry.setGroupId(payoutGroupId);
				createCRPayableLedgerEntry.setLedgerEntryType(LedgerEntryType.SALARY.name());
				createCRPayableLedgerEntry.setTransactionId(transId);
				createCRPayableLedgerEntry.setCashDrawerId(payOutTransaction.getCashDrawerId());
				createCRPayableLedgerEntry.setPayoutRecepientId(payOutTransaction.getRecepientId());
				createCRPayableLedgerEntry.setSortOrder(1);
				LedgerEntryDAO.getInstance().save(createCRPayableLedgerEntry, session);

				payOutTransaction.setAccountProcessed(true);
				PayOutTransactionDAO.getInstance().saveOrUpdate(payOutTransaction, session);

			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

}