package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.PriceRule;
import com.floreantpos.model.PriceTable;
import com.floreantpos.model.PriceTableItem;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PriceTableDAO extends BasePriceTableDAO {

	private static final String EMPTY_STRING = ""; //$NON-NLS-1$
	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$
	private static final String EMPTY_DOT_STRING = "."; //$NON-NLS-1$
	private static final String EMPTY_SPACE_STRING = " "; //$NON-NLS-1$

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PriceTableDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PriceTable) {
			PriceTable priceTable = (PriceTable) obj;
			this.checkForeignRelation(priceTable, s);
			priceTable.setDeleted(Boolean.TRUE);
			super.update(priceTable, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	private void checkForeignRelation(PriceTable priceTable, Session s) {
		List<PriceRule> priceRules = PriceRuleDAO.getInstance().getPriceRulesByPriceTable(priceTable, s);
		if (priceRules != null && !priceRules.isEmpty()) {
			String message = priceTable.getName() + EMPTY_SPACE_STRING + Messages.getString("PriceTableDAO.4"); //$NON-NLS-1$
			String details = this.constructExceptionDetailsByMenuItems(priceTable, priceRules);
			throw new PosException(message, details);
		}
	}

	private String constructExceptionDetailsByMenuItems(PriceTable priceTable, List<PriceRule> priceRules) {
		if (priceRules != null && !priceRules.isEmpty()) {
			StringBuilder builder = new StringBuilder(priceTable.getName() + EMPTY_SPACE_STRING + Messages.getString("PriceTableDAO.5")); //$NON-NLS-1$
			for (int i = 0; i < priceRules.size(); i++) {
				String message = (i + 1) + EMPTY_DOT_STRING + EMPTY_SPACE_STRING + priceRules.get(i).getName();
				builder.append(EMPTY_NEWLINE_STRING).append(message);
			}
			return builder.toString();
		}
		return EMPTY_STRING;
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<PriceTable> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Deprecated
	public void releaseParentAndDelete(PriceTable priceTable) {
		this.delete(priceTable);
	}

	public void saveOrUpdatePriceTableList(List<PriceTable> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<PriceTable> iterator = dataList.iterator(); iterator.hasNext();) {
				PriceTable item = (PriceTable) iterator.next();
				PriceTable existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					updatePriceTableItems(existingItem, session);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					updatePriceTableItems(item, session);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	private void updatePriceTableItems(PriceTable existingItem, Session session) throws Exception {
		if (existingItem.getPriceTableItems() == null || existingItem.getPriceTableItems().isEmpty()) {
			return;
		}
		for (PriceTableItem priceTableItem : existingItem.getPriceTableItems()) {
			PriceTableItem existingPriceTableItem = PriceTableItemDAO.getInstance().get(priceTableItem.getId());
			if (existingPriceTableItem != null) {
				if (!BaseDataServiceDao.get().shouldSave(priceTableItem.getLastUpdateTime(), existingPriceTableItem.getLastUpdateTime())) {
					PosLog.info(getClass(), priceTableItem.getId() + " already updated"); //$NON-NLS-1$
					continue;
				}
				long version = existingPriceTableItem.getVersion();
				PropertyUtils.copyProperties(existingPriceTableItem, priceTableItem);
				existingPriceTableItem.setVersion(version);
				update(existingPriceTableItem, session);
			}
			else {
				save(priceTableItem, session);
			}
		}
	}

	public PriceTable findByOutletId(String id, String outletId) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			criteria.add(Restrictions.eq(PriceTable.PROP_ID, id));
			criteria.add(Restrictions.eq(PriceTable.PROP_OUTLET_ID, outletId));
			criteria.setMaxResults(1);
			this.addDeletedFilter(criteria);
			return (PriceTable) criteria.uniqueResult();
		}
	}
}