package com.floreantpos.report;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import com.floreantpos.model.BloodGroupType;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Department;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.GlobalConfig;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.ReportType;
import com.floreantpos.model.Specimen;
import com.floreantpos.model.Store;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.GlobalConfigDAO;
import com.floreantpos.model.dao.SpecimenDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.POSUtil;

import net.sf.jasperreports.engine.JRBand;
import net.sf.jasperreports.engine.JRElement;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRGroup;
import net.sf.jasperreports.engine.JRSection;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;
import net.sf.jasperreports.engine.type.FooterPositionEnum;

public class LabTestReportUtil {
	public static JasperPrint createJasperPrint(TicketItem ticketItem) throws JRException {
		return createJasperPrint(ticketItem, null);
	}

	public static JasperPrint createJasperPrint(TicketItem ticketItem, Map<String, Object> mapForPreview) throws JRException {
		LabTestResultReportModel testResultReportModel = createModels(ticketItem);

		HashMap<String, Object> map = new HashMap<String, Object>();
		Outlet outlet = DataProvider.get().getOutlet();
		ReportUtil.populateMedlogicsProperties(map, outlet);
		map.put("reportTitle", "Lab test result report"); //$NON-NLS-1$ 
		JasperReport header = ReportUtil.getReport("medlogics_report_customer_header"); //$NON-NLS-1$
		map.put("head", header); //$NON-NLS-1$
		populateReportParameter(map, ticketItem);

		Store store = DataProvider.get().getStore();
		int reportHeaderMargin = store.getReportHeaderMargin();
		int reportFooterMargin = store.getReportFooterMargin();
		if (mapForPreview != null) {
			map.putAll(mapForPreview);
			reportHeaderMargin = POSUtil.parseInteger((String) mapForPreview.get("headerMarginValue"));
			reportFooterMargin = POSUtil.parseInteger((String) mapForPreview.get("footerMarginValue"));
		}
		JasperReport masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$

		String reportType = ticketItem.getReportType();
		MenuItem menuItem = ticketItem.getMenuItem();
		if (menuItem != null) {
			reportType = menuItem.getReportType();
		}

		if (!reportType.equals(ReportType.TABULAR_REPORT.name())) {
			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-plain-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$
		}

		//		if (reportType.equals(ReportType.PLAIN_TABULAR_REPORT.name())) {
		//			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-plain-tabular-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$
		//		}

		if (reportType.equals(ReportType.POSITIVE_NEGATIVE.name())) {
			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-positive-negative-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$
		}

		if (reportType.equals(ReportType.PLAIN_REPORT_SIDE_BY_SIDE.name())) {
			masterReport = ReportUtil.adjustReportHeaderAndFooter("lab-test-result-plain-side-by-side-report", reportFooterMargin, reportHeaderMargin); //$NON-NLS-1$
		}

		JRGroup[] groups = masterReport.getGroups();
		for (JRGroup group : groups) {
			if (group.getFooterPositionValue() == FooterPositionEnum.STACK_AT_BOTTOM) {
				JRSection section = group.getGroupFooterSection();
				JRBand[] bands = section.getBands();
				for (JRBand band : bands) {
					boolean visibleLabTechnician = POSUtil
							.getBoolean(GlobalConfigDAO.getInstance().getProperty(GlobalConfig.PRINT_LAB_TEST_TECHNICIAN_IN_REPORT), true); //$NON-NLS-1$
					if (!visibleLabTechnician) {
						if (band != null) {
							JRElement inchargeFrame = band.getElementByKey("lab_incharge");
							if (inchargeFrame != null) {
								inchargeFrame.setX(0);
							}
						}
					}
				}
			}
		}

		JasperPrint jasperPrint = JasperFillManager.fillReport(masterReport, map, new JRTableModelDataSource(testResultReportModel));
		return jasperPrint;
	}

	public static void populateTestItemParameter(Map<String, Object> contextMap, TicketItem ticketItem) {
		List<TestItem> testItems = ticketItem.getTestItems();
		final StringBuilder items = new StringBuilder(""); //$NON-NLS-1$
		MenuItem menuItem = ticketItem.getMenuItem();
		if (testItems != null && testItems.size() > 0) {
			testItems.forEach(item -> {
				StringBuilder itemRow = new StringBuilder("<tr>"); //$NON-NLS-1$

				if (!ReportType.PLAIN_REPORT.name().equals(menuItem.getReportType())) {
					itemRow.append("<td style = \"padding-bottom: 5px\" ><b>"); //$NON-NLS-1$
					itemRow.append(item.getName());
					itemRow.append("</b></td>"); //$NON-NLS-1$
					itemRow.append("</tr>"); //$NON-NLS-1$
					itemRow.append("<tr>"); //$NON-NLS-1$
				}
				itemRow.append("<td>"); //$NON-NLS-1$
				itemRow.append(item.getResult());
				itemRow.append("</td>"); //$NON-NLS-1$
				itemRow.append("</tr>"); //$NON-NLS-1$

				items.append(itemRow.toString());
			});
		}
		contextMap.put("testItemsValue", "<table style = 'width: 100%'>" + items.toString() + "</table>");

	}

	public static void populateReportParameter(Map<String, Object> map, TicketItem ticketItem) {
		Ticket ticket = ticketItem.getTicket();
		Customer patient = ticket.getCustomer();
		Store store = DataProvider.get().getStore();

		String method = ticketItem.getProperty("method"); //$NON-NLS-1$
		String billName = StringUtils.isBlank(ticketItem.getGroupName()) ? "Laboratory Report" : ticketItem.getGroupName() + " report";

		Boolean visibleReportHeader = POSUtil.getBoolean(GlobalConfigDAO.getInstance().getProperty(GlobalConfig.REPORT_PRINT_HEADER), true); //$NON-NLS-1$
		map.put("visibleReportHeader", visibleReportHeader); //$NON-NLS-1$

		Boolean visibleGroupInReport = POSUtil.getBoolean(GlobalConfigDAO.getInstance().getProperty(GlobalConfig.PRINT_GROUP_IN_REPORT), true); //$NON-NLS-1$
		map.put("visibleGroupInReport", visibleGroupInReport); //$NON-NLS-1$

		Boolean visibleLabTestFooterInReport = POSUtil.getBoolean(GlobalConfigDAO.getInstance().getProperty(GlobalConfig.PRINT_LAB_TEST_FOOTET_IN_REPORT),
				true); //$NON-NLS-1$
		map.put("visibleLabTestFooterInReport", visibleLabTestFooterInReport); //$NON-NLS-1$
		Boolean visibleLabTechnician = POSUtil.getBoolean(GlobalConfigDAO.getInstance().getProperty(GlobalConfig.PRINT_LAB_TEST_TECHNICIAN_IN_REPORT), true); //$NON-NLS-1$
		map.put("visibleLabTechnician", visibleLabTechnician); //$NON-NLS-1$

		Boolean visibleLabIncharge = POSUtil.getBoolean(GlobalConfigDAO.getInstance().getProperty(GlobalConfig.PRINT_LAB_TEST_INCHARGE_IN_REPORT), true); //$NON-NLS-1$
		map.put("visibleLabIncharge", visibleLabIncharge); //$NON-NLS-1$

		Boolean visibleLabDoctor = POSUtil.getBoolean(GlobalConfigDAO.getInstance().getProperty(GlobalConfig.PRINT_LAB_DOCTOR_IN_REPORT), true); //$NON-NLS-1$
		map.put("visibleLabDoctor", visibleLabDoctor); //$NON-NLS-1$

		map.put("barcode", ticket.getId()); //$NON-NLS-1$
		map.put("billNo", ticket.getId()); //$NON-NLS-1$

		map.put("instrumentUsed", "Instrument Used: " + method); //$NON-NLS-1$
		map.put("bill", billName); //$NON-NLS-1$
		map.put("methodName", method == null ? "" : method); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("date", DateUtil.formatDateWithTime(DateUtil.convertServerTimeToBrowserTime(ticketItem.getCreateDate()))); //$NON-NLS-1$
		Date deliveryDate = ticketItem.getDeliveryDate();
		if (deliveryDate != null) {
			map.put("dateReceived", DateUtil.formatDateWithTime(DateUtil.convertServerTimeToBrowserTime(deliveryDate))); //$NON-NLS-1$
		}
		map.put("labId", store.getUuid()); //$NON-NLS-1$

		map.put("patientId", patient.getId()); //$NON-NLS-1$
		String patientName = patient.getName();
		if (patientName != null) {
			patientName = patientName.trim();
		}
		map.put("ptName", patientName); //$NON-NLS-1$
		map.put("phone", patient.getMobileNo()); //$NON-NLS-1$
		map.put("sex", patient.getPatientGender()); //$NON-NLS-1$
		Date dateOfBirth = patient.getDateOfBirth();
		if (dateOfBirth != null) {
			map.put("age", String.valueOf(DateUtil.calculatePatientAge(DateUtil.convertDateToLocalDate(dateOfBirth)))); //$NON-NLS-1$
		}
		String doctorName = ticket.getDoctorName();
		if (doctorName != null) {
			doctorName = doctorName.trim();
		}
		map.put("consultant", doctorName); //$NON-NLS-1$
		map.put("testName", ticketItem.getName()); //$NON-NLS-1$
		map.put("sampleId", ticketItem.getLabTestId()); //$NON-NLS-1$

		String bloodGroup = patient.getBloodGroup();
		if (StringUtils.isNotBlank(bloodGroup)) {
			BloodGroupType bloodGroupType = BloodGroupType.fromNameString(bloodGroup);
			if (bloodGroupType != null) {
				bloodGroup = bloodGroupType.getDisplayString();
			}
			else {
				bloodGroup = ""; //$NON-NLS-1$
			}
		}
		map.put("bloodGroup", bloodGroup); //$NON-NLS-1$
		map.put("colCategoryName", ticketItem.getCategoryName()); //$NON-NLS-1$
		map.put("colNameOfTest", "<b>Name of test</b>"); //$NON-NLS-1$ 
		map.put("colResult", "<b>Result</b>"); //$NON-NLS-1$ 
		map.put("colReferenceValue", "<b>Reference value</b>"); //$NON-NLS-1$ 
		map.put("colApproveResultNote", "<b>Comment</b>"); //$NON-NLS-1$ 
		map.put("colUnit", "Unit"); //$NON-NLS-1$ 
		User owner = ticket.getOwner();
		String serverName = "";
		if (owner != null) {
			serverName = owner.getFullName();
		}

		map.put("colPrintedBy", "Printed: " + DateUtil.formatDateWithTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp())) + " " + "by" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				+ " " + serverName); //$NON-NLS-1$

		String note = ticketItem.getReportDescription();
		map.put("colNote", "Comment:"); //$NON-NLS-1$ 
		if (StringUtils.isNotBlank(note)) {
			map.put("note", note); //$NON-NLS-1$
		}

		String departmentName = ""; //$NON-NLS-1$
		String doctorId = ticket.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			String departmentId = DoctorDAO.getInstance().get(doctorId).getDoctorDepartmentId();
			Department department = DataProvider.get().getDepartmentById(departmentId);
			if (department != null) {
				departmentName = department.getName() != null ? department.getName().trim() : department.getName();
			}
		}
		map.put("department", departmentName); //$NON-NLS-1$

		String specimenName = ""; //$NON-NLS-1$
		String specimenId = ticketItem.getSpecimenId();
		if (StringUtils.isNotBlank(specimenId)) {
			Specimen specimen = SpecimenDAO.getInstance().get(specimenId);
			if (specimen != null) {
				specimenName = specimen.getName() != null ? specimen.getName().trim() : specimen.getName();
			}
		}
		map.put("sampleType", specimenName); //$NON-NLS-1$

		String labDoctorId = ticketItem.getLabDoctorId();
		String labDoctorName = "";
		String doctorDesignation = "";
		if (StringUtils.isNotBlank(labDoctorId)) {
			Doctor labDoctor = DoctorDAO.getInstance().get(labDoctorId);
			if (labDoctor != null) {
				labDoctorName = labDoctor.getName(); //$NON-NLS-1$
				doctorDesignation = labDoctor.getDoctorDesignation();
				if (doctorDesignation != null) {
					doctorDesignation = doctorDesignation.trim();
				}
				else {
					labDoctorName = "";
					doctorDesignation = "";
				}
			}
		}
		map.put("colLabDoctorName", labDoctorName);
		map.put("colLabDoctorDesignation", POSUtil.simplifySentence(doctorDesignation, 1)); //$NON-NLS-1$

		String labInchargeId = ticketItem.getLabInchargeId();
		String labInchargeName = "";
		String inchargeDesignation = "";
		if (StringUtils.isNotBlank(labInchargeId)) {
			User labInchargeUser = UserDAO.getInstance().get(labInchargeId, DataProvider.get().getCurrentOutletId());
			if (labInchargeUser != null) {
				labInchargeName = labInchargeUser.getFullName(); //$NON-NLS-1$
				inchargeDesignation = labInchargeUser.getDesignation();
				if (inchargeDesignation != null) {
					inchargeDesignation = inchargeDesignation.trim();
				}
				else {
					labInchargeName = "";
					inchargeDesignation = "";
				}
			}
		}
		map.put("colLabInchargeName", labInchargeName);
		map.put("colLabInchargeDesignation", POSUtil.simplifySentence(inchargeDesignation, 1)); //$NON-NLS-1$

		String labTechnicianId = ticketItem.getLabTechnicianId();
		String technicianName = "";
		String technicianDesignation = "";
		if (StringUtils.isNotBlank(labTechnicianId)) {
			User labTechnicianUser = UserDAO.getInstance().get(labTechnicianId, DataProvider.get().getCurrentOutletId());
			if (labTechnicianUser != null) {
				technicianName = labTechnicianUser.getFullName(); //$NON-NLS-1$
				technicianDesignation = labTechnicianUser.getDesignation();
				if (technicianDesignation != null) {
					technicianDesignation = technicianDesignation.trim();
				}
				else {
					technicianName = "";
					technicianDesignation = "";
				}
			}
		}
		map.put("colLabTechnician", technicianName);
		map.put("colLabTechnicianDesignation", POSUtil.simplifySentence(technicianDesignation, 1)); //$NON-NLS-1$

		map.put("reportDescription", ticketItem.getReportDescription()); //$NON-NLS-1$
	}

	public static LabTestResultReportModel createModels(TicketItem ticketItem) {
		List<TestItem> testItems = new ArrayList();
		LabTestResultReportModel testResultReportModel = new LabTestResultReportModel(ticketItem);

		List<TestItem> tItems = ticketItem.getTestItems();
		if (tItems != null) {
			for (TestItem item : tItems) {
				testItems.add(item);
			}
		}
		testResultReportModel.setRows(testItems);
		return testResultReportModel;
	}
}
