/**
 * ************************************************************************
d * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.ShopFloor;
import com.floreantpos.model.ShopSeat;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.ShopTableStatus;
import com.floreantpos.model.ShopTableTicket;
import com.floreantpos.model.ShopTableType;
import com.floreantpos.model.TableStatus;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.User;
import com.floreantpos.util.POSUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ShopTableDAO extends BaseShopTableDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ShopTableDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	//	@Override
	//	protected void updateTime(Object model) {
	//		ShopTable shopTable = (ShopTable) model;
	//		if (shopTable.getSeats() != null) {
	//			for (ShopSeat shopSeat : shopTable.getSeats()) {
	//				super.updateTime(shopSeat);
	//			}
	//		}
	//		super.updateTime(model);
	//	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public Order getDefaultOrder() {
		return Order.asc(ShopTable.PROP_ID);
	}

	@Override
	public List<ShopTable> findAll() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(ShopTable.PROP_ID));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public int getNextTableNumber() {
		try (Session session = this.createNewSession()) {
			/*if (considerDeletedTables) {
				Criteria criteria = session.createCriteria(this.getReferenceClass());
				criteria.add(Restrictions.eq(ShopTable.PROP_DELETED, Boolean.TRUE));
				criteria.setProjection(Projections.min(ShopTable.PROP_ID));
				Integer smallestAvailableId = POSUtil.getInteger(criteria.uniqueResult()) - 1;
				if (smallestAvailableId >= 0) {
					return smallestAvailableId;
				}
			}*/
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.setProjection(Projections.max(ShopTable.PROP_ID));
			return POSUtil.getInteger(criteria.uniqueResult());
		}
	}

	public ShopTable getByNumber(int tableNumber) {
		Session session = null;
		try {
			session = createNewSession();

			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(ShopTable.PROP_ID, tableNumber));

			return (ShopTable) criteria.uniqueResult();
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTable> getByFloor(ShopFloor shopFloor) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(ShopTable.PROP_FLOOR_ID, shopFloor.getId()));
			criteria.addOrder(Order.asc(ShopTable.PROP_ID));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTable> getAllUnassigned() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.isNull(ShopTable.PROP_FLOOR_ID));
			this.addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(ShopTable.PROP_ID));
			return criteria.list();
		}
	}

	public List<ShopTable> getByNumbers(Collection<Integer> tableNumbers) {
		if (tableNumbers == null || tableNumbers.size() == 0) {
			return null;
		}

		Session session = null;

		try {
			session = createNewSession();

			Criteria criteria = session.createCriteria(getReferenceClass());
			Disjunction disjunction = Restrictions.disjunction();

			for (Integer tableNumber : tableNumbers) {
				disjunction.add(Restrictions.eq(ShopTable.PROP_ID, tableNumber));
			}
			criteria.add(disjunction);

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTable> getTables(Ticket ticket) {
		return getByNumbers(ticket.getTableNumbers());
	}

	public void bookedTables(List<ShopTableStatus> tables) {
		if (tables == null) {
			return;
		}

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			bookTables(tables, session);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void bookTables(List<ShopTableStatus> tables, Session session) {
		for (ShopTableStatus shopTableStatus : tables) {
			shopTableStatus.setTableStatus(TableStatus.Seat);
			ShopTableStatusDAO.getInstance().saveOrUpdate(shopTableStatus, session);
		}
	}

	public void occupyTables(Ticket ticket) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			occupyTables(ticket, session);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void occupyTables(Ticket ticket, Session session) {
		List<Integer> tableNumbers = ticket.getTableNumbers();
		List<String> ticketNumbers = new ArrayList<>();
		ticketNumbers.add(ticket.getId());
		User owner = ticket.getOwner();

		for (Integer tableId : tableNumbers) {
			ShopTableStatus shopTableStatus = (ShopTableStatus) ShopTableStatusDAO.getInstance().get(tableId, ticket.getOutletId(), session);
			if (shopTableStatus == null) {
				shopTableStatus = new ShopTableStatus();
				shopTableStatus.setId(tableId);
			}

			String ownerId = null;
			String firstName = null;
			if (owner != null) {
				ownerId = owner.getId();
				firstName = owner.getFirstName();
			}

			shopTableStatus.setTableTicket(ticket.getId(), ticket.getTokenNo(), ownerId, firstName, ticket.getCreateDate());
			shopTableStatus.setTableStatus(TableStatus.Serving);
			shopTableStatus.setShouldPublishMqtt(ticket.isShouldPublishMqtt());
			ShopTableStatusDAO.getInstance().saveOrUpdate(shopTableStatus, session);
		}
	}

	public void freeBookedTables(List<ShopTableStatus> tableStatusList) {
		if (tableStatusList == null) {
			return;
		}
		Transaction transaction = null;
		try (Session session = this.createNewSession()) {
			transaction = session.beginTransaction();

			freeBookedTables(tableStatusList, session);

			transaction.commit();
		} catch (Exception e) {
			transaction.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		}
	}

	public void freeBookedTables(List<ShopTableStatus> tableStatusList, Session session) {
		for (ShopTableStatus shopTableStatus : tableStatusList) {
			List<ShopTableTicket> ticketNumbers = shopTableStatus.getTicketNumbers();
			if (ticketNumbers == null || ticketNumbers.isEmpty()) {
				shopTableStatus.setTableStatus(TableStatus.Available);
				shopTableStatus.removeProperty(ShopTableStatus.SEAT_TIME);
				ShopTableStatusDAO.getInstance().saveOrUpdate(shopTableStatus, session);
			}
		}
	}

	public void freeBookedShopTables(List<ShopTable> bookedTables, Session session) {
		for (ShopTable shopTable : bookedTables) {
			ShopTableDAO.getInstance().refresh(shopTable);
			shopTable.setCurrentBookingId(null);
			shopTable.removeProperty(ShopTable.RESERVATION_NUMBER);
			shopTable.removeProperty(AppConstants.SHOP_TABLE_CUSTOMER_ID);
			shopTable.setTableStatus(TableStatus.Available);
			shopTable.setCustomerName(null);
			ShopTableDAO.getInstance().update(shopTable, session);

			ShopTableStatus shopTableStatus = shopTable.getShopTableStatus();
			ShopTableStatusDAO.getInstance().refresh(shopTableStatus, session);
			if (shopTableStatus == null) {
				continue;
			}
			shopTableStatus.removeProperty(ShopTableStatus.SEAT_TIME);
			shopTableStatus.setTableStatus(TableStatus.Available);
			ShopTableStatusDAO.getInstance().update(shopTableStatus, session);
		}
	}

	public void freeTables(List<ShopTableStatus> tableStatusList) {
		if (tableStatusList == null) {
			return;
		}

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (ShopTableStatus shopTableStatus : tableStatusList) {
				shopTableStatus.setTableStatus(TableStatus.Available);
				shopTableStatus.removeProperty(ShopTableStatus.SEAT_TIME);
				ShopTableStatusDAO.getInstance().saveOrUpdate(shopTableStatus, session);
			}

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void freeTablesByNumbers(List<Integer> tableNumbers) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			freeTables(tableNumbers, session);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void freeTicketTables(Ticket ticket, Session session) {
		if (ticket == null || session == null) {
			return;
		}
		List<Integer> tableNumbers = ticket.getTableNumbers();
		if (tableNumbers == null) {
			return;
		}
		for (Integer tableNumber : tableNumbers) {
			ShopTableStatus shopTableStatus = (ShopTableStatus) session.get(ShopTableStatus.class, tableNumber);
			List<ShopTableTicket> ticketNumbers = shopTableStatus.getTicketNumbers();
			if (ticketNumbers == null) {
				continue;
			}
			Iterator<ShopTableTicket> iterator = ticketNumbers.iterator();
			while (iterator.hasNext()) {
				ShopTableTicket shopTableTicket = iterator.next();
				String ticketId = shopTableTicket.getTicketId();
				if (ticketId.equals(ticket.getId())) {
					iterator.remove();
				}
			}
			shopTableStatus.setTicketNumbers(ticketNumbers);
			if (ticketNumbers.isEmpty()) {
				shopTableStatus.setTableStatus(TableStatus.Available);
				shopTableStatus.removeProperty(ShopTableStatus.SEAT_TIME);
			}
			saveOrUpdate(shopTableStatus, session);
		}
	}

	public void freeTables(List<Integer> tableNumbers, Session session) {
		for (Integer tableId : tableNumbers) {
			ShopTableStatus shopTableStatus = (ShopTableStatus) session.get(ShopTableStatus.class, tableId);
			shopTableStatus.setTicketId(null);
			shopTableStatus.setTableStatus(TableStatus.Available);
			shopTableStatus.removeProperty(ShopTableStatus.SEAT_TIME);
			saveOrUpdate(shopTableStatus, session);
		}
	}

	public void releaseTables(Ticket ticket) {
		List<ShopTable> tables = getTables(ticket);

		if (tables == null)
			return;

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (ShopTable shopTable : tables) {
				shopTable.setTableStatus(TableStatus.Available);
				ShopTableStatus status = shopTable.getShopTableStatus();
				status.removeProperty(ShopTableStatus.SEAT_TIME);
				ShopTableStatusDAO.getInstance().update(status);
				saveOrUpdate(shopTable, session);
			}

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void releaseAndDeleteTicketTables(Ticket ticket) {

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			releaseTables(ticket);
			ticket.setTableNumbers(null);
			TicketDAO.getInstance().saveOrUpdate(ticket);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	@Override
	public void delete(ShopTable shopTable) throws HibernateException {
		shopTable.setDeleted(Boolean.TRUE);
		super.update(shopTable);
	}

	public void deleteTables(Collection<ShopTable> tables) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (ShopTable shopTable : tables) {
				super.delete(shopTable, session);
			}

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTableType> getTableByTypes(List<ShopTableType> types) {
		List<String> typeIds = new ArrayList<String>();
		for (ShopTableType shopTableType : types) {
			typeIds.add(shopTableType.getId());
		}

		Session session = getSession();
		Criteria criteria = session.createCriteria(ShopTable.class);
		criteria.createAlias("types", "t"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.in("t.id", typeIds)); //$NON-NLS-1$
		//criteria.add(Restrictions("t.id", typeIds)); //$NON-NLS-1$
		criteria.addOrder(Order.asc(ShopTable.PROP_ID));
		return criteria.list();
	}

	public void createNewTables(int totalNumberOfTableHaveToCreate) {

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (int i = 0; i < totalNumberOfTableHaveToCreate; i++) {
				ShopTable table = new ShopTable();
				table.setId(i + 1);
				table.setCapacity(4);
				super.save(table, session);
			}

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}

	}

	public void updateTableStatus(List<Integer> tableNumbers, Integer status, Ticket ticket, boolean saveTicket) {
		try (Session session = this.createNewSession()) {
			this.updateTableStatus(tableNumbers, status, ticket, saveTicket, session);
		}
	}

	public void updateTableStatus(List<Integer> tableNumbers, Integer status, Ticket ticket, boolean saveTicket, Session session) {
		if (tableNumbers == null || tableNumbers.isEmpty()) {
			return;
		}
		Transaction transaction = null;
		try {
			transaction = session.beginTransaction();
			updateTableStatus(tableNumbers, status, ticket, session);
			if (saveTicket) {
				session.saveOrUpdate(ticket);
			}
			transaction.commit();
		} catch (Exception e0) {
			if (transaction != null) {
				transaction.rollback();
			}
			PosLog.error(getClass(), e0);
			throw new RuntimeException(e0);
		}
	}

	public void updateTableStatus(List<Integer> tableNumbers, Integer status, Ticket ticket, Session session) {
		String ticketId = null;
		List<String> tickets = new ArrayList<>();
		String userId = null;
		String userName = null;

		if (ticket != null) {
			ticketId = ticket.getId();
			userId = ticket.getOwner().getId();
			userName = ticket.getOwner().getFirstName();
			tickets.add(ticket.getId());
		}
		//String whereClause = "";
		for (Iterator iterator = tableNumbers.iterator(); iterator.hasNext();) {
			Integer integer = (Integer) iterator.next();
			ShopTableStatus tableStatus = ShopTableStatusDAO.getInstance().get(integer, ticket.getOutletId(), session);
			if (tableStatus != null) {
				if (ticket != null)
					tableStatus.setTableTicket(ticketId, ticket.getTokenNo(), userId, userName, ticket.getCreateDate());
				else
					tableStatus.setTicketId(null);

				session.saveOrUpdate(tableStatus);
			}
			/*whereClause += "id = " + integer;
			if (iterator.hasNext()) {
				whereClause += " or ";
			}*/
		}
		/*String sql = "";
		if (ticket == null) {
			sql = "update ShopTable set ticketId=null, userId=null, userName=null, tableStatusNum=null where %s";
			sql = String.format(sql, whereClause);
		}
		else {
			sql = "update ShopTable set ticketId='%s', userId='%s', userName='%s', tableStatusNum=%s where %s";
			sql = String.format(sql, ticketId, userId, userName, status, whereClause);
		}
		Query query = session.createQuery(sql);
		query.executeUpdate();*/
	}

	public List<ShopTable> findByCapacity(String searchSubject) {

		Session session = null;

		try {
			session = createNewSession();

			Criteria criteria = session.createCriteria(getReferenceClass());

			if (StringUtils.isEmpty(searchSubject)) {
				return null;
			}
			if (StringUtils.isNumeric(searchSubject)) {
				criteria.add(Restrictions.eq(ShopTable.PROP_CAPACITY, Integer.valueOf(searchSubject)));
			}
			//			else {
			//				criteria.createAlias("types", "types");
			//				criteria.add(Restrictions.ilike("types.name", searchSubject, MatchMode.ANYWHERE));
			//			}
			addDeletedFilter(criteria);
			List<ShopTable> list = criteria.list();
			if (list != null || list.size() > 0) {
				for (ShopTable shopTable : list) {
					Hibernate.initialize(shopTable.getTypes());
				}
				return list;
			}
			return null;
		} catch (Exception e) {
			PosLog.error(getReferenceClass(), e.getMessage(), e);
		} finally {
			closeSession(session);
		}
		return null;

	}

	public List<String> getTableNames(List tableNumbers) {

		Session session = null;

		try {
			session = createNewSession();

			String queryString = String.format("select %s, %s from %s table where ", ShopTable.PROP_ID, ShopTable.PROP_NAME, ShopTable.class.getSimpleName()); //$NON-NLS-1$
			Query query = session.createQuery(queryString + "table.id IN (:tableNumbers)"); //$NON-NLS-1$
			query.setParameterList("tableNumbers", tableNumbers); //$NON-NLS-1$
			List list = query.list();
			if (list == null || list.isEmpty()) {
				return null;
			}
			List<String> names = new ArrayList<String>();
			for (int i = 0; i < list.size(); i++) {
				Object[] obj = (Object[]) list.get(i);
				String tableName = (String) obj[1];
				if (StringUtils.isNotBlank(tableName)) {
					names.add(tableName);
				}
				else {
					names.add(String.valueOf(obj[0]));
				}
			}
			return names;
		} catch (Exception e) {
			PosLog.error(ShopTableDAO.class, e.getMessage(), e);
		} finally {
			closeSession(session);
		}
		return null;

	}

	//	public ShopTable loadWithSeats(Integer tableId) {
	//		if (tableId == null) {
	//			return null;
	//		}
	//		Session session = null;
	//		try {
	//			session = createNewSession();
	//			ShopTable shopTable = get(tableId, session);
	//			Hibernate.initialize(shopTable.getSeats());
	//			Set<ShopSeat> seats = shopTable.getSeats();
	//			for (ShopSeat shopSeat : seats) {
	//				shopSeat.setShopTable(shopTable);
	//			}
	//			return shopTable;
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	public void initializeSeats(ShopTable shopTable) {
		if (shopTable == null || shopTable.getId() == null) {
			return;
		}
		if (Hibernate.isInitialized(shopTable.getSeats())) {
			return;
		}
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(shopTable);
			Hibernate.initialize(shopTable.getSeats());
		} finally {
			closeSession(session);
		}
	}

	public void initializeTablesSeats(List<ShopTable> tables) {
		Session session = null;
		try {
			session = createNewSession();
			for (ShopTable table : tables) {
				session.refresh(table);
				Hibernate.initialize(table.getSeats());
			}
		} finally {
			closeSession(session);
		}

	}

	public void detachFloor(List<ShopTable> selectedTables) {
		if (selectedTables == null || selectedTables.isEmpty())
			return;
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (ShopTable shopTable : selectedTables) {
				shopTable.setFloorId(null);
				saveOrUpdate(shopTable, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTable> findAllWithTypes() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(ShopTable.class);
			this.addDeletedFilter(criteria);
			List<ShopTable> list = criteria.list();
			for (ShopTable shopTable : list) {
				Hibernate.initialize(shopTable.getTypes());
			}
			return list;
		}
	}

	public void initializeTypes(ShopTable shopTable) {
		if (shopTable == null || shopTable.getId() == null) {
			return;
		}
		if (Hibernate.isInitialized(shopTable.getTypes())) {
			return;
		}
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(shopTable);
			Hibernate.initialize(shopTable.getTypes());
		} finally {
			closeSession(session);
		}

	}

	public void updateTableList(List<ShopTable> tables, Session session) {
		for (ShopTable shopTable : tables) {
			update(shopTable, session);
		}
	}

	public void updateTableList(List<ShopTable> tables) {
		if (tables == null || tables.isEmpty())
			return;
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			updateTableList(tables, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTable> findByBookingInfo(String bookingInfoId) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(ShopTable.class);
			criteria.add(Restrictions.eq(ShopTable.PROP_CURRENT_BOOKING_ID, bookingInfoId));
			List<ShopTable> list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public Long getTotalCapacity(Date fromDate, Date toDate, BookingInfo currentBookingInfo) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(ShopTable.class);
			criteria.setProjection(Projections.sum(ShopTable.PROP_CAPACITY));
			criteria.add(Restrictions.eq(ShopTable.PROP_RESERVABLE, true));
			Long totalCapacity = (Long) criteria.uniqueResult();
			if (totalCapacity == null) {
				totalCapacity = 0L;
			}
			criteria = session.createCriteria(BookingInfo.class);
			criteria.add(Restrictions.and(Restrictions.gt(BookingInfo.PROP_TO_DATE, fromDate), Restrictions.lt(BookingInfo.PROP_FROM_DATE, toDate)))
					.add(Restrictions.or(Restrictions.eq(BookingInfo.PROP_STATUS, BookingInfo.STATUS_OPEN),
							Restrictions.eq(BookingInfo.PROP_STATUS, BookingInfo.STATUS_SEAT)));

			List<BookingInfo> list = criteria.list();
			List<ShopTable> bookedTableList = new ArrayList<ShopTable>();
			for (Iterator iterator = list.iterator(); iterator.hasNext();) {
				BookingInfo bookingInfo = (BookingInfo) iterator.next();
				for (ShopTable shopTable : bookingInfo.getTables()) {
					bookedTableList.add(shopTable);
				}
			}
			int bookedTableCapacity = 0;
			for (ShopTable shopTable : bookedTableList) {
				bookedTableCapacity += shopTable.getCapacity();
			}

			criteria.setProjection(null);
			criteria.setProjection(Projections.sum(BookingInfo.PROP_GUEST_COUNT));
			Long totalGuestCount = (Long) criteria.uniqueResult();
			if (totalGuestCount == null) {
				totalGuestCount = 0L;
			}

			boolean isContain = false;
			List<BookingInfo> bookingInfoByDate = BookingInfoDAO.getInstance().getAllBookingInfoByDate(fromDate, toDate);
			for (BookingInfo bookingInfo2 : bookingInfoByDate) {
				if (currentBookingInfo.equals(bookingInfo2)) {
					isContain = true;
					break;
				}
			}

			int currentBookingCapacity = 0;
			int currentBookingGuestCount = 0;
			if (currentBookingInfo.getId() != null) {
				List<ShopTable> shopTablesOfCurrentBookingInfo = currentBookingInfo.getTables();
				for (ShopTable shopTable : shopTablesOfCurrentBookingInfo) {
					currentBookingCapacity += shopTable.getCapacity();
				}
				currentBookingGuestCount = currentBookingInfo.getGuestCount();
			}

			int totalBookedNumber = 0;
			for (BookingInfo bookingInfo : list) {
				if (bookingInfo.getTables() != null && !bookingInfo.getTables().isEmpty()) {
					for (ShopTable shopTable : bookingInfo.getTables()) {
						if (bookingInfo.getGuestCount() > shopTable.getCapacity()) {
							totalBookedNumber += bookingInfo.getGuestCount();
						}
						else {
							totalBookedNumber += shopTable.getCapacity();
						}
					}
				}
				else {
					totalBookedNumber += bookingInfo.getGuestCount();
				}
			}

			int sumTotalCapacity = 0;
			int differenceCapacity = totalBookedNumber - bookedTableCapacity;
			List<ShopTable> shopTables = findAll();
			List<ShopTable> availableTables = (List<ShopTable>) CollectionUtils.subtract(shopTables, bookedTableList);
			for (ShopTable shopTable : availableTables) {
				sumTotalCapacity = sumTotalCapacity + shopTable.getCapacity();
			}

			sumTotalCapacity -= differenceCapacity;
			if (isContain) {
				if (currentBookingCapacity > currentBookingGuestCount) {
					sumTotalCapacity += currentBookingCapacity;
				}
				else {
					sumTotalCapacity += currentBookingGuestCount;
				}
			}

			if (sumTotalCapacity < 0) {
				sumTotalCapacity = 0;
			}
			return (long) sumTotalCapacity;
			//			}
			//			else {
			//				if (bookedTableCapacity < totalGuestCount) {
			//					totalCapacity = totalCapacity - bookedTableCapacity - totalGuestCount;
			//				}
			//				else {
			//					totalCapacity = totalCapacity - bookedTableCapacity;
			//				}
			//			}
			//
			//			if (totalCapacity < 0) {
			//				totalCapacity = 0L;
			//			}
			//			return totalCapacity;
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTable> findReservableTables() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(ShopTable.class).createAlias(ShopTable.PROP_SHOP_TABLE_STATUS, "status"); //$NON-NLS-1$
			criteria.add(Restrictions.ne("status." + ShopTableStatus.PROP_TABLE_STATUS_NUM, TableStatus.Disable.getValue())); //$NON-NLS-1$
			criteria.add(Restrictions.eq(ShopTable.PROP_RESERVABLE, true));
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(ShopTable.PROP_ID));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public boolean deleteAllTables() {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			Criteria criteria = session.createCriteria(ShopTable.class);
			criteria.createAlias(ShopTable.PROP_SHOP_TABLE_STATUS, "shopTableStatus"); //$NON-NLS-1$
			criteria.add(Restrictions.or(Restrictions.eq("shopTableStatus." + ShopTableStatus.PROP_TABLE_STATUS_NUM, TableStatus.Available.getValue()), //$NON-NLS-1$
					Restrictions.eq("shopTableStatus." + ShopTableStatus.PROP_TABLE_STATUS_NUM, TableStatus.Disable.getValue()))); //$NON-NLS-1$
			addDeletedFilter(criteria);
			List<ShopTable> availableTableList = criteria.list();
			if (availableTableList == null || availableTableList.isEmpty()) {
				throw new PosException("Available tables not found!"); //$NON-NLS-1$
			}
			for (ShopTable table : availableTableList) {
				List<BookingInfo> bookingList = BookingInfoDAO.getInstance().findByShopTable(table, session);
				boolean isContainNotClosedBookingInfo = false;
				if (bookingList != null && !bookingList.isEmpty()) {
					for (BookingInfo info : bookingList) {
						if (!info.isClosed()) {
							isContainNotClosedBookingInfo = true;
							break;
						}
						info.setTables(null);
						BookingInfoDAO.getInstance().saveOrUpdate(info, session);
					}
				}
				if (isContainNotClosedBookingInfo) {
					continue;
				}
				table.setFloor(null);
				table.setTypes(null);
				table.setDeleted(true);
				ShopTableDAO.getInstance().saveOrUpdate(table, session);
			}
			tx.commit();

			criteria = session.createCriteria(ShopTable.class);
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			Object object = criteria.uniqueResult();
			if (object instanceof Long) {
				Long number = (Long) object;
				return number > 0 ? false : true;
			}
			return false;
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
				throw e;
			}
		} finally {
			if (session != null) {
				session.close();
			}
		}
		return false;
	}

	public void saveOrUpdateShoptables(List<ShopTable> dataList, String outletId, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null) {
			return;
		}
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<ShopTable> iterator = dataList.iterator(); iterator.hasNext();) {
				ShopTable shopTable = (ShopTable) iterator.next();
				updateShopTableOutlet(shopTable, outletId);

				ShopTable existingItem = get(shopTable.getId(), shopTable.getOutletId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(shopTable.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getReferenceClass(), shopTable.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
				}
				initializeSeats(existingItem);
				initializeTypes(existingItem);

				Set<ShopSeat> seats = shopTable.getSeats();
				List<ShopTableType> types = shopTable.getTypes();
				shopTable.setSeats(null);
				shopTable.setTypes(null);
				shopTable.setUpdateLastUpdateTime(updateLastUpdateTime);
				shopTable.setUpdateSyncTime(updateSyncTime);

				saveOrSetVersionShopTable(session, shopTable, existingItem, updateLastUpdateTime, updateSyncTime);
				saveOrSetVersionShopSeat(session, shopTable, existingItem, seats);
				saveOrSetVersionShopTableTypes(session, existingItem, existingItem, types);

				shopTable.setSeats(seats);
				shopTable.setTypes(types);
				update(shopTable, session);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public void saveOrUpdateShopTable(ShopTable bean, String outletId) {
		Transaction tx = null;
		Session session = null;
		try {
			session = this.createNewSession();
			tx = session.beginTransaction();
			ShopTableStatus status = bean.getShopTableStatus();
			status.setOutletId(outletId);

			ShopTableDAO.getInstance().saveOrUpdate(bean, session);
			ShopTableStatusDAO.getInstance().saveOrUpdate(status, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	private void updateShopTableOutlet(ShopTable item, String outletId) throws Exception {
		if (outletId == null) {
			return;
		}
		if (StringUtils.isBlank(item.getOutletId())) {
			item.setOutletId(outletId);
		}
		Set<ShopSeat> shopSeats = item.getSeats();
		if (shopSeats != null && shopSeats.size() > 0) {
			for (ShopSeat shopSeat : shopSeats) {
				if (StringUtils.isBlank(shopSeat.getOutletId())) {
					shopSeat.setOutletId(outletId);
				}
			}
		}
		if (item.getShopTableStatus() != null) {
			if (StringUtils.isBlank(item.getShopTableStatus().getOutletId())) {
				item.getShopTableStatus().setOutletId(outletId);
			}
		}
	}

	public void saveOrSetVersionShopSeat(Session session, ShopTable item, ShopTable existingItem, Set<ShopSeat> seats) {
		if (seats != null && seats.size() > 0) {
			for (ShopSeat shopSeat : seats) {
				shopSeat.setTableId(item.getId());
				saveOrUpdateShopSeat(session, shopSeat);
			}
		}
	}

	private void saveOrUpdateShopSeat(Session session, ShopSeat shopSeat) {
		ShopSeatDAO dao = ShopSeatDAO.getInstance();
		ShopSeat existingShopSeat = dao.get(shopSeat.getId());
		if (existingShopSeat == null) {
			dao.save(shopSeat, session);
		}
		else {
			shopSeat.setVersion(existingShopSeat.getVersion());
		}
	}

	public void saveOrSetVersionShopTableTypes(Session session, ShopTable item, ShopTable existingItem, List<ShopTableType> types) {
		if (types != null && types.size() > 0) {
			for (ShopTableType shopTableType : types) {
				saveOrUpdateShopTableType(session, shopTableType);
			}
		}
	}

	private void saveOrUpdateShopTableType(Session session, ShopTableType tableType) {
		ShopTableTypeDAO dao = ShopTableTypeDAO.getInstance();
		ShopTableType existingTableType = dao.get(tableType.getId(), session);
		if (existingTableType == null) {
			tableType.setUpdateLastUpdateTime(false);
			dao.save(tableType, session);
		}
		else {
			tableType.setVersion(existingTableType.getVersion());
		}
	}

	private void saveOrSetVersionShopTable(Session session, ShopTable shopTable, ShopTable existingItem, boolean updateLastUpdateTime, boolean updateSyncTime)
			throws Exception {
		if (existingItem == null) {
			saveOrSetVersionShopTableStatus(session, shopTable);
			ShopTableDAO.getInstance().save(shopTable, session);
		}
		else {
			saveOrSetVersionShopTableStatus(session, shopTable);
			shopTable.setVersion(existingItem.getVersion());
		}
	}

	private void saveOrSetVersionShopTableStatus(Session session, ShopTable shopTable) {
		ShopTableStatus shopTableStatus = shopTable.getShopTableStatus();
		ShopTableStatus existingShopTableStatus = ShopTableStatusDAO.getInstance().get(shopTableStatus.getId(), shopTable.getOutletId(), session);
		if (existingShopTableStatus != null) {
			shopTableStatus.setVersion(existingShopTableStatus.getVersion());
		}
		else {
			shopTableStatus.setUpdateLastUpdateTime(false);
			ShopTableStatusDAO.getInstance().save(shopTableStatus, session);
		}
	}

	public void transferTickets(Ticket ticket, List<Integer> newTatableNumbers) throws Exception {
		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			ShopTableStatusDAO.getInstance().removeTicketFromShopTableStatus(ticket, session);
			ticket.setTableNumbers(newTatableNumbers);
			ShopTableDAO.getInstance().occupyTables(ticket, session);
			TicketDAO.getInstance().update(ticket, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTable> findAllEnableTable() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(ShopTable.class).createAlias(ShopTable.PROP_SHOP_TABLE_STATUS, "status"); //$NON-NLS-1$
			criteria.add(Restrictions.ne("status." + ShopTableStatus.PROP_TABLE_STATUS_NUM, TableStatus.Disable.getValue())); //$NON-NLS-1$
			criteria.addOrder(Order.asc(ShopTable.PROP_ID));
			addDeletedFilter(criteria);
			List<ShopTable> list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public ShopTable get(Integer id, String outletId) {
		return get(new ShopTable(id, outletId));
	}

	public ShopTable loadWithSeats(Integer tableId, String outletId) {
		if (tableId == null) {
			return null;
		}
		Session session = null;
		try {
			session = createNewSession();
			ShopTable shopTable = get(new ShopTable(tableId, outletId), session);
			Hibernate.initialize(shopTable.getSeats());
			Set<ShopSeat> seats = shopTable.getSeats();
			for (ShopSeat shopSeat : seats) {
				shopSeat.setShopTable(shopTable);
			}
			return shopTable;
		} finally {
			closeSession(session);
		}
	}

	public List<ShopTable> findByOutletId(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(ShopTable.class); //$NON-NLS-1$
			criteria.add(Restrictions.eq(ShopTable.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$
			addDeletedFilter(criteria);
			List<ShopTable> list = criteria.list();
			return list;
		}
	}

}