package com.floreantpos.model.dao;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.TestItem;
import com.floreantpos.swing.PaginationSupport;

public class TestItemDAO extends BaseTestItemDAO {

	public TestItemDAO() {
	}

	@Override
	protected void delete(Object obj, Session s) {
		TestItem testItem = (TestItem) obj;
		checkIfTestItemCanbeDeleted(testItem, s);
		testItem.setDeleted(true);
		update(testItem, s);
	}

	private void checkIfTestItemCanbeDeleted(TestItem testItem, Session session) {
		String details = ""; //$NON-NLS-1$;

		List<MenuItem> menuItems = MenuItemDAO.getInstance().getMenuItemsByTestUnitId(testItem.getId(), session);
		if (menuItems != null && !menuItems.isEmpty()) {
			details = String.format("Test unit %s is being used by the following tests:", testItem.getName());
			for (int i = 0; i < menuItems.size(); i++) {
				MenuItem menuItem = menuItems.get(i);
				details += String.format("\n%d. %s", (i + 1), menuItem.getName()); //$NON-NLS-1$
			}
		}

		if (StringUtils.isNotBlank(details)) {
			throw new PosException(String.format("Test unit <b>%s</b> is being used by other tests", testItem.getName()), details);
		}
	}

	public void findByName(PaginationSupport tableModel, String searchString) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TestItem.class);

			addDeletedFilter(criteria);
			addCriteriaPropertyExcludeFilter(criteria, TestItem.PROP_PROPERTIES, TestItem.JESON_PROP_VISIBLE, false);
			criteria.add(Restrictions.ilike(TestItem.PROP_NAME, searchString, MatchMode.ANYWHERE));

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(TestItem.PROP_NAME).ignoreCase());
			criteria.addOrder(Order.asc(TestItem.PROP_SORT_ORDER));
			tableModel.setRows(criteria.list());
		}
	}

	public List<TestItem> findByGroupId(String groupId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TestItem.class);

			addDeletedFilter(criteria);
			addCriteriaPropertyExcludeFilter(criteria, TestItem.PROP_PROPERTIES, TestItem.JESON_PROP_VISIBLE, false);
			if (StringUtils.isNotBlank(groupId)) {
				criteria.add(Restrictions.eq(TestItem.PROP_TEST_ITEM_GROUP_ID, groupId));
			}
			else {
				criteria.add(Restrictions.eqOrIsNull(TestItem.PROP_TEST_ITEM_GROUP_ID, groupId));
			}
			return criteria.list();
		}

	}

	public TestItem findByName(String testUnitName) {
		try (Session session = createNewSession()) {
			return findByName(testUnitName, session);
		}
	}

	public TestItem findByName(String testUnitName, Session session) {
		Criteria criteria = session.createCriteria(TestItem.class);

		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(TestItem.PROP_NAME, testUnitName).ignoreCase());
		criteria.setMaxResults(1);
		return (TestItem) criteria.uniqueResult();
	}

}