/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.report;

import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.floreantpos.POSConstants;
import com.floreantpos.model.Address;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Store;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.PaperSize;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.CurrencyUtil;

import net.sf.jasperreports.engine.JRChild;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JasperCompileManager;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.design.JRDesignBand;
import net.sf.jasperreports.engine.design.JRDesignSubreport;
import net.sf.jasperreports.engine.design.JasperDesign;
import net.sf.jasperreports.engine.util.JRLoader;
import net.sf.jasperreports.engine.xml.JRXmlLoader;

public class ReportUtil {
	private static Log logger = LogFactory.getLog(ReportUtil.class);

	//	private final static String USER_REPORT_DIR = "/printerlayouts/"; //$NON-NLS-1$
	private final static String DEFAULT_REPORT_DIR = "/com/floreantpos/report/template/"; //$NON-NLS-1$
	private static SimpleDateFormat simpleDateFormat = new SimpleDateFormat(DateUtil.getStoreDateFormat() + " hh:mm a"); //$NON-NLS-1$

	public static String TIME_FORMAT = "HH'h 'mm'm'"; //$NON-NLS-1$
	//	public static void populateRestaurantProperties(Map<String, Object> map) {
	//		populateRestaurantProperties(map, false);
	//	}
	//
	//	public static void populateRestaurantProperties(Map<String, Object> map, boolean isPortrait) {
	//		populateRestaurantProperties(map, isPortrait, DataProvider.get().getOutlet());
	//	}

	public static void populateRestaurantProperties(Map<String, Object> map, Outlet outlet) {
		populateRestaurantProperties(map, false, outlet);
	}

	public static void populateRestaurantProperties(Map<String, Object> map, boolean isPortrait, Outlet outlet) {
		JasperReport reportHeader = null;
		if (isPortrait) {
			reportHeader = ReportUtil.getReport(ReportConstants.REPORT_HEADER_PORTRAIT);
		}
		else {
			reportHeader = ReportUtil.getReport(PaperSize.getReportNameAccording2Size(ReportConstants.REPORT_HEADER));
		}

		Store store = StoreDAO.getRestaurant();
		if (store != null) {
			map.put("restaurantName", store.getName()); //$NON-NLS-1$
			//			map.put("addressLine1", store.getAddressLine1()); //$NON-NLS-1$
			//			map.put("addressLine2", store.getAddressLine2()); //$NON-NLS-1$
			//			map.put("addressLine3", store.getAddressLine3()); //$NON-NLS-1$
			//			map.put("phone", store.getTelephone()); //$NON-NLS-1$
			if (outlet != null) {
				map.put("addressLine1", outlet.getAddressLine1()); //$NON-NLS-1$
				if (outlet.getAddressLine2() != null) {
					map.put("addressLine2", outlet.getAddressLine2()); //$NON-NLS-1$
				}
				if (outlet.getAddressLine3() != null) {
					map.put("addressLine3", outlet.getAddressLine3()); //$NON-NLS-1$
				}
				if (outlet.getTelephone() != null) {
					map.put("phone", outlet.getTelephone()); //$NON-NLS-1$
				}
			}
			map.put("reportHeader", reportHeader); //$NON-NLS-1$
			populatePatternExpression(map, outlet);
		}
	}

	public static void populateMedlogicsProperties(Map<String, Object> map, Outlet outlet) {
		populateMedlogicsProperties(map, outlet, PaperSize.A4);
	}

	public static void populateMedlogicsProperties(Map<String, Object> map, Outlet outlet, PaperSize paperSize) {
		JasperReport reportHeader = null;
		if (paperSize == paperSize.A5) {
			reportHeader = getDefaultReport("medlogics_report_header_A5"); //$NON-NLS-1$
		}
		else {
			reportHeader = getDefaultReport(ReportConstants.MEDLOGICS_REPORT_HEADER);
		}

		Store store = StoreDAO.getRestaurant();
		if (store != null) {
			map.put("labName", store.getName()); //$NON-NLS-1$
			if (outlet != null && outlet.getAddressLine2() != null) {
				map.put("addressLine1", outlet.getAddressLine1()+ ", " + outlet.getAddressLine2()); //$NON-NLS-1$
				if (outlet.getAddressLine2() != null) {
					map.put("addressLine2", ""); //$NON-NLS-1$
				}
				Address address = outlet.getAddress();
				String websiteAddress = outlet.getWebsiteAddress();
				if (address != null  || StringUtils.isNotBlank(websiteAddress)) {
					map.put("hotline", "Hotline: " + address.getTelephone()+", "+"Web: " + "<a style='color: black' href='" + websiteAddress + "'>" + websiteAddress + "</a>"); //$NON-NLS-1$
				}
//				if (StringUtils.isNotBlank(websiteAddress)) {
//					map.put("webLink", "Web: " + "<a style='color: black' href='" + websiteAddress + "'>" + websiteAddress + "</a>"); //$NON-NLS-1$
//				}
			}
			map.put("reportHeader", reportHeader); //$NON-NLS-1$

			//			ImageIcon storeLogo = store.getStoreLogo();
			//			if (storeLogo != null) {
			//				map.put("storeLogo", storeLogo.getImage()); // $NON-NLS-1$
			//			}
			ReportUtil.populatePatternExpression(map, outlet);
			map.put("footerPage", ReportUtil.reportLabelWithBoldTag(POSConstants.PAGE));//$NON-NLS-1$
		}
	}

	public static void populatePatternExpression(Map<String, Object> map, Outlet outlet) {

		Locale locale = Locale.getDefault(Locale.Category.FORMAT);
		map.put(JRParameter.REPORT_LOCALE, locale);
		int decimalPlace = outlet.getDecimalPlace();

		switch (decimalPlace) {
			case 2:
				map.put("patternEXP", "#,##0.00"); //$NON-NLS-1$//$NON-NLS-2$
				break;

			case 3:
				map.put("patternEXP", "#,##0.000"); //$NON-NLS-1$//$NON-NLS-2$
				break;

			case 4:
				map.put("patternEXP", "#,##0.0000"); //$NON-NLS-1$//$NON-NLS-2$
				break;

			default:
				map.put("patternEXP", "#,##0.00"); //$NON-NLS-1$//$NON-NLS-2$
				break;
		}

	}

	public static void populatePatternExpression(Map<String, Object> map, Store store) {
		Locale locale = Locale.getDefault(Locale.Category.FORMAT);
		map.put(JRParameter.REPORT_LOCALE, locale);
		int decimalPlace = store.getDecimalPlace();

		switch (decimalPlace) {
			case 2:
				map.put("patternEXP", "#,##0.00"); //$NON-NLS-1$//$NON-NLS-2$
				break;

			case 3:
				map.put("patternEXP", "#,##0.000"); //$NON-NLS-1$//$NON-NLS-2$
				break;

			case 4:
				map.put("patternEXP", "#,##0.0000"); //$NON-NLS-1$//$NON-NLS-2$
				break;

			default:
				map.put("patternEXP", "#,##0.00"); //$NON-NLS-1$//$NON-NLS-2$
				break;
		}
	}

	public static JasperReport getReport(String reportName) {
		//		InputStream resource = null;
		//
		//		try {
		//			resource = ReceiptPrintService.class.getResourceAsStream(USER_REPORT_DIR + reportName + ".jasper"); //$NON-NLS-1$
		//			if (resource == null) {
		//				if (new File(ReceiptPrintService.class.getResource(USER_REPORT_DIR + reportName + ".jrxml").getFile()) != null) { //$NON-NLS-1$
		//					return compileReport(reportName);
		//				}
		//				else {
		//					throw new Exception();
		//				}
		//
		//			}
		//			else {
		//				return (JasperReport) JRLoader.loadObject(resource);
		//			}
		//		} catch (Exception e) {
		//			//			logger.error(Messages.getString("ReportUtil.8") + reportName + " from user directory, loading default report"); //$NON-NLS-1$ //$NON-NLS-2$
		//			return getDefaultReport(reportName);
		//
		//		} finally {
		//			IOUtils.closeQuietly(resource);
		//		}

		return getDefaultReport(reportName);
	}

	//	private static JasperReport compileReport(String reportName) throws Exception {
	//		InputStream in = null;
	//		InputStream in2 = null;
	//		FileOutputStream out = null;
	//		File jasperFile = null;
	//
	//		try {
	//			File jrxmlFile = new File(ReceiptPrintService.class.getResource(USER_REPORT_DIR + reportName + ".jrxml").getFile()); //$NON-NLS-1$
	//			File dir = jrxmlFile.getParentFile();
	//			jasperFile = new File(dir, reportName + ".jasper"); //$NON-NLS-1$
	//
	//			in = ReceiptPrintService.class.getResourceAsStream(USER_REPORT_DIR + reportName + ".jrxml"); //$NON-NLS-1$
	//			out = new FileOutputStream(jasperFile);
	//			JasperCompileManager.compileReportToStream(in, out);
	//
	//			in2 = ReceiptPrintService.class.getResourceAsStream(USER_REPORT_DIR + reportName + ".jasper"); //$NON-NLS-1$
	//			return (JasperReport) JRLoader.loadObject(in2);
	//
	//		} catch (Exception e) {
	//			PosLog.info(ReportUtil.class, e + ""); //$NON-NLS-1$
	//			IOUtils.closeQuietly(out);
	//			if (jasperFile != null) {
	//				jasperFile.delete();
	//			}
	//
	//			throw e;
	//		}
	//
	//		finally {
	//			IOUtils.closeQuietly(in);
	//			IOUtils.closeQuietly(in2);
	//			IOUtils.closeQuietly(out);
	//		}
	//	}

	private static JasperReport getDefaultReport(String reportName) {
		InputStream resource = null;

		try {

			resource = ReportUtil.class.getResourceAsStream(DEFAULT_REPORT_DIR + reportName + ".jasper"); //$NON-NLS-1$
			return (JasperReport) JRLoader.loadObject(resource);

		} catch (Exception e) {
			logger.error(e);
			return null;

		} finally {
			IOUtils.closeQuietly(resource);
		}
	}

	public static JasperDesign getDefaultReportJrxml(String reportName) {
		InputStream resource = null;

		try {
			resource = ReportUtil.class.getResourceAsStream(DEFAULT_REPORT_DIR + reportName + ".jrxml"); //$NON-NLS-1$
			return JRXmlLoader.load(resource);

		} catch (Exception e) {
			logger.error(e);
			return null;

		} finally {
			IOUtils.closeQuietly(resource);
		}
	}

	public static void populateReportHeader(HashMap<String, Object> map, Outlet outlet, Date fromDate, Date toDate, String userName) {
		populateReportHeader(map, outlet, fromDate, toDate, userName, false);
	}

	public static void populateReportHeader(HashMap<String, Object> map, Outlet outlet, Date fromDate, Date toDate, String userName, boolean isMonthFormat) {
		String outletName = outlet == null ? POSConstants.ALL : outlet.getName();

		if (isMonthFormat) {
			populateReportTimeMonthFormat(map, fromDate, toDate);
		}
		else {
			populateReportTime(map, fromDate, toDate);
		}

		if (userName != null) {
			map.put("userType", ReportUtil.reportLabelWithBoldTag(POSConstants.USER_TYPE) + " " + userName); //$NON-NLS-1$//$NON-NLS-2$
		}
		else {
			map.put("systemTotal", reportLabelWithBoldTag(POSConstants.SYSTEM_TYPE) + " " + POSConstants.SYSTEM_TOTAL);//$NON-NLS-1$//$NON-NLS-2$
		}
		map.put("outletName", reportLabelWithBoldTag("Outlet: ") + " " + outletName); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("currency", reportLabelWithBoldTag(POSConstants.CURRENCY) + " " + CurrencyUtil.getCurrencyName() + CurrencyUtil.getCurrencySymbolWithBracket()); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static void populateReportTime(HashMap<String, Object> map, Date fromDate, Date toDate) {
		//map.put("reportTime", reportLabelWithBoldTag(POSConstants.REPORT_TIME) + DateUtil.formatReportDate(new Date())); //$NON-NLS-1$
		map.put("reportTime", reportLabelWithBoldTag(POSConstants.REPORT_TIME) + DateUtil.formatDateWithBrowserTimeOffset(new Date(), simpleDateFormat)); //$NON-NLS-1$
		if (fromDate != null) {
			map.put("fromDate", reportLabelWithBoldTag(POSConstants.FROM_DATE) + DateUtil.formatReportDate(fromDate)); //$NON-NLS-1$
		}
		if (toDate != null) {
			map.put("toDate", reportLabelWithBoldTag(POSConstants.TO_DATE) + DateUtil.formatReportDate(toDate)); //$NON-NLS-1$
		}
	}

	public static void populateReportTimeMonthFormat(HashMap<String, Object> map, Date fromDate, Date toDate) {
		map.put("reportTime", reportLabelWithBoldTag(POSConstants.REPORT_TIME) + DateUtil.formatDateWithBrowserTimeOffset(new Date(), simpleDateFormat)); //$NON-NLS-1$
		if (fromDate != null) {
			map.put("fromDate", reportLabelWithBoldTag(POSConstants.FROM_DATE) + DateUtil.format("MMMMM yyyy", fromDate)); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (toDate != null) {
			map.put("toDate", reportLabelWithBoldTag(POSConstants.TO_DATE) + DateUtil.format("MMMMM yyyy", toDate)); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	public static void populateReportFooter(HashMap<String, Object> properties) {
		properties.put("footerPage", reportLabelWithBoldTag(POSConstants.PAGE));//$NON-NLS-1$
	}

	public static String reportLabelWithBoldTag(String tag) {
		String boldTag = "<b>" + tag + "</b>";//$NON-NLS-1$//$NON-NLS-2$
		return boldTag;
	}

	public static String getBoldText(String text) {
		StringBuilder builder = new StringBuilder();
		builder.append("<strong style=\"font-weight:500 px\">"); //$NON-NLS-1$
		builder.append(text);
		builder.append("</strong>"); //$NON-NLS-1$
		return builder.toString();
	}

	public static String getTextWithBracket(String text) {
		return " (" + text + ")"; //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static JasperReport adjustReportHeaderAndFooter(String reportName, int reportFooterMargin, int reportHeaderMargin) throws JRException {
		if (reportFooterMargin > 0 || reportHeaderMargin > 0) {
			JasperDesign jasperDesign = ReportUtil.getDefaultReportJrxml(reportName);
			JRDesignBand pageHeaderBand = (JRDesignBand) jasperDesign.getPageHeader();
			JRDesignBand pageFooterBand = (JRDesignBand) jasperDesign.getPageFooter();
			pageHeaderBand.setHeight(pageHeaderBand.getHeight() + reportHeaderMargin);

			List<JRChild> children = pageHeaderBand.getChildren();
			for (JRChild jrChild : children) {
				if (jrChild instanceof JRDesignSubreport) {
					JRDesignSubreport subreport = (JRDesignSubreport) jrChild;
					subreport.setY(subreport.getY() + reportHeaderMargin);
				}
			}
			pageFooterBand.setHeight(pageFooterBand.getHeight() + reportFooterMargin);
			return JasperCompileManager.compileReport(jasperDesign);
		}
		return ReportUtil.getReport(reportName);
	}

	public static long getHour(long valueInMS) {
		return valueInMS / (60 * 60 * 1000);
	}

	public static long getMin(long valueInMS) {
		long hour = valueInMS / (60 * 60 * 1000);
		valueInMS = valueInMS - hour * (60 * 60 * 1000);
		return valueInMS / (60 * 1000);
	}
}
