/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseBookingInfo;
import com.floreantpos.model.util.DataProvider;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "userName", "customer", "doctor", "referrer", "user" })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement
public class BookingInfo extends BaseBookingInfo implements TimedModel, PropertyContainer2 {
	private static final long serialVersionUID = 1L;

	public static final String STATUS_CANCEL = "cancel"; //$NON-NLS-1$
	public static final String STATUS_CLOSE = "close"; //$NON-NLS-1$
	public static final String STATUS_NO_APR = "no appear"; //$NON-NLS-1$
	public static final String STATUS_SEAT = "seat"; //$NON-NLS-1$
	public static final String STATUS_DELAY = "delay"; //$NON-NLS-1$
	public static final String STATUS_OPEN = "open"; //$NON-NLS-1$
	public static final String STATUS_CONFIRM = "confirm"; //$NON-NLS-1$
	public static final String STATUS_EXPIRED = "expired"; //$NON-NLS-1$

	private transient JsonObject propertiesContainer;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public BookingInfo() {
	}

	/**
	 * Constructor for primary key
	 */
	public BookingInfo(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/
	public String toString() {
		return getId() == null ? "" : getId().toString();
	}

	@XmlTransient
	private Customer customer;
	@XmlTransient
	private Doctor doctor;
	@XmlTransient
	private Agent referrer;

	public void setUser(User user) {
		setUserId(user == null ? null : user.getId());
	}

	public void setCustomer(Customer customer) {
		this.customer = customer;

		if (customer == null) {
			setCustomerId(null);
			removeProperty(Ticket.CUSTOMER_NAME);
			removeProperty("customer.firstName");
			removeProperty(Ticket.CUSTOMER_LAST_NAME);
			removeProperty(Ticket.CUSTOMER_PHONE);
			removeProperty(Ticket.CUSTOMER_TAX_EXEMPT);
			removeProperty(Ticket.CUSTOMER_EMAIL);
			return;
		}
		setCustomerId(customer.getId());

		String name = customer.getFirstName();
		if (org.apache.commons.lang.StringUtils.isNotBlank(customer.getLastName())) {
			name += " " + customer.getLastName();
		}

		addProperty(Ticket.CUSTOMER_NAME, name);
		addProperty("customer.firstName", customer.getFirstName());
		addProperty(Ticket.CUSTOMER_LAST_NAME, customer.getLastName());
		addProperty(Ticket.CUSTOMER_PHONE, customer.getMobileNo());
		addProperty(Ticket.CUSTOMER_TAX_EXEMPT, customer.isTaxExempt().toString());
		addProperty(Ticket.CUSTOMER_EMAIL, customer.getEmail());
	}

	@XmlTransient
	public Customer getCustomer() {
		String customerId = getCustomerId();
		if (customerId == null) {
			return null;
		}
		if (customer != null && customerId.equals(customer.getId())) {
			return customer;
		}
		return DataProvider.get().getCustomer(customerId);
	}

	@XmlTransient
	public String getCustomerName() {
		if (customer != null) {
			return customer.getName();
		}
		return getProperty(Ticket.CUSTOMER_NAME);
	}

	public BookingInfo clone(BookingInfo source) {
		return (BookingInfo) SerializationUtils.clone(source);
	}

	@XmlTransient
	public String getPatientName() {
		return getCustomerName();
	}

	@XmlTransient
	public String getPatientPhoneNo() {
		if (customer != null) {
			return customer.getName();
		}
		return getProperty(Ticket.CUSTOMER_PHONE);
	}

	public void setDoctor(Doctor doctor) {
		this.doctor = doctor;
		if (doctor == null) {
			setDoctorId(null);
			removeProperty("doctorName");
			removeProperty("doctorDesignation");
			removeProperty("doctorDepartment");
			removeProperty("doctorPhone");
			removeProperty("docctorEmail");
			return;
		}
		setDoctorId(doctor.getId());

		addProperty("doctorName", doctor.getName());
		addProperty("doctorDesignation", doctor.getDoctorDesignation());
		addProperty("doctorDepartment", doctor.getDepartmentName());
		addProperty("doctorPhone", doctor.getMobileNo());
		addProperty("docctorEmail", doctor.getEmail());
	}

	@XmlTransient
	public String getDoctorName() {
		if (doctor != null) {
			return doctor.getName();
		}
		return getProperty("doctorName");
	}

	@XmlTransient
	@JsonIgnore
	public List<Symptom> getSymptoms() {
		List<Symptom> symptoms = new ArrayList<>();
		if (!hasProperty("symptoms")) { //$NON-NLS-1$
			return symptoms;
		}

		com.google.gson.JsonArray jsonArray = getPropertyStore().getAsJsonArray("symptoms"); //$NON-NLS-1$
		for (int i = 0; i < jsonArray.size(); i++) {
			JsonObject jsonObject = jsonArray.get(i).getAsJsonObject();
			Symptom symptom = new Symptom();
			symptom.setName(getString(jsonObject, Symptom.PROP_NAME));
			symptom.setId(getString(jsonObject, Symptom.PROP_ID));
			symptoms.add(symptom);
		}
		return symptoms;
	}

	private String getString(JsonObject jsonObject, String key) {
		if (!jsonObject.has(key)) {
			return null;
		}

		return jsonObject.get(key).getAsString();
	}

	public void setSymptoms(List<Symptom> symptoms) {
		if (symptoms == null) {
			symptoms = new ArrayList<>();
		}
		com.google.gson.JsonArray symptomsArray = new com.google.gson.JsonArray();
		for (Symptom symptom : symptoms) {
			JsonObject jsonObject = new JsonObject();
			jsonObject.addProperty(Symptom.PROP_ID, symptom.getId());
			jsonObject.addProperty(Symptom.PROP_NAME, symptom.getName());
			symptomsArray.add(jsonObject);
		}
		getPropertyStore().add("symptoms", symptomsArray); //$NON-NLS-1$
		setProperties(propertiesContainer.toString());
	}

	public void putAdmittedBy(String admittedBy) {
		addProperty("admittedBy", admittedBy); //$NON-NLS-1$
	}

	public String getAdmittedBy() {
		return getProperty("admittedBy"); //$NON-NLS-1$
	}

	public void putAdmittedByPhoneNumber(String admittedByPhoneNumber) {
		addProperty("admittedByPhoneNumber", admittedByPhoneNumber); //$NON-NLS-1$
	}

	public String getAdmittedByPhoneNumber() {
		return getProperty("admittedByPhoneNumber"); //$NON-NLS-1$
	}

	public void putAdmissionReason(String patientAdmissionReason) {
		addProperty("admissionReason", patientAdmissionReason); //$NON-NLS-1$
	}

	public String getAdmissionReason() {
		return getProperty("admissionReason"); //$NON-NLS-1$
	}

	public String getRoomDisplay() {
		StringBuilder bedStringBuilder = new StringBuilder();
		List<Bed> beds = getBeds();
		if (beds != null && beds.size() > 0) {
			for (Iterator iterator = beds.iterator(); iterator.hasNext();) {
				Bed bed = (Bed) iterator.next();
				if (bed.getRoom() == null) {
					continue;
				}
				bedStringBuilder.append(bed.getRoom().getName());
				if (iterator.hasNext()) {
					bedStringBuilder.append(", "); //$NON-NLS-1$
				}
			}
		}
		return bedStringBuilder.toString();
	}

	public String getBedDisplay() {
		StringBuilder bedStringBuilder = new StringBuilder();
		List<Bed> beds = getBeds();
		if (beds != null && beds.size() > 0) {
			for (Iterator iterator = beds.iterator(); iterator.hasNext();) {
				Bed bed = (Bed) iterator.next();
				if (bed.getRoom() == null) {
					continue;
				}
				bedStringBuilder.append(bed.getNumber());
				if (iterator.hasNext()) {
					bedStringBuilder.append(", "); //$NON-NLS-1$
				}
			}
		}
		return bedStringBuilder.toString();
	}

	public void putPatientTemperature(String temperature) {
		addProperty("Patient.temperature", temperature); //$NON-NLS-1$
	}

	public String getPatientTemperature() {
		return getProperty("Patient.temperature", ""); //$NON-NLS-1$
	}

	public void putPatientBpm(String bpm) {
		addProperty("Patient.bpm", bpm); //$NON-NLS-1$
	}

	public String getPatientBpm() {
		return getProperty("Patient.bpm", ""); //$NON-NLS-1$
	}

	public void putPatientSex(String sex) {
		addProperty("Patient.sex", sex); //$NON-NLS-1$
	}

	public String getPatientSex() {
		return getProperty("Patient.sex", ""); //$NON-NLS-1$
	}

	public void putPatientPressureUp(String pressureUp) {
		addProperty("Patient.pressure.up", pressureUp); //$NON-NLS-1$
	}

	public String getPatientPressureUp() {
		return getProperty("Patient.pressure.up", ""); //$NON-NLS-1$
	}

	public void putPatientPressureDown(String pressureDown) {
		addProperty("Patient.pressure.down", pressureDown); //$NON-NLS-1$
	}

	public String getPatientPressureDown() {
		return getProperty("Patient.pressure.down", ""); //$NON-NLS-1$
	}

	public void putPatientAgeYear(String ageYear) {
		addProperty("Patient.age.year", ageYear); //$NON-NLS-1$
	}

	public String getPatientAgeYear() {
		return getProperty("Patient.age.year", ""); //$NON-NLS-1$
	}

	public void putPatientAgeMonth(String ageMonth) {
		addProperty("Patient.age.month", ageMonth); //$NON-NLS-1$
	}

	public String getPatientAgeMonth() {
		return getProperty("Patient.age.month", ""); //$NON-NLS-1$
	}

	public void putPatientAgeDay(String ageDay) {
		addProperty("Patient.age.day", ageDay); //$NON-NLS-1$
	}

	public String getPatientAgeDay() {
		return getProperty("Patient.age.day", ""); //$NON-NLS-1$
	}

	public void putPatientAdmissionType(String admissionType) {
		addProperty("Patient.admissionType", admissionType); //$NON-NLS-1$
	}

	public String getPatientAdmissionType() {
		return getProperty("Patient.admissionType", ""); //$NON-NLS-1$
	}

	public void putPatientWeight(String heightInCm) {
		addProperty("PatientWeight", heightInCm); //$NON-NLS-1$
	}

	public String getPatientWeight() {
		return getProperty("PatientWeight", ""); //$NON-NLS-1$
	}

	public void putPatientHeightInCm(String heightInCm) {
		addProperty("PatientHeightInCm", heightInCm); //$NON-NLS-1$
	}

	public String getPatientHeightInCm() {
		return getProperty("PatientHeightInCm", ""); //$NON-NLS-1$
	}

	public void putNote(String note) {
		addProperty("note", note); //$NON-NLS-1$
	}

	public String getNote() {
		return getProperty("note", ""); //$NON-NLS-1$
	}

	public void setBookingBeds(List<Bed> beds) {
		super.setBeds(beds);
		if (beds == null || beds.isEmpty()) {
			removeProperty("bedNumber");
			removeProperty("roomName");
			return;
		}
		Bed bed = beds.get(0);
		String bedNumber = bed.getNumber();
		String roomName = bed.getRoom() == null ? null : bed.getRoom().getName();
		addProperty("bedNumber", bedNumber); //$NON-NLS-1$
		addProperty("roomName", roomName); //$NON-NLS-1$
	}

	public String getBedNumber() {
		return getProperty("bedNumber"); //$NON-NLS-1$
	}

	public String getRoomName() {
		return getProperty("roomName"); //$NON-NLS-1$
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	public void putReferrer(Agent referrer) {
		this.referrer = referrer;
		if (referrer == null) {
			setReferrerId(null);
			removeProperty("referrerName");
			removeProperty("referrerPhone");
			return;
		}
		setReferrerId(referrer.getId());

		addProperty("referrerName", referrer.getName());
		addProperty("referrerPhone", referrer.getMobileNo());
	}

	@XmlTransient
	public String getReferrerName() {
		if (referrer != null) {
			return referrer.getName();
		}
		return getProperty("referrerName");
	}

	public Agent getReferrer() {
		String referrerId = getReferrerId();
		if (StringUtils.isBlank(referrerId)) {
			return null;
		}
		if (referrer != null && referrerId.equals(referrer.getId())) {
			return referrer;
		}
		return (Agent) DataProvider.get().getCustomer(referrerId);

	}

}