package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderItem;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.util.POSUtil;

public class PurchaseOrderItemDAO extends BasePurchaseOrderItemDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PurchaseOrderItemDAO() {
	}

	public List<MenuItem> getMenuItems(InventoryVendor vendor) {
		Session session = null;
		Criteria criteria = null;
		try {
			List<MenuItem> menuItems = new ArrayList<>();
			session = getSession();
			criteria = session.createCriteria(PurchaseOrderItem.class);
			criteria.setProjection(Projections.distinct(Projections.property(PurchaseOrderItem.PROP_MENU_ITEM_ID)));
			criteria.createAlias(PurchaseOrderItem.PROP_PURCHASE_ORDER, "p"); //$NON-NLS-1$
			criteria.add(Restrictions.eq("p.vendor", vendor)); //$NON-NLS-1$

			List<Integer> list = criteria.list();

			if (list != null) {
				for (Integer item : list) {
					MenuItem menuItem = MenuItemDAO.getInstance().getReplenishedMenuItem(item, session);
					if (menuItem != null) {
						menuItems.add(menuItem);
					}
				}
			}
			return menuItems;
		} finally {
			closeSession(session);
		}
	}

	public void loadCustomers(PurchaseOrderItem purchaseOrderItem) {
		if (purchaseOrderItem == null || purchaseOrderItem.getId() == null)
			return;
		if (Hibernate.isInitialized(purchaseOrderItem.getCustomers())) {
			return;
		}
		Session session = createNewSession();
		try {
			session.refresh(purchaseOrderItem);
			Hibernate.initialize(purchaseOrderItem.getCustomers());
		} finally {
			closeSession(session);
		}
	}

	public void initializeCustomers(PurchaseOrderItem purchaseOrderItem) {
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(purchaseOrderItem);
			if (!Hibernate.isInitialized(purchaseOrderItem.getCustomers())) {
				Hibernate.initialize(purchaseOrderItem.getCustomers());
			}
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public PurchaseOrderItem findCostByVendor(InventoryVendor vendor, MenuItem menuItem) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrderItem.class);
			criteria.createAlias(PurchaseOrderItem.PROP_PURCHASE_ORDER, "p"); //$NON-NLS-1$
			criteria.add(Restrictions.eq("p." + PurchaseOrder.PROP_VENDOR, vendor)); //$NON-NLS-1$
			//criteria.add(Restrictions.eq("p." + PurchaseOrder.PROP_STATUS, PurchaseOrder.ORDER_FULLY_RECEIVED)); //$NON-NLS-1$
			criteria.addOrder(Order.desc("p." + PurchaseOrder.PROP_CREATED_DATE)); //$NON-NLS-1$
			criteria.add(Restrictions.eq(PurchaseOrderItem.PROP_MENU_ITEM_ID, menuItem.getId()));
			criteria.setMaxResults(1);
			return (PurchaseOrderItem) criteria.uniqueResult();
		}
	}

	public void loadPurchaseItemBatch(Pagination tableModel, MenuItem menuItem, String batchSearchString) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrderItem.class);
			if (menuItem != null) {
				criteria.add(Restrictions.eq(PurchaseOrderItem.PROP_MENU_ITEM_ID, menuItem.getId())); //$NON-NLS-1$
			}

			if (StringUtils.isNotBlank(batchSearchString)) {
				criteria.add(Restrictions.ilike(PurchaseOrderItem.PROP_BATCH_NUMBER, batchSearchString, MatchMode.ANYWHERE));
			}
			criteria.add(Restrictions.neOrIsNotNull(PurchaseOrderItem.PROP_BATCH_NUMBER, ""));

			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.groupProperty(PurchaseOrderItem.PROP_BATCH_NUMBER), PurchaseOrderItem.PROP_BATCH_NUMBER);
			pList.add(Projections.min(PurchaseOrderItem.PROP_ID), PurchaseOrderItem.PROP_ID);
			pList.add(Projections.groupProperty(PurchaseOrderItem.PROP_EXPIRE_DATE), PurchaseOrderItem.PROP_EXPIRE_DATE);
			pList.add(Projections.groupProperty(PurchaseOrderItem.PROP_UNIT_COST), PurchaseOrderItem.PROP_UNIT_COST);
			pList.add(Projections.groupProperty(PurchaseOrderItem.PROP_SALES_PRICE), PurchaseOrderItem.PROP_SALES_PRICE);
			criteria.setProjection(pList);

			tableModel.setNumRows(rowCount(criteria));
			criteria.setProjection(pList);
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.desc(PurchaseOrderItem.PROP_EXPIRE_DATE));
			criteria.addOrder(Order.asc(PurchaseOrderItem.PROP_BATCH_NUMBER));

			criteria.setResultTransformer(Transformers.aliasToBean(PurchaseOrderItem.class));
			tableModel.setRows(criteria.list());
		}

	}

	public List<PurchaseOrderItem> findPoItemWithinDate(Date startDate, Date endDate, List<MenuGroup> groups, Outlet outlet) {
		try (Session session = createNewSession()) {
			List<String> groupNameList = groups.stream().map(t -> t.getName()).collect(Collectors.toList());

			Criteria criteria = session.createCriteria(PurchaseOrderItem.class).createAlias(PurchaseOrderItem.PROP_PURCHASE_ORDER, "po");
			if (startDate != null) {
				criteria.add(Restrictions.ge("po." + PurchaseOrder.PROP_CREATED_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt("po." + PurchaseOrder.PROP_CREATED_DATE, endDate));
			}
			
			if (!groupNameList.isEmpty()) {
				criteria.add(Restrictions.in(PurchaseOrderItem.PROP_GROUP_NAME, groupNameList));
			}
			
			if (outlet != null) {
				criteria.add(Restrictions.eq("po." + PurchaseOrder.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$
			}
			criteria.add(Restrictions.gt(PurchaseOrderItem.PROP_QUANTITY_RECEIVED, 0.0));

			criteria.addOrder(Order.desc(PurchaseOrder.PROP_CREATED_DATE));
			
			return criteria.list();
		}
	}

}