package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.Bed;
import com.floreantpos.model.Block;
import com.floreantpos.model.Floor;
import com.floreantpos.model.Room;
import com.floreantpos.model.RoomType;
import com.floreantpos.swing.PaginatedListModel;

public class RoomDAO extends BaseRoomDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public RoomDAO() {
	}

	public void loadData(PaginatedListModel<Room> tableModel, String name, Floor floor, Block block, String roomType) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(name)) {
				criteria.add(Restrictions.eq(Room.PROP_NAME, name).ignoreCase());
			}
			if (floor != null) {
				criteria.add(Restrictions.eq(Room.PROP_FLOOR_ID, floor.getId()));
			}
			if (block != null) {
				criteria.add(Restrictions.eq(Room.PROP_BLOCK_ID, block.getId()));
			}
			if (StringUtils.isNotBlank(roomType)) {
				criteria.add(Restrictions.eq(Room.PROP_ROOM_TYPE, roomType).ignoreCase());
			}
			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setData(criteria.list());
		}
	}

	public List<String> getCabinIds() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(Room.PROP_ID));
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Room.PROP_ROOM_TYPE, RoomType.CABIN.name()).ignoreCase());
			return criteria.list();
		}
	}

	public List<String> getWardIds() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(Room.PROP_ID));
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Room.PROP_ROOM_TYPE, RoomType.WARD.name()).ignoreCase());
			return criteria.list();
		}
	}

	public List<Room> findByFloor(String floorId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Room.PROP_FLOOR_ID, floorId));
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		Room data = (Room) obj;
		if (data == null) {
			throw new PosException("Data not found.");
		}
		data.setDeleted(Boolean.TRUE);
		saveOrUpdate(data, session);
	}

	public void saveOrUpdateRoom(Room room, List<Bed> beds) {
		Transaction tx = null;
		try (Session session = getInstance().createNewSession()) {
			tx = session.beginTransaction();
			saveOrUpdate(room, session);
			if (beds != null) {
				for (Bed bed : beds) {
					bed.setRoomId(room.getId());
					BedDAO.getInstance().saveOrUpdate(bed, session);
				}
			}
			tx.commit();

		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
		}
	}

	public List<String> getRoomIds(String bedType) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(Room.PROP_ID));
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Room.PROP_ROOM_TYPE, bedType).ignoreCase());
			return criteria.list();
		}
	}

	public List<Room> findByIds(List<String> roomIds) {
		if (roomIds == null || roomIds.isEmpty()) {
			return new ArrayList<Room>();
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.in(Room.PROP_ID, roomIds));
			return criteria.list();
		}
	}
}