package com.floreantpos.model;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.model.base.BaseBloodInventoryTransaction;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

public class BloodInventoryTransaction extends BaseBloodInventoryTransaction implements PropertyContainer2, TimedModel {
	private static final long serialVersionUID = 1L;

	private transient JsonObject propertiesContainer;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public BloodInventoryTransaction() {
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for primary key
	 */
	public BloodInventoryTransaction(java.lang.String id) {
		super(id);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/*[CONSTRUCTOR MARKER END]*/

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	public String getQuantityDisplay() {
		Integer qty = getQuantity();
		return qty + " Bag";
	}

	public String getVolumeDisplay() {
		String volumeDisplay = StringUtils.EMPTY;
		Double volume = getVolume();
		if (volume > 0) {
			volumeDisplay = NumberUtil.formatNumberIfNeeded(volume);
		}
		String unit = getUnit();
		if (StringUtils.isNotBlank(unit)) {
			volumeDisplay += " " + unit;
		}

		return volumeDisplay;
	}

	public InventoryTransactionType getTransactionType() {
		return InventoryTransactionType.fromInt(super.getType());
	}

	public void putBeforeOnHandQty(double quantity) {
		addProperty("Onhandqty.before", NumberUtil.formatAmount(quantity));
	}

	public double getBeforeOnHandQty() {
		if (hasProperty("Onhandqty.before")) { //$NON-NLS-1$
			return POSUtil.parseDouble(getProperty("Onhandqty.before")); //$NON-NLS-1$
		}
		return 0;
	}

	public void putAfterOnHandQty(double quantity) {
		addProperty("Onhandqty.after", NumberUtil.formatAmount(quantity));
	}

	public Double getAfterOnHandQty() {
		if (hasProperty("Onhandqty.after")) { //$NON-NLS-1$
			return POSUtil.parseDouble(getProperty("Onhandqty.after")); //$NON-NLS-1$
		}
		return null;
	}

	public void setDonor(Donor donor) {
		if (donor != null) {
			setDonorId(donor.getId());
			putDonorName(donor.getName());
			putDonorPhone(donor.getContactNumber());
		}
		else {
			setDonorId(null);
			putDonorName(null);
			putDonorPhone(null);
		}
	}

	public void setPatient(Patient patient) {
		if (patient != null) {
			setPatientId(patient.getId());
			addProperty("patient.name", patient.getName());
			addProperty("patient.mobile", patient.getMobileNo());
		}
		else {
			setPatientId(null);
			addProperty("patient.name", null);
			addProperty("patient.mobile", null);
		}
	}

	public String getDonorName() {
		return getProperty("donor.name", StringUtils.EMPTY);
	}

	public void putDonorName(String donorName) {
		addProperty("donor.name", donorName);
	}

	public String getDonorMobile() {
		return getProperty("donor.mobile", StringUtils.EMPTY);
	}

	public void putDonorPhone(String donorMobile) {
		addProperty("donor.mobile", donorMobile);
	}

	public String getPatientName() {
		return getProperty("patient.name", StringUtils.EMPTY);
	}

	public String getPatientMobile() {
		return getProperty("patient.mobile", StringUtils.EMPTY);
	}

}