package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang3.SerializationUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.BloodDonationHistory;
import com.floreantpos.model.BloodInventory;
import com.floreantpos.model.BloodInventoryTransaction;
import com.floreantpos.model.Donor;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.Patient;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginatedListModel;

public class BloodInventoryTransactionDAO extends BaseBloodInventoryTransactionDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		BloodInventoryTransaction bean = (BloodInventoryTransaction) obj;
		if (bean == null) {
			throw new PosException("Blood inventory transaction not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<BloodInventoryTransaction> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BloodInventoryTransaction.class);
			addDeletedFilter(criteria);

			criteria.addOrder(Order.asc(BloodInventory.PROP_TRANSACTION_DATE));
			return criteria.list();
		}
	}

	public void adjustBloodInventoryStock(BloodInventoryTransaction bloodInvTrans, Session session) {
		double transQty = bloodInvTrans.getQuantity();

		BloodInventory bloodInventory = null;

		PosLog.debug(getClass(),
				bloodInvTrans.getBloodGroup() + " |" + bloodInvTrans.getTransactionType() + "| |" + transQty + "| " + bloodInvTrans.getInventoryType());

		if (bloodInvTrans.getTransactionType() == InventoryTransactionType.IN) {
			String inventoryToLocation = bloodInvTrans.getToLocationId();
			bloodInventory = BloodInventoryDAO.getInstance().getBloodInventoryStock(bloodInvTrans, inventoryToLocation, session);
			updateOnHandQty(bloodInvTrans, bloodInventory);
			createBloodStockItem(bloodInvTrans, bloodInventory, session);
			this.saveOrUpdate(bloodInvTrans, session);
		}

		if (bloodInvTrans.getTransactionType() == InventoryTransactionType.OUT) {
			String inventoryFromLocation = bloodInvTrans.getFromLocationId();
			bloodInventory = BloodInventoryDAO.getInstance().getBloodInventoryStock(bloodInvTrans, inventoryFromLocation, session);
			updateOnHandQty(bloodInvTrans, bloodInventory);
			createBloodStockItem(bloodInvTrans, bloodInventory, session);
			this.saveOrUpdate(bloodInvTrans, session);
		}
	}

	private void createBloodStockItem(BloodInventoryTransaction trans, BloodInventory bloodInventory, Session session) {
		if (bloodInventory == null) {
			bloodInventory = new BloodInventory();
			if (trans.getTransactionType() == InventoryTransactionType.IN) {
				bloodInventory.setOutletId(trans.getToOultetId());
			}
			else {
				bloodInventory.setOutletId(trans.getOutletId());
			}

			//			if (inventoryLocation != null) {
			//				bloodInventory.setLocationId(inventoryLocation.getId());
			//				bloodInventory.setOutletId(inventoryLocation.getOutletId());
			//			}

			bloodInventory.setBloodGroup(trans.getBloodGroup());
			bloodInventory.setBloodType(trans.getBloodType());
			bloodInventory.setVolume(trans.getVolume());
			bloodInventory.setUnit(trans.getUnit());

			//			bloodInventory.setUnitType(trans.getUnitType());
			//			bloodInventory.setUnitCost(trans.getUnitCost());
			//			if (trans.getUnitPrice() > 0.0) {
			//				bloodInventory.setUnitPrice(trans.getUnitPrice());
			//			}
		}

		if (trans.getTransactionType() == InventoryTransactionType.IN) {
			bloodInventory.setQuantity(bloodInventory.getQuantity() + trans.getQuantity());
		}
		if (trans.getTransactionType() == InventoryTransactionType.OUT) {
			if (bloodInventory.getQuantity() < trans.getQuantity()) {
				throw new PosException("There is not enough stock in inventory");
			}
			bloodInventory.setQuantity(bloodInventory.getQuantity() - trans.getQuantity());
		}

		bloodInventory.setTransactionDate(StoreDAO.getServerTimestamp());
		BloodInventoryDAO.getInstance().saveOrUpdate(bloodInventory, session);

	}

	private void updateOnHandQty(BloodInventoryTransaction bloodInventoryTransaction, BloodInventory bloodInventory) {
		double quantityInHand = bloodInventory == null ? 0 : bloodInventory.getQuantity();
		bloodInventoryTransaction.putBeforeOnHandQty(quantityInHand);
		if (bloodInventoryTransaction.getTransactionType() == InventoryTransactionType.IN) {
			bloodInventoryTransaction.putAfterOnHandQty(quantityInHand + bloodInventoryTransaction.getQuantity());
		}
		else if (bloodInventoryTransaction.getTransactionType() == InventoryTransactionType.OUT) {
			bloodInventoryTransaction.putAfterOnHandQty(quantityInHand - bloodInventoryTransaction.getQuantity());
		}
	}

	private BloodInventoryTransaction createBloodInventoryTransaction(BloodDonationHistory bloodDonationHistory, InventoryTransactionType transactionType) {
		BloodInventoryTransaction bloodInventoryTransaction = new BloodInventoryTransaction();

		bloodInventoryTransaction.setType(transactionType.getType());
		bloodInventoryTransaction.setUserId(DataProvider.get().getCurrentUser().getId());
		bloodInventoryTransaction.setOutletId(DataProvider.get().getCurrentOutletId());
		bloodInventoryTransaction.setToOultetId(DataProvider.get().getCurrentOutletId());

		Integer stockDifference = bloodDonationHistory.getQuantity();

		bloodInventoryTransaction.setTransactionDate(StoreDAO.getServerTimestamp());
		bloodInventoryTransaction.setQuantity(stockDifference);

		bloodInventoryTransaction.setBloodGroup(bloodDonationHistory.getBloodGroup());
		bloodInventoryTransaction.setBloodType(bloodDonationHistory.getBloodType());
		bloodInventoryTransaction.setUnit(bloodDonationHistory.getUnit());
		bloodInventoryTransaction.setVolume(bloodDonationHistory.getVolume());

		bloodInventoryTransaction.setDonor(bloodDonationHistory.getDonor());
		bloodInventoryTransaction.setBagNumber(bloodDonationHistory.getBloodBagNo());

		String strReason = (stockDifference > 0) ? InventoryTransaction.REASON_ADJUST_IN : InventoryTransaction.REASON_ADJUST_OUT;
		bloodInventoryTransaction.setInventoryType(strReason);

		if (stockDifference > 0) {
			bloodInventoryTransaction.setType(InventoryTransactionType.IN.getType());
			//bloodInventoryTransaction.setToLocationId(location);
		}
		else {
			bloodInventoryTransaction.setType(InventoryTransactionType.OUT.getType());
			//bloodInventoryTransaction.setFromLocationId(location);
		}
		return bloodInventoryTransaction;
	}

	public void adjustBloodInventory(BloodDonationHistory bloodDonationHistory, InventoryTransactionType transactionType) throws Exception {

		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();
			BloodInventoryTransaction invTrans = createBloodInventoryTransaction(bloodDonationHistory, transactionType);

			BloodInventoryTransactionDAO.getInstance().adjustBloodInventoryStock(invTrans, session);
			transaction.commit();
		}

		//ActionHistoryDAO.logJournalForInventoryTrans(invTrans);
	}

	public void loadTransactions(Patient patient, Donor donor, PaginatedListModel<BloodInventoryTransaction> dataModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BloodInventoryTransaction.class);

			if (patient != null) {
				criteria.add(Restrictions.eq(BloodInventoryTransaction.PROP_PATIENT_ID, patient.getId()));
			}

			if (donor != null) {
				criteria.add(Restrictions.eq(BloodInventoryTransaction.PROP_DONOR_ID, donor.getId()));
			}

			dataModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());

			criteria.addOrder(Order.desc(BloodInventoryTransaction.PROP_TRANSACTION_DATE));

			dataModel.setData(criteria.list());
		}
	}

	public void saveTransaction(BloodInventoryTransaction bloodInventoryTransaction, List<BloodInventoryTransaction> selectedBags) {
		try (Session session = createNewSession()) {
			Transaction beginTransaction = session.beginTransaction();

			for (BloodInventoryTransaction selectedBag : selectedBags) {
				BloodInventoryTransaction cloneBloodInventoryTrans = SerializationUtils.clone(bloodInventoryTransaction);
				cloneBloodInventoryTrans.setQuantity(selectedBag.getQuantity());
				cloneBloodInventoryTrans.setBagNumber(selectedBag.getBagNumber());
				cloneBloodInventoryTrans.setDonorId(selectedBag.getDonorId());
				cloneBloodInventoryTrans.putDonorName(selectedBag.getDonorName());
				cloneBloodInventoryTrans.putDonorPhone(selectedBag.getDonorMobile());
				BloodInventoryTransactionDAO.getInstance().adjustBloodInventoryStock(cloneBloodInventoryTrans, session);

				BloodInventoryTransaction trans = session.get(BloodInventoryTransaction.class, selectedBag.getId());
				trans.setBloodIssued(true);
				session.saveOrUpdate(trans);
			}

			beginTransaction.commit();
		}
	}

}