/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;

import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.ExpenseTransaction;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.dao.BalanceUpdateTransactionDAO;
import com.floreantpos.model.dao.PosTransactionDAO;

public class UpdateDBTo416 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo416(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void updateTransactionFields() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		int totalTransactions = rowCount();
		PosLog.info(UpdateDBTo416.class, "Found total transactions: " + totalTransactions); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalTransactions) {
			try (Session session = sessionFactory.openSession()) {
				Transaction transaction = session.beginTransaction();

				Pagination<PosTransaction> pagination = new Pagination<PosTransaction>(count, increment);
				pagination.setNumRows(totalTransactions);
				findPosTransactions(pagination, session);

				List<PosTransaction> posTransactions = pagination.getDataList();
				for (PosTransaction posTransaction : posTransactions) {
					StringBuilder sb = new StringBuilder();

					String accountManagerId = posTransaction.getProperty("user.AMId"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(accountManagerId)) {
						posTransaction.setAccountManagerId(accountManagerId);

						if ("1".equals(accountManagerId)) {
							posTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_STORE);
						}
						else {
							posTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_ACM);
						}

						sb.append("accountManagerId: " + accountManagerId);
						sb.append(", ");
					}

					String bankAccountId = posTransaction.getProperty("bank_account_id"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(bankAccountId)) {
						posTransaction.setBankAccountId(bankAccountId);

						sb.append("bankAccountId: " + bankAccountId);
						sb.append(", ");
					}

					String vendorId = posTransaction.getProperty("vendor_id"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(vendorId)) {
						posTransaction.setVendorId(vendorId);

						sb.append("vendorId: " + vendorId);
						sb.append(", ");
					}

					//					String sourceType = posTransaction.getProperty("source_type"); //$NON-NLS-1$//$NON-NLS-2$
					//					if (StringUtils.isNotBlank(sourceType)) {
					//						posTransaction.setSourceType(sourceType);
					//
					//						sb.append("sourceType: " + sourceType);
					//						sb.append(", ");
					//					}

					String sourceId = posTransaction.getProperty("source_id"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(sourceId)) {
						posTransaction.putBalanceUpdateTransId(sourceId);
						posTransaction.setSourceId(null);
						sb.append("sourceId: " + sourceId);

					}

					String message = sb.toString();
					if (StringUtils.isNotBlank(message)) {
						PosLog.info(getClass(), "PosTransaction " + posTransaction.getId() + " is updating from property to field: " + message);
						PosTransactionDAO.getInstance().saveOrUpdate(posTransaction, session);
					}

				}
				transaction.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

	private void findPosTransactions(Pagination<PosTransaction> pagination, Session session) {
		Criteria criteria = createTransactionsCriteria(session);
		criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(),
				String.format("Searching from: %s to %s", pagination.getCurrentRowIndex(), (pagination.getCurrentRowIndex() + pagination.getPageSize()))); //$NON-NLS-1$

		pagination.setRows(criteria.list());

	}

	public int rowCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createTransactionsCriteria(session);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createTransactionsCriteria(Session session) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		return criteria;
	}

	public void updateBalanceUpdateTransactionFields() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.setProjection(Projections.rowCount());

			int totalTransactions = ((Long) criteria.uniqueResult()).intValue();

			PosLog.info(UpdateDBTo416.class, "Found total balance update transactions: " + totalTransactions); //$NON-NLS-1$

			criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.setMaxResults(totalTransactions);

			List<BalanceUpdateTransaction> posTransactions = criteria.list();
			
			Transaction transaction = session.beginTransaction();
			for (BalanceUpdateTransaction posTransaction : posTransactions) {

				String memoNo = posTransaction.getProperty("memo_no");
				String bankAccountId = posTransaction.getProperty("bank_account_id");

				if (StringUtils.isBlank(memoNo) && StringUtils.isBlank(bankAccountId)) {
					continue;
				}

				posTransaction.setMemoNo(memoNo);
				posTransaction.setBankAccountId(bankAccountId);

				BalanceUpdateTransactionDAO.getInstance().saveOrUpdate(posTransaction, session);
				PosLog.info(getClass(), "Updated balance update transaction " + posTransaction.getId()); //$NON-NLS-1$

			}
			transaction.commit();
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}
}