package com.floreantpos.report;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosLog;
import com.openhtmltopdf.java2d.api.BufferedImagePageProcessor;
import com.openhtmltopdf.java2d.api.Java2DRendererBuilder;
import com.openhtmltopdf.outputdevice.helper.BaseRendererBuilder.PageSizeUnits;

public class HtmlToImageConverter {

	public static List<BufferedImage> toImages(String body, String footer, Map<String, Object> dataMap) throws IOException {
		return toImages(body, footer, dataMap, null);
	}

	public static List<BufferedImage> toImages(String body, String footer, Map<String, Object> dataMap, String customFontPath) throws IOException {
		body = body.replaceAll("&", "&amp;"); //$NON-NLS-1$ //$NON-NLS-2$
		body = body.replaceAll("&nbsp;", " "); //$NON-NLS-1$ //$NON-NLS-2$
		float pageHeight = 11.9f;
		List<BufferedImage> convertedHtmlBodyImages = convertToImages(body, pageHeight, customFontPath);
		List<BufferedImage> images = new ArrayList<>();
		if (convertedHtmlBodyImages != null && convertedHtmlBodyImages.size() > 0) {
			int pageSize = convertedHtmlBodyImages.size();
			int count = 1;
			for (BufferedImage contentImage : convertedHtmlBodyImages) {
				int bodyWidth = contentImage.getWidth();
				int bodyHeight = contentImage.getHeight();
				if (count == pageSize) {
					BufferedImage trimmedContentImage = trim(contentImage);
					if (trimmedContentImage != null) {
						images.add(trimmedContentImage);
					}
					int contentHeight = trimmedContentImage == null ? 0 : trimmedContentImage.getHeight();

					BufferedImage footerImage = trim(convertToSingleImage(footer, 4f, customFontPath));
					int footerHeight = footerImage.getHeight();

					int gap = bodyHeight - contentHeight;
					if (footerHeight > gap) {
						gap = bodyHeight - footerHeight;
					}
					int spaceBeforeSignature = 60;
					int blankSpace = gap - footerHeight + spaceBeforeSignature;
					images.add(fillBlankSpace(bodyWidth, blankSpace));
					images.add(footerImage);
				}
				else {
					images.add(contentImage);
				}
				count++;
			}
		}
		return images;
	}

	private static BufferedImage convertToSingleImage(String htmlContent, float height, String customFontPath) throws IOException {
		List<BufferedImage> sections = convertToImages(htmlContent, height, customFontPath);
		if (sections == null || sections.isEmpty()) {
			return null;
		}
		return sections.get(0);
	}

	private static List<BufferedImage> convertToImages(String htmlContent, float height, String customFontPath) throws IOException {
		float pageWidth = 9.3f;
		PageSizeUnits PAGE_SIZE_LETTER_UNITS = PageSizeUnits.INCHES;

		Java2DRendererBuilder builder = new Java2DRendererBuilder();
		builder.withHtmlContent(htmlContent, "/"); //$NON-NLS-1$
		builder.useDefaultPageSize(pageWidth, height, PAGE_SIZE_LETTER_UNITS);
		builder.useEnvironmentFonts(true);
		if (StringUtils.isNotBlank(customFontPath)) {
			File fontFile = new File(customFontPath);
			if (fontFile == null || !fontFile.exists()) {
				PosLog.debug(HtmlToImageConverter.class, "Font not found. Path: " + customFontPath); //$NON-NLS-1$
			}
			builder.useFont(fontFile, "SolaimanLipi"); //$NON-NLS-1$
		}
		BufferedImagePageProcessor pageProcessor = new BufferedImagePageProcessor(BufferedImage.TYPE_INT_RGB, 2);
		builder.toPageProcessor(pageProcessor);
		builder.runPaged();
		return pageProcessor.getPageImages();
	}

	private static BufferedImage trim(BufferedImage img) {
		int width = img.getWidth();
		int height = getTrimmedHeight(img);
		if (height <= 0) {
			return null;
		}
		BufferedImage newImg = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
		Graphics g = newImg.createGraphics();
		g.drawImage(img, 0, 0, null);
		return newImg;
	}

	private static int getTrimmedHeight(BufferedImage img) {
		int width = img.getWidth();
		int height = img.getHeight();
		int trimmedHeight = 0;

		for (int i = 0; i < width; i++) {
			for (int j = height - 1; j >= 0; j--) {
				if (img.getRGB(i, j) != Color.WHITE.getRGB() && j > trimmedHeight) {
					trimmedHeight = j;
					break;
				}
			}
		}

		return trimmedHeight;
	}

	public static BufferedImage fillBlankSpace(int actualWidth, int blankSpace) {
		BufferedImage blankSpaceImg = new BufferedImage(actualWidth, blankSpace, BufferedImage.TYPE_INT_ARGB);
		Graphics2D g2 = blankSpaceImg.createGraphics();
		g2.setColor(Color.white);
		g2.dispose();
		return blankSpaceImg;
	}

	//	private void saveHtmlFile(String htmlContent) {
	//		File file = new File("C:/Users/shahin/Desktop/Report/report_org.html");
	//		if (!file.exists()) {
	//			try {
	//				file.createNewFile();
	//			} catch (IOException e) {
	//				e.printStackTrace();
	//			}
	//		}
	//		FileWriter fw;
	//		try {
	//			fw = new FileWriter(file);
	//			fw.write(htmlContent);
	//			fw.close();
	//		} catch (Exception e) {
	//			e.printStackTrace();
	//		}
	//	}
}
