package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.BloodDonationHistory;
import com.floreantpos.model.BloodDonationHistory.PaymentStatus;
import com.floreantpos.model.BloodGroupType;
import com.floreantpos.model.Donor;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.UnitType;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginatedListModel;

public class BloodDonationHistoryDAO extends BaseBloodDonationHistoryDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		BloodDonationHistory bean = (BloodDonationHistory) obj;
		if (bean == null) {
			throw new PosException("Blood donation history not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<BloodDonationHistory> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BloodDonationHistory.class);
			addDeletedFilter(criteria);

			criteria.addOrder(Order.asc(BloodDonationHistory.PROP_DONATION_DATE));
			return criteria.list();
		}
	}

	public void loadHistory(Donor donor, BloodGroupType groupType, PaginatedListModel<BloodDonationHistory> dataModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BloodDonationHistory.class);
			addDeletedFilter(criteria);

			if (donor != null) {
				criteria.add(Restrictions.eq(BloodDonationHistory.PROP_DONOR, donor));
			}

			if (groupType != null) {
				criteria.add(Restrictions.eq(BloodDonationHistory.PROP_BLOOD_GROUP, groupType.name()));
			}

			dataModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());

			criteria.addOrder(Order.desc(BloodDonationHistory.PROP_CREATE_DATE));

			dataModel.setData(criteria.list());
		}
	}

	public void cancelDonation(BloodDonationHistory donationHistory) {
		if (donationHistory == null) {
			return;
		}

		try (Session session = createNewSession()) {
			Transaction txn = session.beginTransaction();
			donationHistory.setStatus(BloodDonationHistory.Status.CANCEL.name());

			String ticketId = donationHistory.getTicketId();

			if (StringUtils.isNotBlank(ticketId)) {
				Ticket ticket = TicketDAO.getInstance().get(new Ticket(ticketId, donationHistory.getOutletId()), session);
				if (ticket != null) {
					TicketDAO.getInstance().loadFullTicket(ticket);
					ticket.setVoided(true);
					User currentUser = DataProvider.get().getCurrentUser();
					ticket.setVoidedBy(currentUser);
					ticket.setClosed(true);
					ticket.setClosingDate(StoreDAO.getServerTimestamp());

					for (TicketItem ticketItem : ticket.getTicketItems()) {
						ticketItem.setVoided(true);
						ticketItem.setVoidDate(StoreDAO.getServerTimestamp());
					}

					for (PosTransaction posTransaction : ticket.getTransactions()) {
						posTransaction.setVoided(true);
						posTransaction.setVoidedByUser(currentUser);
						posTransaction.setVoidDate(StoreDAO.getServerTimestamp());
					}
				}

				TicketDAO.getInstance().saveOrUpdate(ticket, session);
			}

			saveOrUpdate(donationHistory);
			txn.commit();
		}
	}

	public void saveOrUpdateDonation(BloodDonationHistory donationHistory, boolean isNewDonation) {
		if (donationHistory == null) {
			return;
		}
		try (Session session = createNewSession()) {
			Transaction txn = session.beginTransaction();

			if (StringUtils.isBlank(donationHistory.getOutletId())) {
				donationHistory.setOutletId(DataProvider.get().getCurrentOutletId());
			}

			if (isNewDonation) {
				if (!donationHistory.isFreeDonation()) {
					createNewTicket(donationHistory, session);
				}
				save(donationHistory, session);
			}
			else {
				String ticketId = donationHistory.getTicketId();
				if (ticketId == null && !donationHistory.isFreeDonation()) {
					createNewTicket(donationHistory, session);
				}
				else {
					Ticket ticket = TicketDAO.getInstance().get(new Ticket(ticketId, donationHistory.getOutletId()), session);
					if (ticket != null) {
						TicketDAO.getInstance().loadFullTicket(ticket);
						if (ticket.getPaidAmount() == 0 && donationHistory.isFreeDonation()) {
							ticket.setDeleted(true);
						}
						else if (ticket.getPaidAmount() == 0 && !donationHistory.isFreeDonation()) {
							ticket.setDeleted(false);
						}
						if (donationHistory.getPaymentStatus() == PaymentStatus.UNPAID) {
							for (TicketItem ticketItem : ticket.getTicketItems()) {
								if (ProductType.BLOOD.name().equals(ticketItem.getProductType())) {
									ticketItem.setUnitPrice(donationHistory.getBillAmount());
									break;
								}
							}
							ticket.calculatePrice();

							Donor donor = donationHistory.getDonor();
							ticket.setBloodDonorId(donor.getId());
							ticket.putDonorName(donor.getName());
						}
						TicketDAO.getInstance().saveOrUpdate(ticket, session);
					}
				}
				saveOrUpdate(donationHistory, session);
			}
			txn.commit();
		}
	}

	private void createNewTicket(BloodDonationHistory donationHistory, Session session) {
		Ticket ticket = new Ticket();
		ticket.setOutletId(donationHistory.getOutletId());
		ticket.setTicketType(TicketType.BLOOD_BANK);
		ticket.setOwner(DataProvider.get().getCurrentUser());
		ticket.setOrderType(DataProvider.get().getOrderType());

		TicketItem ticketItem = new TicketItem();
		ticketItem.setName("Blood donor bill");
		ticketItem.setQuantity(1d);
		ticketItem.setUnitPrice(donationHistory.getBillAmount());
		ticketItem.setUnitName("each");
		ticketItem.setUnitType(UnitType.UNIT.name());
		ticketItem.setProductType(ProductType.BLOOD.name());

		ticketItem.setTicket(ticket);
		ticket.addToticketItems(ticketItem);
		ticket.calculatePrice();

		Donor donor = donationHistory.getDonor();
		ticket.setBloodDonorId(donor.getId());
		ticket.putDonorName(donor.getName());

		TicketDAO.getInstance().saveOrUpdate(ticket, session);

		donationHistory.setTicketId(ticket.getId());
	}

}