/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class InventoryVendorDAO extends BaseInventoryVendorDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InventoryVendorDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof InventoryVendor) {
			InventoryVendor inventoryVendor = (InventoryVendor) obj;
			inventoryVendor.setDeleted(Boolean.TRUE);
			super.update(inventoryVendor, s);
			InventoryVendorItemsDAO.getInstance().nullifyByInventoryVendor(inventoryVendor, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<InventoryVendor> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.addOrder(getDefaultOrder());
			return criteria.list();
		}
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveAll(List<InventoryVendor> vendorList) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (InventoryVendor vendor : vendorList) {
				session.saveOrUpdate(vendor);
			}

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateInventoryVendors(List<InventoryVendor> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateInventoryVendors(dataList, updateLastUpdateTime, updateSyncTime, false, false);
	}

	public void saveOrUpdateInventoryVendors(List<InventoryVendor> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean saveNewDataOnly,
			boolean forceUpdate) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<InventoryVendor> iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryVendor item = (InventoryVendor) iterator.next();
				InventoryVendor existingItem = get(item.getId());
				if (existingItem != null) {
					if (!forceUpdate && (saveNewDataOnly || !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime()))) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	@SuppressWarnings("unchecked")
	public List<InventoryVendor> findByName(String name) {
		if (StringUtils.isEmpty(name)) {
			return null;
		}
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(InventoryVendor.PROP_NAME, name).ignoreCase());
			return criteria.list();
		}
	}

	public void findByNameOrEmail(PaginationSupport tableModel, String searchString) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Disjunction disjunction = Restrictions.disjunction();

			if (StringUtils.isNotEmpty(searchString)) {
				disjunction.add(Restrictions.or(Restrictions.ilike(InventoryVendor.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(InventoryVendor.PROP_EMAIL, searchString, MatchMode.ANYWHERE)));

			}
			addDeletedFilter(criteria);
			criteria.add(disjunction);

			criteria.setProjection(Projections.rowCount());
			Number uniqueResult = (Number) criteria.uniqueResult();
			tableModel.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}

	}
}