package com.floreantpos.util;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.Date;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.id.IdentifierGenerator;

import com.floreantpos.PosLog;
import com.floreantpos.model.OutdoorTicket;
import com.floreantpos.model.SequenceNumber;
import com.floreantpos.model.dao.OutdoorTicketDAO;
import com.floreantpos.model.dao.SequenceNumberDAO;
import com.floreantpos.model.util.DataProvider;

public class OutdoorTicketIdGenerator implements IdentifierGenerator {
	@Override
	public Serializable generate(SessionImplementor session, Object object) throws HibernateException {
		Class<? extends Object> clazz = object.getClass();
		Serializable generatedId = null;
		try {
			Method method = clazz.getMethod("getId", (Class<?>[]) null); //$NON-NLS-1$
			if (method != null) {
				Object id = method.invoke(object, (Object[]) null);
				if (id != null) {
					generatedId = (Serializable) id;
				}
			}

			if (generatedId == null) {
				generatedId = generateOutdoorTicketId((Session) session);
			}
		} catch (Exception e) {
			PosLog.error(OutdoorTicketIdGenerator.class, e);
		}

		return generatedId;
	}

	private static String generateOutdoorTicketId(Session session) {
		return generateOutdoorTicketId(session, new Date());
	}

	public static String generateOutdoorTicketId(OutdoorTicket outdoorTicket) {
		try (Session session = OutdoorTicketDAO.getInstance().createNewSession()) {
			String outletId = outdoorTicket.getOutletId();
			if (StringUtils.isBlank(outletId)) {
				outletId = DataProvider.get().getOutletId();
			}
			return generateOutdoorTicketId(session, outdoorTicket.getCreateDate());
		}
	}

	public static String generateOutdoorTicketId(Session session, Date createDate) {

		if (createDate == null) {
			createDate = new Date();
		}

		String prefix = SequenceNumber.yearMonthDayFormat.format(createDate);
		String ticketId = prefix + SequenceNumber.threeDigitDecimalFormat
				.format(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.OUTDOOR_TICKET_SEQUENCE_NUMBER, prefix));
		while (OutdoorTicketDAO.getInstance().get(ticketId, session) != null) {
			ticketId = prefix + SequenceNumber.threeDigitDecimalFormat
					.format(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.OUTDOOR_TICKET_SEQUENCE_NUMBER, prefix));
		}

		return ticketId;
	}
}
