package com.floreantpos.model;

import java.util.Set;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.model.base.BaseBloodDonationHistory;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.GsonUtil;
import com.floreantpos.util.NumberUtil;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

public class BloodDonationHistory extends BaseBloodDonationHistory implements PropertyContainer2, TimedModel {
	private static final long serialVersionUID = 1L;

	private transient JsonObject propertiesContainer;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private PurchaseOrder purchaseOrder;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public BloodDonationHistory() {
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for primary key
	 */
	public BloodDonationHistory(java.lang.String id) {
		super(id);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/*[CONSTRUCTOR MARKER END]*/

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	public String getQuantityDisplay() {
		Integer qty = getQuantity();
		return qty + " Bag";
	}

	public String getVolumnDisplay() {
		String volumeDisplay = StringUtils.EMPTY;
		Double volume = getVolume();
		if (volume > 0) {
			volumeDisplay = NumberUtil.formatNumberIfNeeded(volume);
		}
		String unit = getUnit();
		if (StringUtils.isNotBlank(unit)) {
			volumeDisplay += " " + unit;
		}

		return volumeDisplay;
	}

	public String getDonorName() {
		String donorName = getProperty("donor.name", StringUtils.EMPTY);
		if (StringUtils.isBlank(donorName)) {
			InventoryVendor donor = getDonor();
			return donor == null ? StringUtils.EMPTY : donor.getName();
		}
		return donorName;
	}

	public void putDonorName(String name) {
		addProperty("donor.name", name);
	}

	public String getDonorPhone() {
		String donorPhone = getProperty("donor.phone", StringUtils.EMPTY);
		if (StringUtils.isBlank(donorPhone)) {
			InventoryVendor donor = getDonor();
			return donor == null ? StringUtils.EMPTY : donor.getContactNumber();
		}
		return donorPhone;
	}

	public void putDonorPhone(String phone) {
		addProperty("donor.phone", phone);
	}

	public String getWeight() {
		return getProperty("donor.weight", StringUtils.EMPTY);
	}

	public void putWeight(String weight) {
		addProperty("donor.weight", weight);
	}

	public String getPulse() {
		return getProperty("donor.pulse", StringUtils.EMPTY);
	}

	public void putPulse(String pulse) {
		addProperty("donor.pulse", pulse);
	}

	public String getHb() {
		return getProperty("donor.hb", StringUtils.EMPTY);
	}

	public void putHb(String hb) {
		addProperty("donor.hb", hb);
	}

	public String getBp() {
		return getProperty("donor.bp", StringUtils.EMPTY);
	}

	public void putBp(String bp) {
		addProperty("donor.bp", bp);
	}

	public String getTemperature() {
		return getProperty("donor.temperature", StringUtils.EMPTY);
	}

	public void putTemperature(String temperature) {
		addProperty("donor.temperature", temperature);
	}

	public String getLastDonationPeriod() {
		return getProperty("lastDonationPeriod");
	}

	public void putLastDonationPeriod(String lastDonationPeriod) {
		addProperty("lastDonationPeriod", lastDonationPeriod);
	}

	public Boolean isDonatedPreviously() {
		return getBooleanProperty("donatedPreviously", null);
	}

	public void putDonatedPreviously(Boolean donatedPreviously) {
		addProperty("donatedPreviously", donatedPreviously == null ? null : donatedPreviously.toString());
	}

	public Set<String> getRecentActivities() {
		String value = getProperty("recentActivities", "[]");
		return GsonUtil.convertJSONToSet(value, String.class);
	}

	public void putRecentActivities(Set<String> recentActivities) {
		addProperty("recentActivities", GsonUtil.createGson().toJson(recentActivities));
	}

	public Set<String> getDiseases() {
		String value = getProperty("diseases", "[]");
		return GsonUtil.convertJSONToSet(value, String.class);
	}

	public void putDiseases(Set<String> diseases) {
		addProperty("diseases", GsonUtil.createGson().toJson(diseases));
	}

	public Set<String> getMedications() {
		String value = getProperty("medications", "[]");
		return GsonUtil.convertJSONToSet(value, String.class);
	}

	public void putMedications(Set<String> medications) {
		addProperty("medications", GsonUtil.createGson().toJson(medications));
	}

	public Set<String> getSurgeryHistory() {
		String value = getProperty("surgeryHistory", "[]");
		return GsonUtil.convertJSONToSet(value, String.class);
	}

	public void putSurgeryHistory(Set<String> surgeryHistory) {
		addProperty("surgeryHistory", GsonUtil.createGson().toJson(surgeryHistory));
	}

	public boolean isFreeDonation() {
		return getBooleanProperty("free.donation", false);
	}

	public void putFreeDonation(boolean b) {
		addProperty("free.donation", String.valueOf(b));
	}

	public double getBillAmount() {
		return getDoubleProperty("bill.amount");
	}

	public void putBillAmount(double amount) {
		addProperty("bill.amount", String.valueOf(amount));
	}

	public InventoryVendor getDonor() {
		String vendorId = getVendorId();
		if (StringUtils.isBlank(vendorId)) {
			return null;
		}
		return (InventoryVendor) DataProvider.get().getObjectOf(InventoryVendor.class, vendorId);
	}

	public PurchaseOrder getPurchaseOrder() {
		if (purchaseOrder != null) {
			return purchaseOrder;
		}

		String purchaseOrderId = getPurchaseOrderId();
		if (StringUtils.isNotBlank(purchaseOrderId)) {
			this.purchaseOrder = (PurchaseOrder) DataProvider.get().getObjectOf(PurchaseOrder.class, purchaseOrderId);
		}

		return purchaseOrder;
	}

	public String getPaymentStatusDisplay() {
		PaymentStatus paymentStatus = getPaymentStatus();
		if (paymentStatus == null) {
			return StringUtils.EMPTY;
		}
		return paymentStatus.getNameDisplay();
	}

	public PaymentStatus getPaymentStatus() {
		if (isFreeDonation()) {
			return PaymentStatus.FREE;
		}

		PurchaseOrder ticket = getPurchaseOrder();
		if (ticket != null) {
			if (ticket.isFullPaid()) {
				return PaymentStatus.PAID;
			}
			else if (ticket.getPaidAmount() != 0.0) {
				return PaymentStatus.PARTIALLY_PAID;
			}
			else {
				return PaymentStatus.UNPAID;
			}
		}
		return null;
	}

	public static enum PaymentStatus {
		FREE("Free"), PAID("Paid"), PARTIALLY_PAID("Partially Paid"), UNPAID("Unpaid");

		private final String displayName;

		PaymentStatus(String displayName) {
			this.displayName = displayName;
		}

		public String getNameDisplay() {
			return displayName;
		}

		public static PaymentStatus fromString(String name) {
			if (StringUtils.isBlank(name)) {
				return null;
			}

			PaymentStatus[] values = values();
			for (PaymentStatus status : values) {
				if (name.equals(status.name())) {
					return status;
				}
			}

			return null;
		}
	}

	public static enum Status {
		CONSENT_COLLECTED("Consent Collected"), COMPLETE("Complete"), CANCEL("Cancel");

		private final String displayName;

		Status(String displayName) {
			this.displayName = displayName;
		}

		public String getNameDisplay() {
			return displayName;
		}

		public static Status fromString(String name) {
			if (StringUtils.isBlank(name)) {
				return null;
			}

			Status[] values = values();
			for (Status status : values) {
				if (name.equals(status.name())) {
					return status;
				}
			}

			return null;
		}
	}
}