/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PayoutReason;
import com.floreantpos.model.PayoutRecepient;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.swing.PaginationSupport;

public class PayOutTransactionDAO extends BasePayOutTransactionDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PayOutTransactionDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void createPayoutTransaction(PayoutReason reason, PayoutRecepient recepient, String note, double payoutAmount, User currentUser,
			CashDrawer cashDrawer) throws Exception {
		savePayoutTransaction(reason, recepient, note, payoutAmount, currentUser, "", null, false);
	}

	public void savePayoutTransaction(PayoutReason reason, PayoutRecepient recepient, String note, double payoutAmount, User currentUser, 
			String batchNo, Date transactionTime, boolean isExpenses) throws Exception {

		CashDrawer cashDrawer = UserDAO.getInstance().getCashDrawerWithCashBalance(currentUser);
		if (cashDrawer == null) {
			throw new PosException("Cash register is closed.");
		}
		if (cashDrawer.getDrawerAccountable() < payoutAmount) {
			throw new PosException("Insufficient cash register balance.");
		}
		Date serverTimestamp = StoreDAO.getServerTimestamp();

		PayOutTransaction payOutTransaction = new PayOutTransaction();
		payOutTransaction.setPaymentType(PaymentType.CASH);
		payOutTransaction.setTransactionType(TransactionType.OUT.name());
		payOutTransaction.setTransactionTime(serverTimestamp);
		payOutTransaction.setReason(reason);
		payOutTransaction.setRecepient(recepient);
		payOutTransaction.setNote(note);
		payOutTransaction.setAccountProcessed(true);
		if (StringUtils.isNotBlank(batchNo)) {
			payOutTransaction.setBatchNo(batchNo);
		}
		payOutTransaction.setAmount(Double.valueOf(payoutAmount));

		payOutTransaction.setCashDrawer(cashDrawer);
		payOutTransaction.setUser(currentUser);
		payOutTransaction.setServer(currentUser);
		payOutTransaction.setOutletId(currentUser.getOutletId());
		if (transactionTime != null) {
			payOutTransaction.setEventTime(transactionTime);
		}
		else {
			payOutTransaction.setEventTime(serverTimestamp);
		}

		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();

			save(payOutTransaction, session);

			LedgerEntryDAO.getInstance().savePayoutLedgerEntry(payOutTransaction, session);

			transaction.commit();
		}
	}

	public void revertTransaction(PayOutTransaction payOutTransaction) {
		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();
			payOutTransaction.setVoided(true);
			payOutTransaction.setVoidDate(StoreDAO.getServerTimestamp());
			
			update(payOutTransaction, session);
			
			LedgerEntryDAO.getInstance().savePayoutLedgerEntry(payOutTransaction, true, session);
			
			transaction.commit();
		}
	}

	public List<PayOutTransaction> getUnsettled(Terminal terminal, User cashier) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(PayOutTransaction.PROP_DRAWER_RESETTED, Boolean.FALSE));
			if (cashier == null) {
				criteria.add(Restrictions.eq(PayOutTransaction.PROP_TERMINAL_ID, terminal == null ? null : terminal.getId()));
				//criteria.add(Restrictions.ne("paymentReceiver", DrawerType.STAFF_BANK));
			}
			else {
				criteria.add(Restrictions.eq(PayOutTransaction.PROP_USER_ID, cashier.getId()));
				//criteria.add(Restrictions.eq("paymentReceiver", DrawerType.STAFF_BANK));
			}

			List list = criteria.list();
			return list;
		}
	}

	public List<PayOutTransaction> findPayoutReport(Date fromDate, Date toDate, String batchNo) {
		return findPayoutReport(fromDate, toDate, batchNo, null, null);
	}

	public List<PayOutTransaction> findPayoutReport(Date fromDate, Date toDate, String batchNo, String recipientId, String reasonId) {
		return findPayoutReport(fromDate, toDate, batchNo, recipientId, reasonId, null);
	}

	public List<PayOutTransaction> findPayoutReport(Date fromDate, Date toDate, String batchNo, String recipientId, String reasonId, String userId) {
		Pagination pagination = new Pagination<>(0, -1);
		findPayoutReport(fromDate, toDate, batchNo, recipientId, reasonId, pagination, userId);
		return pagination.getDataList();
	}

	public void findPayoutReport(Date fromDate, Date toDate, String batchNo, String recipientId, String reasonId, PaginationSupport listModel, String userId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(PayOutTransaction.PROP_DRAWER_RESETTED, Boolean.FALSE));
			criteria.add(Restrictions.eq(PayOutTransaction.PROP_VOIDED, Boolean.FALSE));

			if (fromDate != null && toDate != null) {
				criteria.add(Restrictions.between(PayOutTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
			}
			if (StringUtils.isNotBlank(batchNo)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_BATCH_NO, batchNo));
			}
			if (StringUtils.isNotBlank(recipientId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_RECEPIENT_ID, recipientId));
			}
			if (StringUtils.isNotBlank(reasonId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_REASON_ID, reasonId));
			}

			if (StringUtils.isNotBlank(userId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_USER_ID, userId));
			}

			//			if (isShowExpenseOnly) {
			//				criteria.add(Restrictions.eq(PayOutTransaction.PROP_EXPENSE, true));
			//			}
			//			else {
			//				criteria.add(Restrictions.ne(PayOutTransaction.PROP_EXPENSE, true));
			//			}

			listModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			criteria.addOrder(Order.desc(PayOutTransaction.PROP_TRANSACTION_TIME));
			listModel.setRows(criteria.list());
		}
	}
}