package com.orocube.cloudpos.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Vector;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Session;
import org.hibernate.Transaction;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceSubType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.BankAccount;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.DonorPaymentTransaction;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PayoutReason;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseRefundTransaction;
import com.floreantpos.model.PurchaseTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.ActionHistoryDAO;
import com.floreantpos.model.dao.BalanceUpdateTransactionDAO;
import com.floreantpos.model.dao.LedgerEntryDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TerminalDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.payment.CloudPaymentListener;
import com.floreantpos.payment.common.Payable;
import com.floreantpos.report.ReceiptPrintService.PaymentByCurrency;
import com.floreantpos.services.PostPaymentProcessor;
import com.floreantpos.ui.RefreshableView;
import com.floreantpos.util.DefaultDataInserter;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class SettleProcessor {
	private Vector<CloudPaymentListener> paymentListeners = new Vector<CloudPaymentListener>(3);

	private List<Payable> payableList;
	private double tenderAmount;
	private User currentUser;
	private RefreshableView refreshableView;
	private final TransactionType transactionType;
	private User accountsManager;

	public SettleProcessor(User currentUser, RefreshableView refreshableView, TransactionType transactionType) {
		super();
		this.currentUser = currentUser;
		this.refreshableView = refreshableView;
		this.transactionType = transactionType;
	}

	public List<Payable> getPayableList() {
		return payableList;
	}

	public void setPayableList(List<Payable> payableObjectList) {
		this.payableList = payableObjectList;
	}

	public double getTenderAmount() {
		return tenderAmount;
	}

	public void setTenderAmount(double tenderAmount) {
		this.tenderAmount = tenderAmount;
	}

	public void addPaymentListener(CloudPaymentListener paymentListener) {
		this.paymentListeners.add(paymentListener);
	}

	public void removePaymentListener(CloudPaymentListener paymentListener) {
		this.paymentListeners.remove(paymentListener);
	}

	public void doSettle(PaymentType paymentType, double tenderAmount) throws Exception {
		this.doSettle(paymentType, tenderAmount, null, ""); //$NON-NLS-1$
	}

	public void doSettle(PaymentType paymentType, double tenderAmount, CustomPayment customPayment, String refNo) throws Exception {
		this.tenderAmount = tenderAmount;
		if (payableList == null) {
			return;
		}

		switch (paymentType) {
			case CASH:
				doPayByCash(paymentType);
				break;

			case CUSTOM_PAYMENT:

				List<PosTransaction> transactions = new ArrayList<>();
				for (Payable payable : payableList) {
					PosTransaction transaction = null;
					if (payable instanceof Ticket) {
						Ticket ticket = (Ticket) payable;
						if (ticket.getTicketType() == TicketType.BLOOD_BANK) {
							transaction = new DonorPaymentTransaction();
							transaction.setTransactionTime(StoreDAO.getServerTimestamp());
						}
						else {
							transaction = payable.createTransaction(paymentType);
							transaction.setCashDrawer(currentUser.getCurrentCashDrawer());
						}
					}
					else {
						transaction = payable.createTransaction();
					}
					transaction.setProject(payable.getProject());
					transaction.setCustomPaymentFieldName(customPayment.getRefNumberFieldName());
					transaction.setCustomPaymentId(customPayment.getId());
					transaction.setCustomPaymentName(customPayment.getName());
					transaction.setCustomPaymentRef(refNo);
					transaction.setConfirmPayment(customPayment.isConfirmPaymentImmediately());

					BankAccount linkedBankAccount = customPayment.getLinkedBankAccount();
					if (customPayment.isConfirmPaymentImmediately()) {
						transaction.setLinkedBankAccount(linkedBankAccount);
					}
					else if (linkedBankAccount != null) {
						transaction.putLinkedBankAccountId(linkedBankAccount.getId());
					}

					transactions.add(transaction);
				}

				doPayByCustomPayment(transactions);
				break;

			case CREDIT_CARD:
			case DEBIT_CARD:
				doPayByCard(tenderAmount);
				break;

			default:
				break;
		}
	}

	private void doPayByCard(final double tenderAmount) {
		throw new PosException("Under construction");
	}

	private void doPayByCash(PaymentType paymentType) throws Exception {
		List transactions = new ArrayList<>();
		Double totalDueAmount = getTotalDueAmount();
		Double dueAmount = 0.0;
		for (Payable payable : payableList) {
			if (NumberUtil.isZero(payable.getDueValue())) {
				continue;
			}
			PosTransaction posTransaction = null;
			if (payable instanceof Ticket) {
				Ticket ticket = (Ticket) payable;
				if (ticket.getTicketType() == TicketType.BLOOD_BANK) {
					posTransaction = new DonorPaymentTransaction();
					posTransaction.setTransactionTime(StoreDAO.getServerTimestamp());
				}
				else {
					posTransaction = payable.createTransaction(paymentType);
					posTransaction.setCashDrawer(currentUser.getCurrentCashDrawer());
				}
			}
			else {
				posTransaction = payable.createTransaction();
			}
			posTransaction.setProject(payable.getProject());
			transactions.add(posTransaction);

			posTransaction.setPaymentType(paymentType);

			setTransactionAmounts(posTransaction);
			dueAmount += payable.getDueValue();
			if (tenderAmount <= dueAmount) {
				break;
			}
		}
		doSettle(transactions, null, true, true);
	}

	public void settle(PosTransaction transaction, PostPaymentProcessor postPaymentProcessor) throws Exception {
		settle(Arrays.asList(transaction), postPaymentProcessor);
	}

	public void settle(List<PosTransaction> transactions, PostPaymentProcessor postPaymentProcessor) throws Exception {
		try {

			doSettle(transactions, postPaymentProcessor, true, true);

		} catch (Exception e) {
			throw e;
		}
	}

	public void doSettle(List<PosTransaction> transactions, PostPaymentProcessor postPaymentProcessor, boolean printReceipt, boolean peformAfterSettle)
			throws Exception {
		doSettle(transactions, postPaymentProcessor, printReceipt, peformAfterSettle, null);
	}

	public void doSettle(List<PosTransaction> transactions, PostPaymentProcessor postPaymentProcessor, boolean printReceipt, boolean peformAfterSettle,
			List<PaymentByCurrency> paymentByCurrencyList) throws Exception {

		double totalDueAmount = getTotalDueAmount();

		List<PosTransaction> completeTransactions = new ArrayList<>();
		List<BalanceUpdateTransaction> accountsManagerBalanceTrans = new ArrayList<>();

		Transaction tx = null;
		try (Session session = TerminalDAO.getInstance().createNewSession()) {
			tx = session.beginTransaction();
			for (int i = 0; i < transactions.size(); i++) {
				PosTransaction transaction = transactions.get(i);

				Payable payable = payableList.get(i);

				double dueAmount = payable.getDueValue();
				if (NumberUtil.isZero(dueAmount)) {
					continue;
				}

				if (!NumberUtil.isZero(tenderAmount)) {
					transaction.setEntityId(payable.getEntityId());
					String purchaePaidNote = "Paid to vendor";
					if (payable instanceof Ticket) {
						Ticket ticket = (Ticket) payable;
						transaction.setTicket(ticket);
						transaction.setTicketId(payable.getEntityId());
						transaction.setServer(ticket.getOwner());
					}
					else if (payable instanceof PurchaseOrder) {
						PurchaseOrder purchaseOrder = (PurchaseOrder) payable;
						transaction.setServerId(purchaseOrder.getCreatedByUserId());
						transaction.setEventTime(DataProvider.get().getServerTimestamp());
						transaction.setNote(purchaePaidNote);
						InventoryVendor vendor = purchaseOrder.getVendor();
						if (vendor != null) {
							transaction.setVendor(vendor);
						}
						if (accountsManager != null) {
							transaction.setAccountManager(accountsManager);
						}
					}

					boolean isRefund = transaction instanceof PurchaseRefundTransaction;
					PayoutReason vendorReason = DefaultDataInserter.addOrGetVendorPaymentReason(isRefund);

					String purchaseRefundNote = "Refund from vendor";
					if (isRefund) {
						transaction.setAmount(tenderAmount);
						double d = dueAmount + transaction.getAmount();
						payable.setDueValue(d);
						transaction.setNote(purchaseRefundNote);
						transaction.setEventTime(DataProvider.get().getServerTimestamp());
						if (accountsManager != null) {
							transaction.setAccountManagerId(accountsManager.getId());
						}
					}
					else {
						transaction.setAmount(tenderAmount > dueAmount ? dueAmount : tenderAmount);
						double d = dueAmount - transaction.getAmount();
						payable.setDueValue(d < 0 ? 0 : d);
					}

					if (transaction instanceof DonorPaymentTransaction && payable instanceof Ticket) {
						Ticket ticket = (Ticket) payable;
						vendorReason = DefaultDataInserter.addOrGetBloodBankReason(ticket.getTicketType());
					}

					transaction.setTransactionType(transactionType.name());
					transaction.setTerminal(DataProvider.get().getCurrentTerminal());
					transaction.setOutletId(DataProvider.get().getCurrentOutletId());
					transaction.setCustomerId(payable.getCustomerId());
					transaction.setUser(DataProvider.get().getCurrentUser());
					transaction.setReason(vendorReason);

					PosTransactionDAO.getInstance().save(transaction, session);
					if (postPaymentProcessor != null) {
						postPaymentProcessor.paymentDone(transaction, session);
					}
					completeTransactions.add(transaction);

					if (payable instanceof PurchaseOrder && transaction.getPaymentType() == PaymentType.CASH) {
						BalanceUpdateTransaction balanceUpdateTransaction;
						PurchaseOrder purchaseOrder = (PurchaseOrder) payable;
						InventoryVendor vendor = purchaseOrder.getVendor();
						if (transaction instanceof PurchaseRefundTransaction) {
							balanceUpdateTransaction = BalanceUpdateTransactionDAO.getInstance().populateAccountsManagerTransaction(null, accountsManager,
									transaction.getAmount(), purchaseRefundNote, transaction.getProjectId(), session, BalanceSubType.EXPENSE_REFUND,
									transaction);
						}
						else {
							balanceUpdateTransaction = BalanceUpdateTransactionDAO.getInstance().saveAccountsManagerExpensesTransaction(
									transaction.getTransactionTime(), accountsManager, /*batchNo*/ null, transaction.getProjectId(), /*recepient*/ null,
									vendorReason, /*subReason*/ null, transaction.getAmount(), purchaePaidNote, vendor, session, false, transaction);
						}
						accountsManagerBalanceTrans.add(balanceUpdateTransaction);
					}
					else if (payable instanceof Ticket && (((Ticket) payable).getTicketType() == TicketType.BLOOD_BANK
							|| ((Ticket) payable).getTicketType() == TicketType.BLOOD_BANK_ISSUE) && transaction.getPaymentType() == PaymentType.CASH) {
						BalanceUpdateTransaction balanceUpdateTransaction = BalanceUpdateTransactionDAO.getInstance().saveAccountsManagerExpensesTransaction(
								transaction.getTransactionTime(), accountsManager, /*batchNo*/ null, transaction.getProjectId(), /*recepient*/ null,
								vendorReason, /*subReason*/ null, transaction.getAmount(), purchaePaidNote, null, session,
								(((Ticket) payable).getTicketType() == TicketType.BLOOD_BANK_ISSUE), transaction);
						accountsManagerBalanceTrans.add(balanceUpdateTransaction);
					}

					if (transaction.getLinkedBankAccount() != null) {
						List customPaymentRefJsonList = null;
						String note = "";
						if (transaction instanceof PurchaseTransaction) {
							note = purchaePaidNote;
						}
						else if (transaction instanceof PurchaseRefundTransaction) {
							note = purchaseRefundNote;
						}
						else {
							customPaymentRefJsonList = transaction.getCustomPaymentRefJsonList();
						}

						BalanceUpdateTransactionDAO.getInstance().saveBankAccountTransaction(transaction.getLinkedBankAccount(), transaction.getPaymentType(),
								transaction, transaction.getAmount(), note, session,
								(customPaymentRefJsonList == null || customPaymentRefJsonList.isEmpty() ? null : customPaymentRefJsonList.toString()));
					}

					if (tenderAmount <= dueAmount) {
						break;
					}
					else {
						tenderAmount -= dueAmount;
					}
				}
			}
			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
				PosLog.error(SettleProcessor.class, x);
			}
		}

		if (peformAfterSettle) {
			doAfterSettleTask(completeTransactions, accountsManagerBalanceTrans, totalDueAmount, true);
		}
	}

	public void doPayByCustomPayment(List<PosTransaction> transactions) throws Exception {
		for (PosTransaction transaction : transactions) {
			transaction.setPaymentType(PaymentType.CUSTOM_PAYMENT);
			transaction.setCaptured(true);
			setTransactionAmounts(transaction);
		}

		doSettle(transactions, null, true, true);
	}

	public void setTransactionAmounts(PosTransaction transaction) {
		setTransactionAmounts(transaction, null);
	}

	public void setTransactionAmounts(PosTransaction transaction, List<PaymentByCurrency> paymentByCurrencies) {
		double cashBackAmount = 0;
		if (paymentByCurrencies != null && paymentByCurrencies.size() > 0) {
			for (PaymentByCurrency paymentByCurrency : paymentByCurrencies) {
				cashBackAmount += paymentByCurrency.cashBackAmount / paymentByCurrency.currency.getExchangeRate();
			}
		}
		if (cashBackAmount > tenderAmount) {
			throw new PosException("Cash back amount cannot be greater than total amount.");
		}
		transaction.setTenderAmount(tenderAmount);
		if (transaction instanceof PurchaseRefundTransaction) {
			transaction.setAmount(tenderAmount);
		}
		else {
			double totalDueAmount = getTotalDueAmount();
			if ((tenderAmount - cashBackAmount) >= totalDueAmount) {
				transaction.setAmount(getTotalDueAmount());
			}
			else {
				transaction.setAmount(tenderAmount - cashBackAmount);
			}
		}

		double changeAmount = NumberUtil.round(tenderAmount - transaction.getAmount());
		transaction.setChangeAmount(changeAmount);

	}

	public void doAfterSettleTask(List<PosTransaction> transactions, List<BalanceUpdateTransaction> accountsManagerBalanceTrans, final double dueAmount,
			boolean printTicket) throws Exception {

		if (NumberUtil.isZero(getTotalDueAmount())) {
			doInformListenerPaymentDone();
		}
		else {
			setPayableList(payableList);
			doInformListenerPaymentUpdate();
		}

		createLedgerEntry(transactions, accountsManagerBalanceTrans);
		createJournalLog(dueAmount, transactions);
	}

	private void createLedgerEntry(List<PosTransaction> transactions, List<BalanceUpdateTransaction> accountsManagerBalanceTrans) {
		if (transactions != null) {
			for (PosTransaction posTransaction : transactions) {
				if (payableList.get(0) instanceof PurchaseOrder) {
					Optional<Payable> findFirst = payableList.stream().filter(t -> t.getEntityId().equals(posTransaction.getEntityId())).findFirst();
					if (findFirst.isPresent()) {
						LedgerEntryDAO.getInstance().savePurchaseBillLedgerEntry((PurchaseOrder) findFirst.get(), posTransaction);
					}
				}
			}

			for (BalanceUpdateTransaction balanceUpdateTransaction : accountsManagerBalanceTrans) {
				LedgerEntryDAO.getInstance().saveExpensesLedgerEntry(balanceUpdateTransaction, true);
			}
		}
	}

	private void doInformListenerPaymentDone() {
		for (CloudPaymentListener paymentListener : paymentListeners) {
			paymentListener.paymentDone();
		}
	}

	public void doInformListenerPaymentUpdate() {
		for (CloudPaymentListener paymentListener : paymentListeners) {
			paymentListener.paymentDataChanged();
		}
	}

	public User getCurrentUser() {
		return currentUser != null ? currentUser : DataProvider.get().getCurrentUser();
	}

	public void setCurrentUser(User currentUser) {
		this.currentUser = currentUser;
	}

	public RefreshableView getRefreshableView() {
		return refreshableView;
	}

	public void cancelPayment() {
		for (CloudPaymentListener paymentListener : paymentListeners) {
			paymentListener.paymentCanceled();
		}
	}

	public TransactionType getTransactionType() {
		return transactionType;
	}

	private double getTotalDueAmount() {
		if (payableList == null) {
			return 0;
		}
		return payableList.stream().mapToDouble(value -> value.getDueValue()).sum();
	}

	public User getAccountsManager() {
		return accountsManager;
	}

	public void setAccountsManager(User accountsManager) {
		this.accountsManager = accountsManager;
	}

	private void createJournalLog(double dueAmount, List<PosTransaction> transactions) {
		StringBuilder descriptionBuilder = new StringBuilder();

		descriptionBuilder.append("Due amount: " + NumberUtil.getCurrencyFormat(dueAmount)); //$NON-NLS-1$
		double total = transactions.stream().mapToDouble(t -> t.getAmount()).sum();
		descriptionBuilder.append(", Paid amount: " + NumberUtil.getCurrencyFormat(total)); //$NON-NLS-1$

		Optional<List<?>> payables = Optional.ofNullable(getPayableList());
		payables.ifPresent(payableList -> {
			Payable payable = (Payable) payableList.get(0);
			if (payable instanceof PurchaseOrder) {
				List<PurchaseOrder> purchaseOrders = (List<PurchaseOrder>) payableList;
				List<String> purchaseOrderIds = POSUtil.getStringIds(purchaseOrders, PurchaseOrder.class, "getOrderId"); //$NON-NLS-1$
				descriptionBuilder.append(", Purchase order: " + purchaseOrderIds); //$NON-NLS-1$
			}
		});

		descriptionBuilder.append(", Transaction: " + POSUtil.getStringIds(transactions, PosTransaction.class)); //$NON-NLS-1$
		descriptionBuilder.append('\n');

		String actionName = null;
		if (transactionType == TransactionType.OUT) {
			actionName = "Pay bill"; //$NON-NLS-1$
		}

		if (StringUtils.isNotBlank(actionName)) {
			ActionHistoryDAO.saveHistory(actionName, descriptionBuilder.toString());
		}
	}
}
