/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.PrinterGroup;
import com.floreantpos.model.TerminalPrinters;
import com.floreantpos.model.VirtualPrinter;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class VirtualPrinterDAO extends BaseVirtualPrinterDAO {

	private static final String EMPTY_STRING = ""; //$NON-NLS-1$
	private static final String EMPTY_DOT_STRING = "."; //$NON-NLS-1$
	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$
	private static final String EMPTY_SPACE_STRING = " "; //$NON-NLS-1$

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public VirtualPrinterDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof VirtualPrinter) {
			VirtualPrinter virtualPrinter = (VirtualPrinter) obj;
			List<PrinterGroup> printerGroups = PrinterGroupDAO.getInstance().getPrinterGroupsByVirtualPrinter(virtualPrinter, s);
			if (printerGroups != null && !printerGroups.isEmpty()) {
				String message = virtualPrinter.getName() + EMPTY_SPACE_STRING + Messages.getString("VirtualPrinterDAO.0"); //$NON-NLS-1$
				String details = this.constructExceptionDetailsByPrinterGroups(virtualPrinter, printerGroups);
				throw new PosException(message, details);
			}
			virtualPrinter.setDeleted(Boolean.TRUE);
			super.update(virtualPrinter, s);
			this.deleteTerminalPrinters(virtualPrinter, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	private void deleteTerminalPrinters(VirtualPrinter virtualPrinter, Session session) {
		String hql = "DELETE FROM %s TP WHERE TP.%s = :printer"; //$NON-NLS-1$
		hql = String.format(hql, TerminalPrinters.REF, TerminalPrinters.PROP_VIRTUAL_PRINTER);
		Query query = session.createQuery(hql);
		query.setParameter("printer", virtualPrinter); //$NON-NLS-1$
		query.executeUpdate();
	}

	private String constructExceptionDetailsByPrinterGroups(VirtualPrinter virtualPrinter, List<PrinterGroup> printerGroups) {
		if (printerGroups != null && !printerGroups.isEmpty()) {
			StringBuilder builder = new StringBuilder(virtualPrinter.getName() + EMPTY_SPACE_STRING + Messages.getString("VirtualPrinterDAO.3")); //$NON-NLS-1$
			for (int i = 0; i < printerGroups.size(); i++) {
				String message = (i + 1) + EMPTY_DOT_STRING + EMPTY_SPACE_STRING + printerGroups.get(i).getName();
				builder.append(EMPTY_NEWLINE_STRING).append(message);
			}
			return builder.toString();
		}
		return EMPTY_STRING;
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<VirtualPrinter> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public void saveOrUpdateVirtualPrinter(List<VirtualPrinter> virtualPrinterList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (virtualPrinterList == null) {
			return;
		}

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<VirtualPrinter> iterator = virtualPrinterList.iterator(); iterator.hasNext();) {
				VirtualPrinter item = (VirtualPrinter) iterator.next();
				VirtualPrinter existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	//Deleted filter for this method is strictly prohibited
	public VirtualPrinter findPrinterByName(String name) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(VirtualPrinter.PROP_NAME, name));
			return (VirtualPrinter) criteria.uniqueResult();
		}
	}

	@SuppressWarnings("unchecked")
	public List<VirtualPrinter> findKitchenPrinters() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(VirtualPrinter.PROP_TYPE, VirtualPrinter.KITCHEN));
			return criteria.list();
		}
	}

	public VirtualPrinter findAReceiptPrinter() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(VirtualPrinter.PROP_TYPE, VirtualPrinter.RECEIPT));
			
			List list = criteria.list();
			if (list.size() > 0) {
				return (VirtualPrinter) list.get(0);
			}
			return null;
		}
	}
}