package com.floreantpos.model;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseSurgeryInfo;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.GsonUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.XMLTransientUtil;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;
import com.orocube.rest.service.ServiceUtils;

public class SurgeryInfo extends BaseSurgeryInfo implements PropertyContainer2, TimedModel {
	private static final long serialVersionUID = 1L;

	private transient JsonObject propertiesContainer;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private transient Patient patient;
	private transient Doctor approveByDoctor;
	private transient MenuItem surgeryItem;
	private transient Room otRoom;
	private transient Bed otBed;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public SurgeryInfo() {
	}

	/**
	 * Constructor for primary key
	 */
	public SurgeryInfo(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public void setPatient(Customer patient) {
		removePatient();
		if (patient == null) {
			return;
		}

		setPatientId(patient.getId());
		addProperty("patient.first_name", patient.getFirstName());
		addProperty("patient.last_name", patient.getLastName());
		addProperty("patient.name", patient.getName());
		addProperty("patient.mobile_no", patient.getMobileNo());

		//patientJournalLog(patient, oldCustomerId);
	}

	@XmlTransient
	@JsonIgnore
	public Patient getPatient() {
		if (this.patient != null) {
			return this.patient;
		}
		String patientId = getPatientId();
		if (StringUtils.isEmpty(patientId)) {
			return null;
		}
		patient = (Patient) DataProvider.get().getObjectOf(Patient.class, patientId);
		return patient;
	}

	public void setAnesthesiologist(Doctor anesthesiologist) {
		if (anesthesiologist != null) {
			setAnesthesiologistId(anesthesiologist.getId());
			setAnesthesiologistName(anesthesiologist.getName());
		}
		else {
			setAnesthesiologistId(null);
			setAnesthesiologistName(null);
		}
	}

	@XmlTransient
	@JsonIgnore
	public Doctor getAnesthesiologist() {
		String anesthesiologistId = getAnesthesiologistId();
		if (StringUtils.isBlank(anesthesiologistId)) {
			return null;
		}

		return (Doctor) DataProvider.get().getObjectOf(Doctor.class, anesthesiologistId);
	}

	public void removePatient() {
		setPatientId(null);
		patient = null;
		removeProperty("patient.first_name");
		removeProperty("patient.last_name");
		removeProperty("patient.name");
		removeProperty("patient.mobile_no");
	}

	@XmlTransient
	public String getPatientName() {
		if (this.patient != null) {
			return this.patient.getName();
		}
		return getProperty("patient.name", "");
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	public String getQuantityDisplay() {
		Integer qty = getQuantity();
		return qty + " Bag";
	}

	public String getVolumeDisplay() {
		String volumeDisplay = StringUtils.EMPTY;
		Double volume = getVolume();
		if (volume > 0) {
			volumeDisplay = NumberUtil.formatNumberIfNeeded(volume);
		}
		String unit = getUnit();
		if (StringUtils.isNotBlank(unit)) {
			volumeDisplay += " " + unit;
		}
		else {
			volumeDisplay += " ml";
		}

		return volumeDisplay;
	}

	public void putSurgeryInfoDiagnosisItems(List<SurgeryDiagnosisItem> surgeryDiagnosisItems) {
		String json = GsonUtil.createGson().toJson(surgeryDiagnosisItems);
		setSurgeryDiagnosisItems(json);
	}

	public void addTodiagnosisItems(SurgeryDiagnosisItem surgeryDiagnosisItem) {
		List<SurgeryDiagnosisItem> surgeryInfoDiagnosisItems = getSurgeryInfoDiagnosisItems();
		surgeryInfoDiagnosisItems.add(surgeryDiagnosisItem);
		putSurgeryInfoDiagnosisItems(surgeryInfoDiagnosisItems);
	}

	public List<SurgeryDiagnosisItem> getSurgeryInfoDiagnosisItems() {

		List<SurgeryDiagnosisItem> surgeryDiagnosisItems = new ArrayList<SurgeryDiagnosisItem>();
		try {
			String surgeryDiagnosisItemsString = getSurgeryDiagnosisItems();
			if (StringUtils.isEmpty(surgeryDiagnosisItemsString)) {
				return surgeryDiagnosisItems;
			}

			surgeryDiagnosisItems = GsonUtil.createGson().fromJson(surgeryDiagnosisItemsString, new TypeToken<ArrayList<SurgeryDiagnosisItem>>() {
			}.getType());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return surgeryDiagnosisItems;
	}

	public void putSurgeryInfoMedicineItems(List<SurgeryMedicineItem> surgeryMedicineItems) {
		String json = GsonUtil.createGson().toJson(surgeryMedicineItems);
		setSurgeryMedicineItems(json);
	}

	public void addTomedicineItems(SurgeryMedicineItem surgeryMedicineItem) {
		List<SurgeryMedicineItem> surgeryInfoMedicineItems = getSurgeryInfoMedicineItems();
		surgeryInfoMedicineItems.add(surgeryMedicineItem);
		putSurgeryInfoMedicineItems(surgeryInfoMedicineItems);
	}

	public List<SurgeryMedicineItem> getSurgeryInfoMedicineItems() {
		List<SurgeryMedicineItem> surgeryMedicineItems = new ArrayList<SurgeryMedicineItem>();
		try {
			String surgeryMedicineItemsString = getSurgeryMedicineItems();
			if (StringUtils.isEmpty(surgeryMedicineItemsString)) {
				return surgeryMedicineItems;
			}

			surgeryMedicineItems = GsonUtil.createGson().fromJson(surgeryMedicineItemsString, new TypeToken<ArrayList<SurgeryMedicineItem>>() {
			}.getType());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return surgeryMedicineItems;
	}

	public void putSurgeryInfoEquipmentItems(List<SurgeryEquipmentItem> equipmentItems) {
		String json = GsonUtil.createGson().toJson(equipmentItems);
		addProperty("equipmentItems", json);
	}

	public void addToequipmentItems(SurgeryEquipmentItem surgeryEquipmentItem) {
		List<SurgeryEquipmentItem> equipmentItems = getEquipmentItems();
		equipmentItems.add(surgeryEquipmentItem);
		putSurgeryInfoEquipmentItems(equipmentItems);
	}

	public List<SurgeryEquipmentItem> getEquipmentItems() {
		return GsonUtil.convertJSONToList(getProperty("equipmentItems", "[]"), SurgeryEquipmentItem.class);
	}

	public void setSurgeryItem(MenuItem surgeryItem) {
		this.surgeryItem = surgeryItem;
		if (surgeryItem != null) {
			setSurgeryItemId(surgeryItem.getId());
			putSurgeryName(surgeryItem.getDisplayName());
		}
		else {
			setSurgeryItemId(null);
			putSurgeryName(null);
		}
	}

	private void putSurgeryName(String displayName) {
		addProperty("surgery_name", displayName);
	}

	@XmlTransient
	@JsonIgnore
	public MenuItem getSurgeryItem() {
		String surgeryItemId = getSurgeryItemId();
		if (StringUtils.isBlank(surgeryItemId)) {
			return null;
		}

		if (surgeryItem != null && surgeryItemId.equals(surgeryItem.getId())) {
			return surgeryItem;
		}

		return surgeryItem = (MenuItem) DataProvider.get().getObjectOf(MenuItem.class, surgeryItemId);
	}

	@XmlTransient
	public String getSurgeryName() {
		if (this.surgeryItem != null) {
			return this.surgeryItem.getDisplayName();
		}
		return getProperty("surgery_name", "");
	}

	public String getRequestedDateDisplay() {
		Date requestedDate = getRequestedDate();
		if (requestedDate == null) {
			return StringUtils.EMPTY;
		}

		return DateUtil.formatReportShortDateWithBrowserTimeOffset(requestedDate);
	}

	public String getRequestedTimeDisplay() {
		String requestedTime = getRequestedTime();
		if (StringUtils.isNotBlank(requestedTime)) {
			MedicationShift requestedTimeShift = MedicationShift.valueOf(requestedTime);
			if (requestedTimeShift != null) {
				return requestedTimeShift.toString();
			}
		}
		return StringUtils.EMPTY;
	}

	public String getApprovedDateDisplay() {
		Date approvedDate = getApprovedDate();
		if (approvedDate == null) {
			return StringUtils.EMPTY;
		}

		return DateUtil.formatReportShortDateWithBrowserTimeOffset(approvedDate);
	}

	public String getApprovedTimeDisplay() {
		String approvedTime = getApprovedTime();
		if (StringUtils.isNotBlank(approvedTime)) {
			MedicationShift requestedTimeShift = MedicationShift.valueOf(approvedTime);
			if (requestedTimeShift != null) {
				return requestedTimeShift.toString();
			}
		}
		return StringUtils.EMPTY;
	}

	public String getUrgencyLevelDisplay() {
		String urgencyLevel = getUrgencyLevel();
		if (StringUtils.isNotBlank(urgencyLevel)) {
			UrgencyLevel urgencyLevelValue = UrgencyLevel.valueOf(urgencyLevel);
			if (urgencyLevelValue != null) {
				return urgencyLevelValue.toString();
			}
		}

		return StringUtils.EMPTY;
	}

	public String getSurgeryStatusDisplay() {
		return OTStatus.fromNameString(getSurgeryStatus()).getDisplayString();
	}

	public String getSurgeryProcedureStatusDisplay() {
		String surgeryProcedureStatus = getSurgeryProcedureStatus();
		if (StringUtils.isNotBlank(surgeryProcedureStatus)) {
			return OTProcedureStatus.fromNameString(surgeryProcedureStatus).getDisplayString();
		}
		return StringUtils.EMPTY;
	}

	public Doctor getApprovedByDoctor() {
		String approvedByDoctorId = getApprovedById();
		if (StringUtils.isBlank(approvedByDoctorId)) {
			return null;
		}

		if (approveByDoctor != null && approvedByDoctorId.equals(approveByDoctor.getId())) {
			return approveByDoctor;
		}

		return approveByDoctor = (Doctor) DataProvider.get().getObjectOf(Doctor.class, approvedByDoctorId);

	}

	@XmlTransient
	@JsonIgnore
	public void setApprovedByDoctor(Doctor approveByDoctor) {
		this.approveByDoctor = approveByDoctor;
		if (approveByDoctor != null) {
			setApprovedById(approveByDoctor.getId());
			putApprovedByName(approveByDoctor.getName());
		}
		else {
			setApprovedById(null);
			putApprovedByName(null);
		}
	}

	private void putApprovedByName(String surgeonName) {
		addProperty("approved_by.surgeon_name", surgeonName);
	}

	@XmlTransient
	public String getSurgeonName() {
		return getProperty("approved_by.surgeon_name", "");
	}

	public Room getOTRoom() {
		String roomId = getOtRoomId();
		if (StringUtils.isBlank(roomId)) {
			return null;
		}

		if (otRoom != null && roomId.equals(otRoom.getId())) {
			return otRoom;
		}

		return otRoom = (Room) DataProvider.get().getObjectOf(Room.class, roomId);

	}

	@XmlTransient
	@JsonIgnore
	public void setOTRoom(Room otRoom) {
		this.otRoom = otRoom;
		if (otRoom != null) {
			setOtRoomId(otRoom.getId());
			putOtRoomName(otRoom.getName());
		}
		else {
			setOtRoomId(null);
			putOtRoomName(null);
		}
	}

	private void putOtRoomName(String otRoomName) {
		addProperty("ot_room_name", otRoomName);
	}

	@XmlTransient
	public String getOtRoomName() {
		return getProperty("ot_room_name", "");
	}

	public Bed getOTBed() {
		String bedId = getOtBedId();
		if (StringUtils.isBlank(bedId)) {
			return null;
		}

		if (otBed != null && bedId.equals(otBed.getId())) {
			return otBed;
		}

		return otBed = (Bed) DataProvider.get().getObjectOf(Bed.class, bedId);

	}

	@XmlTransient
	@JsonIgnore
	public void setOTBed(Bed otBed) {
		this.otBed = otBed;
		if (otBed != null) {
			setOtBedId(otBed.getId());
			putOtBedName(otBed.getNameDisplay());
		}
		else {
			setOtBedId(null);
			putOtBedName(null);
		}
	}

	private void putOtBedName(String otRoomName) {
		addProperty("ot_bed_name", otRoomName);
	}

	@XmlTransient
	public String getOtBedName() {
		return getProperty("ot_bed_name", "");
	}

	public String getOTNote() {
		return getProperty("ot_note", "");
	}

	public void putOTNote(String otNote) {
		addProperty("ot_note", otNote);
	}

	public String getIntraoperativeComplications() {
		return getProperty("complications.intraoperative", "");
	}

	public void putIntraoperativeComplications(String otNote) {
		addProperty("complications.intraoperative", otNote);
	}

	public String getPostoperativeComplications() {
		return getProperty("complications.postoperative", "");
	}

	public void putPostoperativeComplications(String otNote) {
		addProperty("complications.postoperative", otNote);
	}

	public String getSeverityComplicationDisplay() {
		String severityComplication = getSeverityComplication();
		if (StringUtils.isNotBlank(severityComplication)) {
			return OTComplicationSeverity.fromNameString(severityComplication).toString();
		}
		return StringUtils.EMPTY;
	}

	public String getSeverityComplication() {
		return getProperty("complications.severity", "");
	}

	public void putSeverityComplication(String otNote) {
		addProperty("complications.severity", otNote);
	}

	public String getTemparature() {
		return getProperty("temparature"); //$NON-NLS-1$
	}

	public void putTemparature(String temparature) {
		addProperty("temparature", temparature); //$NON-NLS-1$
	}

	public String getBloodPressureDisplay() {
		if (hasProperty("pressure.down")) {
			return getPressureUp() + "/" + getPressureDown();
		}
		return StringUtils.EMPTY;

	}

	public Integer getPressureUp() {
		return getIntProperty("pressure.up"); //$NON-NLS-1$
	}

	public void putPressureUp(int pressureUp) {
		addIntProperty("pressure.up", pressureUp); //$NON-NLS-1$
	}

	public Integer getPressureDown() {
		return getIntProperty("pressure.down"); //$NON-NLS-1$
	}

	public void putPressureDown(int pressureDown) {
		addIntProperty("pressure.down", pressureDown); //$NON-NLS-1$
	}

	public Integer getHeartBitPerMinute() {
		return getIntProperty("heartBitPerMinute"); //$NON-NLS-1$
	}

	public void putHeartBitPerMinute(int heartBitPerMinutes) {
		addIntProperty("heartBitPerMinute", heartBitPerMinutes); //$NON-NLS-1$
	}

	public double getOxygenSaturation() {
		return getDoubleProperty("oxygenSaturation"); //$NON-NLS-1$
	}

	public void putOxygenSaturation(String oxygenSaturation) {
		addProperty("oxygenSaturation", oxygenSaturation); //$NON-NLS-1$
	}

	public void putSurgeryStatusPerformedBy(User compliteBy) {
		if (compliteBy != null) {
			putSurgeryStatusPerformedUserId(compliteBy.getId());
			putSurgeryStatusPerformedUserName(compliteBy.getFullName());
		}
		else {
			putSurgeryStatusPerformedUserId(null);
			putSurgeryStatusPerformedUserName(null);
		}
	}

	public String getSurgeryStatusPerformedUserId() {
		return getProperty("surgery_status.performed_user_id", "");
	}

	public void putSurgeryStatusPerformedUserId(String userId) {
		addProperty("surgery_status.performed_user_id", userId);
	}

	public String getSurgeryStatusPerformedUserName() {
		return getProperty("surgery_status.performed_user_name", "");
	}

	public void putSurgeryStatusPerformedUserName(String userName) {
		addProperty("surgery_status.performed_user_name", userName);
	}

	public Date getSurgeryStatusTime() {
		return getDateProperty("surgery.status_time");
	}

	public void putSurgeryStatusTime(Date surgeryStatusTime) {
		addDateProperty("surgery.status_time", surgeryStatusTime);
	}

	public Integer getBloodLossVolume() {
		return getIntProperty("surgery.blood_loss");
	}

	public void putBloodLossVolume(int bloodVolume) {
		addIntProperty("surgery.blood_loss", bloodVolume);
	}

	public Integer getBloodAdministered() {
		return getIntProperty("surgery.blood_administered");
	}

	public void putBloodAdministered(int bagQty) {
		addIntProperty("surgery.blood_administered", bagQty);
	}

	public String getAnatomicalLocation() {
		return getProperty("surgery.anatomical_location");
	}

	public void putAnatomicalLocation(String anatomicalLocation) {
		addProperty("surgery.anatomical_location", anatomicalLocation);
	}

	public String getTypeOfSpecimen() {
		return getProperty("surgery.type_of_specimen");
	}

	public void putTypeOfSpecimen(String typeOfSpecimen) {
		addProperty("surgery.type_of_specimen", typeOfSpecimen);
	}

	public String getSizeOrDimensions() {
		return getProperty("surgery.size_or_dimensions", "");
	}

	public void putSizeOrDimensions(String sizeOrDimensions) {
		addProperty("surgery.size_or_dimensions", sizeOrDimensions);
	}

	public String getSpecimenDestination() {
		return getProperty("surgery.specimen_destination", "");
	}

	public void putSpecimenDestination(String specimenDestination) {
		addProperty("surgery.specimen_destination", specimenDestination);
	}

	public Ticket getTicket() {
		String ticketProperty = getProperty("ticket");
		if (StringUtils.isBlank(ticketProperty)) {
			return null;
		}
		try {
			return ServiceUtils.convertJsonToTicket(ticketProperty);
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public void putTicket(Ticket ticket) throws Exception {
		XMLTransientUtil.makeXMLTransient(ticket);
		String ticketJson = ServiceUtils.convertTicketToJson(ticket);
		addProperty("ticket", ticketJson);
	}
}