package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.type.StringType;
import org.hibernate.type.Type;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.Appointment;
import com.floreantpos.model.AppointmentStatus;
import com.floreantpos.model.DayOfWeek;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.DoctorTimeSchedule;
import com.floreantpos.model.Patient;
import com.floreantpos.model.util.DateUtil;

public class AppointmentDAO extends BaseAppointmentDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public AppointmentDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Appointment appointment = (Appointment) obj;
		if (appointment == null) {
			throw new PosException("Appointment not found"); //$NON-NLS-1$
		}

		appointment.setDeleted(Boolean.TRUE);
		saveOrUpdate(appointment, session);
		ActionHistoryDAO.saveHistory(ActionHistory.APPOINTMENT_DELETED,
				"Appointment Patient name: " + appointment.getPatientNameByProperty() + ", appointment id: " //$NON-NLS-1$ //$NON-NLS-2$
						+ appointment.getId() + ", phone: " + appointment.getPatientPhoneNo()); //$NON-NLS-1$
	}

	public int rowCount(String appointmentId, Doctor doctor, Date date) {
		return rowCount(appointmentId, doctor, date, true);
	}

	public int rowCount(String appointmentId, Doctor doctor, Date date, boolean filterCanceled) {
		return rowCount(appointmentId, null, doctor, date, filterCanceled);
	}

	public int rowCount(String appointmentId, String shiftId, Doctor doctor, Date date, boolean filterCanceled) {
		return rowCount(appointmentId, shiftId, doctor, date, filterCanceled, null);
	}

	public int rowCount(String appointmentId, String shiftId, Doctor doctor, Date date, boolean filterCanceled, String patientId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			updateCriteria(appointmentId, doctor, date, filterCanceled, criteria);
			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(Appointment.PROP_PATIENT_ID, patientId));
			}
			if (StringUtils.isNotBlank(shiftId)) {
				criteria.add(Restrictions.eq(Appointment.PROP_SHIFT_ID, shiftId));
			}
			Number uniqueResult = (Number) criteria.uniqueResult();
			return uniqueResult == null ? 0 : uniqueResult.intValue();
		}
	}

	public Appointment getAppointment(String appointmentId, String shiftId, Doctor doctor, Date date, boolean filterCanceled) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			updateCriteria(appointmentId, doctor, date, filterCanceled, criteria);
			if (StringUtils.isNotBlank(shiftId)) {
				criteria.add(Restrictions.eq(Appointment.PROP_SHIFT_ID, shiftId));
			}
			return (Appointment) criteria.uniqueResult();
		}
	}

	public List<Appointment> getFutureAppointments(Doctor doctor, String shiftId, Date date) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			updateCriteria(null, doctor, date, null, true, criteria);
			criteria.add(Restrictions.eq(Appointment.PROP_SHIFT_ID, shiftId));
			criteria.add(Restrictions.ne(Appointment.PROP_STATUS, AppointmentStatus.COMPLETED.name()));
			return criteria.list();
		}
	}

	public boolean hasFutureAppointment(Doctor doctor, String shiftId, Date date) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			updateCriteria(null, doctor, date, null, true, criteria);
			criteria.add(Restrictions.eq(Appointment.PROP_SHIFT_ID, shiftId));
			criteria.add(Restrictions.ne(Appointment.PROP_STATUS, AppointmentStatus.COMPLETED.name()));
			criteria.setMaxResults(1);
			Number uniqueResult = (Number) criteria.uniqueResult();
			return uniqueResult == null ? false : uniqueResult.intValue() > 0;
		}
	}

	public Date getMaxAppointmentTime(String appointmentId, Doctor doctor, Date date) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(Appointment.PROP_FROM_DATE));
			updateCriteria(appointmentId, doctor, date, true, criteria);
			criteria.setMaxResults(1);
			criteria.addOrder(Order.desc(Appointment.PROP_FROM_DATE));
			return (Date) criteria.uniqueResult();
		}
	}

	private void updateCriteria(String appointmentId, Doctor doctor, Date date, boolean filterCanceled, Criteria criteria) {
		updateCriteria(appointmentId, doctor, date, date, filterCanceled, criteria);
	}

	private void updateCriteria(String appointmentId, Doctor doctor, Date fromDate, Date toDate, boolean filterCanceled, Criteria criteria) {
		criteria.add(Restrictions.eq(Appointment.PROP_DOCTOR_ID, doctor.getId()));
		if (filterCanceled) {
			addDeletedFilter(criteria);
			criteria.add(Restrictions.ne(Appointment.PROP_STATUS, AppointmentStatus.CANCELLED.name()));
			criteria.add(Restrictions.eq(Appointment.PROP_CLOSED, Boolean.FALSE));
		}
		if (StringUtils.isNotBlank(appointmentId)) {
			criteria.add(Restrictions.ne(Appointment.PROP_ID, appointmentId));
		}
		criteria.add(Restrictions.ge(Appointment.PROP_FROM_DATE, DateUtil.startOfDay(fromDate)));
		if (toDate != null) {
			criteria.add(Restrictions.le(Appointment.PROP_FROM_DATE, DateUtil.endOfDay(toDate)));
		}
	}

	public String getNextAppointmentNumber(Doctor doctor, Date date, DoctorTimeSchedule schedule) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(Appointment.PROP_DOCTOR_ID, doctor.getId()));
			criteria.add(Restrictions.eq(Appointment.PROP_SHIFT_ID, schedule.getId()));
			criteria.add(Restrictions.ge(Appointment.PROP_FROM_DATE, DateUtil.startOfDay(date)));
			criteria.add(Restrictions.le(Appointment.PROP_FROM_DATE, DateUtil.endOfDay(date)));
			Number uniqueResult = (Number) criteria.uniqueResult();
			return String.valueOf(uniqueResult.intValue() + 1);
		}
	}

	public List<Appointment> getAppointments(Date startTime, Date endTime, Doctor doctor) {
		return getAppointments(startTime, endTime, doctor, true);
	}

	public List<Appointment> getAppointments(Date startTime, Date endTime, Doctor doctor, boolean completed) {
		return getAppointments(startTime, endTime, doctor, completed, true);
	}

	public List<Appointment> getAppointments(Date startTime, Date endTime, Doctor doctor, boolean completed, boolean excludeCancelled) {
		return getAppointments(startTime, endTime, doctor, completed, excludeCancelled, null);
	}

	public List<Appointment> getAppointments(Date startTime, Date endTime, Doctor doctor, boolean completed, boolean excludeCancelled, String shiftId) {
		return getAppointments(startTime, endTime, doctor, completed, excludeCancelled, shiftId, false);
	}

	public List<Appointment> getAppointments(Date startTime, Date endTime, Doctor doctor, boolean completed, boolean excludeCancelled, String shiftId,
			boolean filterByExactTime) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (doctor != null) {
				criteria.add(Restrictions.eq(Appointment.PROP_DOCTOR_ID, doctor.getId()));
			}
			if (completed) {
				criteria.add(Restrictions.eq(Appointment.PROP_STATUS, AppointmentStatus.COMPLETED.name()));
			}
			if (excludeCancelled) {
				criteria.add(Restrictions.ne(Appointment.PROP_STATUS, AppointmentStatus.CANCELLED.name()));
			}
			if (StringUtils.isNotBlank(shiftId)) {
				criteria.add(Restrictions.eq(Appointment.PROP_SHIFT_ID, shiftId));
			}
			criteria.add(Restrictions.eq(Appointment.PROP_CLOSED, Boolean.FALSE));
			if (startTime != null && endTime != null) {
				criteria.add(Restrictions.ge(Appointment.PROP_FROM_DATE, filterByExactTime ? startTime : DateUtil.startOfDay(startTime)));
				criteria.add(Restrictions.le(Appointment.PROP_FROM_DATE, filterByExactTime ? endTime : DateUtil.endOfDay(endTime)));
			}
			return criteria.list();
		}
	}

	public boolean hasUpcomingAppointmentOnDay(DayOfWeek dayOfWeek, String doctorId, Date fromDate) {
		if (dayOfWeek == null || doctorId == null || fromDate == null) {
			return false;
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Appointment.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.ne(Appointment.PROP_STATUS, AppointmentStatus.CANCELLED.name()));
			criteria.add(Restrictions.ge(Appointment.PROP_FROM_DATE, fromDate));
			criteria.add(Restrictions.eq(Appointment.PROP_DOCTOR_ID, doctorId));

			criteria.setProjection(Projections
					.distinct(Projections.sqlProjection("TO_CHAR(from_date, 'Day') as days", new String[] { "days" }, new Type[] { new StringType() }))); //$NON-NLS-1$ //$NON-NLS-2$

			List<String> list = criteria.list();
			return list.stream().filter(day -> dayOfWeek.name().equalsIgnoreCase(day.toString().trim())).findFirst().isPresent();
		}
	}

	public boolean hasUpcomingAppointments(String scheduleId, String doctorId, Date fromDate) {
		if (StringUtils.isBlank(doctorId) || StringUtils.isBlank(scheduleId)) {
			return false;
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Appointment.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.ne(Appointment.PROP_STATUS, AppointmentStatus.CANCELLED.name()));

			criteria.add(Restrictions.ge(Appointment.PROP_SHIFT_ID, scheduleId));
			criteria.add(Restrictions.ge(Appointment.PROP_FROM_DATE, fromDate));

			criteria.add(Restrictions.eq(Appointment.PROP_DOCTOR_ID, doctorId));
			criteria.setProjection(Projections.rowCount());
			Number uniqueResult = (Number) criteria.uniqueResult();
			return uniqueResult.longValue() > 0;
		}
	}

	public Appointment getLastAppointment(Doctor doctor, Date date, DoctorTimeSchedule schedule) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Appointment.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Appointment.PROP_DOCTOR_ID, doctor.getId()));
			criteria.add(Restrictions.eq(Appointment.PROP_SHIFT_ID, schedule.getId()));
			criteria.add(Restrictions.ge(Appointment.PROP_FROM_DATE, DateUtil.startOfDay(date)));
			criteria.add(Restrictions.le(Appointment.PROP_FROM_DATE, DateUtil.endOfDay(date)));

			criteria.addOrder(Order.desc(Appointment.PROP_FROM_DATE));
			criteria.setMaxResults(1);
			return (Appointment) criteria.uniqueResult();
		}
	}

	public void createJournal(Appointment appointment, boolean isNewAppointment) {
		try {
			ActionHistoryDAO.saveHistory(isNewAppointment ? ActionHistory.CREATE_APPOINTMENT : ActionHistory.EDIT_APPOINTMENT,
					String.format("Appointment Id: %s, Appointment number: %s, Patient Id: %s, Doctor Id: %s, Appointment time: %s, Status: %s", //$NON-NLS-1$
							appointment.getId(), appointment.getAppoinmentId(), appointment.getPatientId(), appointment.getDoctorId(),
							appointment.getFromDate() == null ? "" : DateUtil.formatFullDateAndTimeAsString(appointment.getFromDate()), //$NON-NLS-1$
							appointment.getStatus()));
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void changeAppoinmentPatient(Appointment oldAppointment, Appointment newAppointment, Patient selectedPatient) {
		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();
			newAppointment.setPatient(selectedPatient);
			save(newAppointment, session);

			oldAppointment.setClosed(true);
			update(oldAppointment, session);
			tx.commit();
		}
	}
}