package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.List;
import java.util.Optional;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.BloodComponentType;
import com.floreantpos.model.BloodDonationHistory;
import com.floreantpos.model.BloodDonationHistory.PaymentStatus;
import com.floreantpos.model.BloodGroupType;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderItem;
import com.floreantpos.model.PurchaseOrderType;
import com.floreantpos.model.UnitType;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginatedListModel;

public class BloodDonationHistoryDAO extends BaseBloodDonationHistoryDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		BloodDonationHistory bean = (BloodDonationHistory) obj;
		if (bean == null) {
			throw new PosException("Blood donation history not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<BloodDonationHistory> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BloodDonationHistory.class);
			addDeletedFilter(criteria);

			criteria.addOrder(Order.asc(BloodDonationHistory.PROP_DONATION_DATE));
			return criteria.list();
		}
	}

	public void loadHistory(InventoryVendor donor, BloodGroupType groupType, PaginatedListModel<BloodDonationHistory> dataModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BloodDonationHistory.class);
			addDeletedFilter(criteria);

			if (donor != null) {
				criteria.add(Restrictions.eq(BloodDonationHistory.PROP_VENDOR_ID, donor.getId()));
			}

			if (groupType != null) {
				criteria.add(Restrictions.eq(BloodDonationHistory.PROP_BLOOD_GROUP, groupType.name()));
			}

			dataModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());

			criteria.addOrder(Order.desc(BloodDonationHistory.PROP_CREATE_DATE));

			dataModel.setData(criteria.list());
		}
	}

	public void collectBlood(BloodDonationHistory donationHistory) {
		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();

			PurchaseOrder purchaseOrder = donationHistory.getPurchaseOrder();
			if (purchaseOrder != null) {
				if (purchaseOrder.getDueAmount() != 0) {
					purchaseOrder.setStatus(PurchaseOrder.ORDER_FULLY_RECEIVED);
				}
				List<PurchaseOrderItem> orderItems = purchaseOrder.getOrderItems();
				BloodComponentType componentType = BloodComponentType.fromString(donationHistory.getBloodType());
				if (orderItems != null && componentType != null) {
					Optional<PurchaseOrderItem> tergetItem = orderItems.stream().filter(t -> "bb_donor_bill".equals(t.getMenuItemId())).findFirst();
					if (tergetItem.isPresent()) {
						PurchaseOrderItem orderItem = tergetItem.get();
						orderItem.setName(orderItem.getName() + " " + componentType.getDisplayName());
						orderItem.setQuantityReceived(1d);
						purchaseOrder.setReceivingDate(StoreDAO.getServerTimestamp());
					}
				}
				PurchaseOrderDAO.getInstance().saveOrUpdate(purchaseOrder, session);
			}
			saveOrUpdate(donationHistory, session);
			BloodInventoryTransactionDAO.getInstance().adjustBloodInventory(donationHistory, InventoryTransactionType.IN, session);

			transaction.commit();
		}
	}

	public void cancelDonation(BloodDonationHistory donationHistory) {
		if (donationHistory == null) {
			return;
		}

		try (Session session = createNewSession()) {
			Transaction txn = session.beginTransaction();
			donationHistory.setStatus(BloodDonationHistory.Status.CANCEL.name());

			PurchaseOrder purchaseOrder = donationHistory.getPurchaseOrder();

			if (purchaseOrder != null) {
				purchaseOrder.setStatus(PurchaseOrder.ORDER_CANCELLED);
				purchaseOrder.setClosingDate(StoreDAO.getServerTimestamp());

				for (PosTransaction posTransaction : purchaseOrder.getTransactions()) {
					posTransaction.setVoided(true);
					posTransaction.setVoidedByUser(DataProvider.get().getCurrentUser());
					posTransaction.setVoidDate(StoreDAO.getServerTimestamp());
				}

				PurchaseOrderDAO.getInstance().saveOrUpdate(purchaseOrder, session);
			}

			saveOrUpdate(donationHistory, session);
			txn.commit();
		}
	}

	public void saveOrUpdateDonation(BloodDonationHistory donationHistory, boolean isNewDonation) {
		if (donationHistory == null) {
			return;
		}
		try (Session session = createNewSession()) {
			Transaction txn = session.beginTransaction();

			if (StringUtils.isBlank(donationHistory.getOutletId())) {
				donationHistory.setOutletId(DataProvider.get().getCurrentOutletId());
			}

			if (isNewDonation) {
				if (!donationHistory.isFreeDonation()) {
					createNewPurchaseOrder(donationHistory, session);
				}
				save(donationHistory, session);
			}
			else {
				PurchaseOrder purchaseOrder = donationHistory.getPurchaseOrder();
				if (purchaseOrder == null && !donationHistory.isFreeDonation()) {
					createNewPurchaseOrder(donationHistory, session);
				}
				else {
					if (purchaseOrder != null) {
						if (purchaseOrder.getPaidAmount() == 0 && donationHistory.isFreeDonation()) {
							purchaseOrder.setDeleted(true);
						}
						else if (purchaseOrder.getPaidAmount() == 0 && !donationHistory.isFreeDonation()) {
							purchaseOrder.setDeleted(false);
						}
						if (donationHistory.getPaymentStatus() == PaymentStatus.UNPAID) {
							for (PurchaseOrderItem ticketItem : purchaseOrder.getOrderItems()) {
								ticketItem.setUnitCost(donationHistory.getBillAmount());
								break;
							}
							purchaseOrder.calculatePrice();

							InventoryVendor donor = donationHistory.getDonor();
							purchaseOrder.setVendor(donor);
						}
						PurchaseOrderDAO.getInstance().saveOrUpdate(purchaseOrder, session);
					}
				}
				saveOrUpdate(donationHistory, session);
			}
			txn.commit();
		}
	}

	private void createNewPurchaseOrder(BloodDonationHistory donationHistory, Session session) {
		InventoryVendor donor = donationHistory.getDonor();
		BloodGroupType groupType = BloodGroupType.fromNameString(donor.getBloodGroup());

		PurchaseOrder purchaseOrder = new PurchaseOrder();
		purchaseOrder.setOrderId(PurchaseOrderDAO.getInstance().getNextOrderSequenceNumber());
		Date serverTimestamp = StoreDAO.getServerTimestamp();
		purchaseOrder.setCreatedDate(serverTimestamp);
		purchaseOrder.setOutletId(donationHistory.getOutletId());
		purchaseOrder.setPurchaseOrderType(PurchaseOrderType.BLOOD_BANK.name());
		purchaseOrder.setProductType(ProductType.BLOOD.name());
		purchaseOrder.setStatus(PurchaseOrder.ORDER_VERIFIED);
		purchaseOrder.setVarificationDate(serverTimestamp);
		purchaseOrder.setInventoryLocation(DataProvider.get().getDefaultInLocation());

		PurchaseOrderItem purchaseOrderItem = new PurchaseOrderItem();
		purchaseOrderItem.setItemQuantity(1d);
		purchaseOrderItem.setItemUnitName("each");
		purchaseOrderItem.setUnitCodeDisplay("ea");
		purchaseOrderItem.putUnitType(UnitType.UNIT.name());
		purchaseOrderItem.setName("Blood (" + groupType.getDisplayString() + ") ");
		purchaseOrderItem.setMenuItemId("bb_donor_bill");

		purchaseOrderItem.setUnitCost(donationHistory.getBillAmount());

		purchaseOrderItem.setPurchaseOrder(purchaseOrder);
		purchaseOrder.addToorderItems(purchaseOrderItem);
		purchaseOrder.calculatePrice();

		purchaseOrder.setVendor(donor);

		TicketDAO.getInstance().saveOrUpdate(purchaseOrder, session);

		donationHistory.setPurchaseOrderId(purchaseOrder.getId());
	}

}