package com.floreantpos.model.util;

import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.TimeZone;

import org.apache.commons.lang3.StringUtils;

import com.floreantpos.model.Bed;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Room;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.StoreDAO;

public class BookingUtil {

	public static Date buildBookingEndTime(Date endDate) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(endDate);
		calendar.add(Calendar.DAY_OF_MONTH, -1);
		Date bookingEndTime = DateUtil.endOfDay(calendar.getTime());
		return bookingEndTime;
	}

	public static void updateRentAndAdmissionCharge(BookingInfo booking, Ticket ticket, List<Bed> selectedBeds, Date admissionDate, Date dischargeDate) {
		updateRentAndAdmissionCharge(booking, ticket, selectedBeds, admissionDate, dischargeDate, true);
	}

	public static void updateRentAndAdmissionCharge(BookingInfo booking, Ticket ticket, List<Bed> selectedBeds, Date admissionDate, Date dischargeDate,
			boolean retainOldBed) {
		if (ticket == null) {
			return;
		}
		booking.setTicket(ticket);

		boolean existRetainedBed = false;
		if (selectedBeds != null && selectedBeds.size() > 0) {
			for (Bed bed : selectedBeds) {
				List<TicketItem> ticketItems = ticket.getTicketItems();
				for (TicketItem ticketItem : ticketItems) {
					if (!ticketItem.isBedItem() || StringUtils.isBlank(ticketItem.getBedId())) {
						continue;
					}
					if (bed.getId().equals(ticketItem.getBedId()) && ticketItem.getBookingReleaseDate() == null) {
						existRetainedBed = true;
						break;
					}
				}
			}
		}
		double bookingDays = 1.0;
		if (admissionDate != null && dischargeDate != null) {
			bookingDays = ChronoUnit.DAYS.between(admissionDate.toInstant(), dischargeDate.toInstant());
			if (bookingDays == 0) {
				bookingDays = 1;
			}
		}
		List<Bed> bookingBeds = booking.getBeds();
		if (!retainOldBed && bookingBeds != null && bookingBeds.size() > 0) {
			String currentBedId = bookingBeds.get(0).getId();
			if (StringUtils.isNotBlank(currentBedId)) {
				List<TicketItem> ticketItems = ticket.getTicketItems();
				for (TicketItem ticketItem : ticketItems) {
					if (!ticketItem.isBedItem() || StringUtils.isBlank(ticketItem.getBedId())) {
						continue;
					}
					if (existRetainedBed && ticketItem.getBedId().equals(selectedBeds.get(0).getId())) {
						continue;
					}
					//if (currentBedId.equals(ticketItem.getBedId())) {
					ticketItem.putBookingReleaseDate(StoreDAO.getServerTimestamp());
					//	break;
					//}
				}
			}
		}
		if (existRetainedBed) {
			booking.setBookingBeds(selectedBeds);
			return;
		}
		if (StringUtils.isBlank(booking.getId())) {
			removeOldBookingBedAndAdmissionCharge(ticket.getTicketItems());
		}
		else if (!retainOldBed) {
			removeCurrentBookingBedIfNeeded(booking, ticket);
		}
		booking.setBookingBeds(selectedBeds);
		double admissionCharge = 0d;
		List<String> addedBedIds = new ArrayList<>();
		if (selectedBeds != null && selectedBeds.size() > 0) {
			for (Bed bed : selectedBeds) {
				Room room = bed.getRoom();
				if (room == null) {
					continue;
				}
				addedBedIds.add(bed.getId());

				MenuItem rentItem = new MenuItem();
				rentItem.setName("Bed charge (" + bed.getNameDisplay() + ")");
				rentItem.setPrice(bed.getRoom().getRentPerBed());
				rentItem.setProductType(ProductType.SERVICES.name());

				TicketItem bedChargeTicketItem = rentItem.convertToTicketItem(ticket, bookingDays);
				bedChargeTicketItem.setBedId(bed.getId());
				bedChargeTicketItem.putBedItem(true);
				bedChargeTicketItem.putShowInAdmissionItemsGrid(true);
				bedChargeTicketItem.putBookingDate(admissionDate);
				bedChargeTicketItem.putBookingReleaseDate(dischargeDate);
				bedChargeTicketItem.putShowInAdmissionItemsGrid(true);
				bedChargeTicketItem.setSortOrder(1);
				ticket.addToticketItems(bedChargeTicketItem);
				admissionCharge = bed.getRoom().getAdmissionCharge();
			}
		}
		else {
			admissionCharge = booking.getTicket().getOutlet().getDefaultAdmissionCharge();
		}
		TicketItem admissionChargeTicketItem = ticket.getAdmissionFeeTicketItem();
		if (admissionChargeTicketItem == null) {
			MenuItem admChargeItem = new MenuItem();
			admChargeItem.setName(Ticket.ADMISSION_CHARGE);
			admChargeItem.setPrice(admissionCharge);
			admChargeItem.setProductType(ProductType.SERVICES.name());
			admissionChargeTicketItem = admChargeItem.convertToTicketItem(ticket, bookingDays);
			admissionChargeTicketItem.putAdmissionChargeItem(Boolean.TRUE);
			admissionChargeTicketItem.putShowInAdmissionItemsGrid(true);
			ticket.addToticketItems(admissionChargeTicketItem);
		}
		//else {
		//	admissionChargeTicketItem.setUnitPrice(admissionCharge);
		//}
		ticket.calculatePrice();
	}

	private static void removeCurrentBookingBedIfNeeded(BookingInfo booking, Ticket ticket) {
		List<Bed> beds = booking.getBeds();
		if (beds == null || beds.size() > 0) {
			Date serverTimestamp = StoreDAO.getServerTimestamp();
			for (Bed bed : beds) {
				for (Iterator<TicketItem> iterator = ticket.getTicketItems().iterator(); iterator.hasNext();) {
					TicketItem ticketItem = (TicketItem) iterator.next();
					if (ticketItem.isVoided() || ticketItem.isAdmissionChargeItem()) {
						continue;
					}
					String bedId = ticketItem.getBedId();
					if (org.apache.commons.lang3.StringUtils.isBlank(bedId) || !bed.getId().equals(bedId)) {
						continue;
					}
					Date bookingDate = ticketItem.getBookingDate();
					double bookingDays = ChronoUnit.DAYS.between(bookingDate.toInstant(), serverTimestamp.toInstant());
					if (bookingDays <= 0) {
						long bookingHour = ChronoUnit.HOURS.between(bookingDate.toInstant(), serverTimestamp.toInstant());
						int bedChangeChargeTime = ticket.getOutlet().getBedChangeChargeTime();
						if (bookingHour < bedChangeChargeTime) {
							if (StringUtils.isNotBlank(ticketItem.getId())) {
								ticket.addDeletedItems(ticketItem);
							}
							iterator.remove();
						}
					}
				}
			}
		}
	}

	private static void removeOldBookingBedAndAdmissionCharge(List<TicketItem> ticketItems) {
		for (Iterator<TicketItem> iterator = ticketItems.iterator(); iterator.hasNext();) {
			TicketItem ticketItem = (TicketItem) iterator.next();
			if (ticketItem.isVoided()) {
				continue;
			}
			if (ticketItem.isAdmissionChargeItem()) {
				iterator.remove();
				continue;
			}
			String bed = ticketItem.getBedId();
			if (org.apache.commons.lang3.StringUtils.isBlank(bed)) {
				continue;
			}
			iterator.remove();
		}
	}

	public static void calculateBedPrice(Ticket ticket) {
		Outlet outlet = ticket.getOutlet();

		Calendar admissionTimeC = Calendar.getInstance();
		Calendar dischargeTimeC = Calendar.getInstance();

		admissionTimeC.setTime(outlet.parseAndGetAdmissionTime());
		dischargeTimeC.setTime(outlet.parseAndGetDischargeTime());

		int admissionHour = admissionTimeC.get(Calendar.HOUR_OF_DAY);
		int dischargeHour = dischargeTimeC.get(Calendar.HOUR_OF_DAY);

		for (TicketItem ticketItem : ticket.getTicketItems()) {
			if (!ticketItem.isBedItem()) {
				continue;
			}
			Date bookingDate = ticketItem.getBookingDate();
			if (bookingDate == null) {
				continue;
			}
			Date bookingReleaseDate = ticketItem.getBedBookingEndTime();
			if (bookingReleaseDate == null) {
				bookingReleaseDate = StoreDAO.getServerTimestamp();
			}
			if (bookingReleaseDate.before(bookingDate)) {
				continue;
			}
			if (bookingDate != null && bookingReleaseDate != null) {
				Calendar startTimeC = Calendar.getInstance();
				startTimeC.setTimeZone(TimeZone.getTimeZone(DateUtil.getBrowserTimeZoneId()));
				startTimeC.setTime(bookingDate);
				if (startTimeC.get(Calendar.HOUR_OF_DAY) > admissionHour) {
					startTimeC.set(Calendar.HOUR_OF_DAY, admissionHour);
				}

				Calendar endTimeC = Calendar.getInstance();
				endTimeC.setTimeZone(TimeZone.getTimeZone(DateUtil.getBrowserTimeZoneId()));
				endTimeC.setTime(bookingReleaseDate);

				boolean dischargeTimeOver = isDischargeTimeOver(dischargeHour, endTimeC);
				if (endTimeC.get(Calendar.HOUR_OF_DAY) < dischargeHour) {
					endTimeC.set(Calendar.HOUR_OF_DAY, dischargeHour);
				}
				double bookingDays = ChronoUnit.DAYS.between(startTimeC.getTime().toInstant(), endTimeC.getTime().toInstant());
				if (dischargeTimeOver) {
					bookingDays++;
				}
				if (bookingDays <= 0) {
					bookingDays = 1;
				}
				//if (ticketItem.getQuantity() < bookingDays) {
				ticketItem.setQuantity(bookingDays);
				//}
			}
		}
	}

	public static boolean isDischargeTimeOver(int dischargeHour, Calendar endTimeC) {
		int dischargeTime = endTimeC.get(Calendar.HOUR_OF_DAY);
		if (dischargeTime > dischargeHour) {
			return true;
		}
		if (dischargeTime == dischargeHour && (endTimeC.get(Calendar.MINUTE) > 0 || endTimeC.get(Calendar.SECOND) > 0)) {
			return true;
		}
		return false;
	}

	public static void calculateBedPrice(Ticket ticket, BookingInfo bookingInfo) {
		Date bookingReleaseDate = bookingInfo.getToDate();
		if (bookingReleaseDate == null) {
			bookingReleaseDate = StoreDAO.getServerTimestamp();
		}
		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			if (!ticketItem.isBedItem()) {
				continue;
			}
			ticketItem.getBedId();
			ticketItem.setBedBookingEndTime(bookingReleaseDate);
		}
		ticket.calculatePrice();
	}
}
