package com.floreantpos.report;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.swing.ImageIcon;

import org.apache.commons.lang3.StringUtils;
import org.castor.core.util.Base64Encoder;

import com.floreantpos.model.Address;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Consent;
import com.floreantpos.model.ConsentTemplate;
import com.floreantpos.model.ConsentTemplateParamProvider;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Store;
import com.floreantpos.model.dao.ConsentTemplateDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.PaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.ImageUtil;
import com.floreantpos.util.POSUtil;

public class ConsentReportUtil {

	public static HashMap<String, Object> createConsentTemplateInfoHtmlPrintMap(ConsentTemplate template, Map<String, Object> mapForPreview) {
		return createConsentTemplateInfoHtmlPrintMap(template, null, mapForPreview);
	}

	public static HashMap<String, Object> createConsentTemplateInfoHtmlPrintMap(ConsentTemplate template, Consent consents, Map<String, Object> mapForPreview) {

		if (template == null && consents != null) {
			template = ConsentTemplateDAO.getInstance().findConsentTemplateByType(consents.getConsentType(), DataProvider.get().getCurrentOutletId());
		}

		HashMap<String, Object> contextMap = new HashMap<String, Object>();

		contextMap.put("StringUtils", StringUtils.class); //$NON-NLS-1$
		Outlet outlet = DataProvider.get().getOutlet();
		ReportUtil.populateMedlogicsProperties(contextMap, outlet);

		if (consents != null) {
			contextMap.put("termsAndConditions", consents.getDescription()); //$NON-NLS-1$
		}
		else if (template != null) {
			contextMap.put("termsAndConditions", template.getDescription()); //$NON-NLS-1$
		}

		if (template != null) {
			contextMap.put("headerTxt", template.getTemplateName()); //$NON-NLS-1$
			contextMap.put("visibleReportHeader", template.isPrintPadHeader()); //$NON-NLS-1$
			contextMap.put("showStoreLogo", template.isPrintStoreLogo());//$NON-NLS-1$
			contextMap.put("receiptHeaderMargin", template.getReportHeaderMargin()); //$NON-NLS-1$
			contextMap.put("receiptFooterMargin", template.getReportFooterMargin()); //$NON-NLS-1$
		}

		String SPACE = " "; //$NON-NLS-1$

		int receiptReportHeaderMargin = outlet.getReceiptReportHeaderMargin();
		int receiptReportFooterMargin = outlet.getReceiptReportFooterMargin();
		if (mapForPreview != null) {
			contextMap.putAll(mapForPreview);
			receiptReportHeaderMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptHeaderMarginValue")); //$NON-NLS-1$
			receiptReportFooterMargin = POSUtil.parseInteger((String) mapForPreview.get("receiptFooterMarginValue")); //$NON-NLS-1$
		}
		if (receiptReportFooterMargin < 20) {
			receiptReportFooterMargin = 20;
		}
		contextMap.put("reportHeaderMargin", receiptReportHeaderMargin + "px"); //$NON-NLS-1$ //$NON-NLS-2$
		contextMap.put("reportFooterMargin", receiptReportFooterMargin + "px"); //$NON-NLS-1$ //$NON-NLS-2$

		Store store = StoreDAO.getRestaurant();
		ImageIcon storeLogo = store.getStoreLogo();
		if (storeLogo != null) {
			ImageIcon imageIcon = ImageUtil.pngToJpg(storeLogo);
			if (imageIcon != null) {
				String imageId = store.getImageId();
				if (StringUtils.isNoneEmpty(imageId)) {
					contextMap.put("logo", new String(Base64Encoder.encode(DataProvider.get().getImageResource(imageId).getImageBytes()))); //$NON-NLS-1$
				}
			}
		}

		contextMap.put("colPrintedBy", //$NON-NLS-1$
				"Printed: " + DateUtil.simplifyDateAndTime(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()), "dd/MM/yy hh:mm a") + SPACE //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
						+ "by" + SPACE + DataProvider.get().getCurrentUser().getFullName()); //$NON-NLS-2$

		return contextMap;
	}

	private static String getValue(String value) {
		if (value == null) {
			return "";
		}
		return value;
	}

	private static String convertToBanglaFormat(String admissionDate) {
		if (admissionDate == null) {
			return "";
		}
		return admissionDate.replaceAll("0", "০").replaceAll("1", "১").replaceAll("2", "২").replaceAll("3", "৩").replaceAll("4", "৪").replaceAll("5", "৫")
				.replaceAll("6", "৬").replaceAll("7", "৭").replaceAll("8", "৮").replaceAll("9", "৯").replaceAll("AM", "সকাল").replaceAll("PM", "");
	}

	private static String format(String ageFormat, Boolean showReportsInBangla) {
		if (showReportsInBangla) {
			switch (ageFormat) {
				case "Y":
					return " বছর";
				case "M":
					return " মাস";
				case "D":
					return " দিন";
				default:
					break;
			}
		}
		return ageFormat;
	}

	public static String getGenderDisplayString(BookingInfo bookingInfo, boolean showReportsInBangla) {
		String sex = bookingInfo.getProperty("sex"); //$NON-NLS-1$
		if (sex == null) {
			return "";
		}
		if (showReportsInBangla) {
			if (sex.equalsIgnoreCase("Male")) { //$NON-NLS-1$
				return "পুরুষ"; //$NON-NLS-1$
			}
			if (sex.equalsIgnoreCase("Female")) { //$NON-NLS-1$
				return "নারী"; //$NON-NLS-1$
			}
		}
		return sex;
	}

	public static PaperSize getPageSize(Map<String, Object> mapForPreview, boolean isPharma, Outlet outlet) {
		PaperSize paperSize;
		if (mapForPreview == null) {
			PaperSize receiptPaperSize;
			if (isPharma) {
				receiptPaperSize = outlet.getPharmacyReceiptPaperSize();
			}
			else {
				receiptPaperSize = outlet.getReceiptPaperSize();
			}
			paperSize = receiptPaperSize;
		}
		else {
			paperSize = (PaperSize) mapForPreview.get("paperSize"); //$NON-NLS-1$
		}
		return paperSize;
	}

	public static void addStoreInfo(HashMap<String, Object> map, Store store, Outlet outlet) {
		StringBuilder storeInfo = new StringBuilder();
		if (outlet != null) {
			storeInfo.append(outlet.getAddressLine1());
			if (StringUtils.isNotBlank(outlet.getAddressLine2())) {
				storeInfo.append(", " + outlet.getAddressLine2());
			}
			storeInfo.append("<br>");
			Address address = outlet.getAddress();
			String websiteAddress = outlet.getWebsiteAddress();
			if (address != null && StringUtils.isNotBlank(address.getTelephone())) {
				storeInfo.append("Hotline: " + address.getTelephone());
			}
			if (StringUtils.isNotBlank(websiteAddress)) {
				if (StringUtils.isNotBlank(address.getTelephone())) {
					storeInfo.append(", ");
				}
				storeInfo.append("Web: " + "<a style='color: black' href='" + websiteAddress + "'>" //$NON-NLS-1$
						+ websiteAddress + "</a>");
			}
			else {
				storeInfo.append("<br>");
			}
		}

		map.put("headerLine2", storeInfo.toString());

		if (outlet.isShowPrintReceiptStoreLogo()) {
			ImageIcon storeLogo = store.getStoreLogo();
			if (storeLogo == null) {
				return;
			}
			ImageIcon imageIcon = ImageUtil.pngToJpg(storeLogo);
			if (imageIcon != null) {
				map.put("storeLogo", imageIcon.getImage()); //$NON-NLS-1$
			}
		}
	}

	public static Map<String, String> getParamsByTemplateBody(String templateBody) {
		Map<String, String> map = new HashMap<>();
		if (StringUtils.isBlank(templateBody)) {
			return map;
		}

		List<String> ignoreParams = Arrays.asList("$date", "$patientName", "$patientGender"/*, "$patientAge", "$patientAddress", "$doctorName", "$dob"*/);

		Map<String, String> paramMap = ConsentTemplateParamProvider.getParamList().stream()
				.collect(Collectors.toMap(t -> t.getParamName(), t -> t.getParamDetails()));

		final Pattern pattern = Pattern.compile("(\\$[A-Za-z]+)", Pattern.MULTILINE);
		final Matcher matcher = pattern.matcher(templateBody);
		while (matcher.find()) {
			String key = matcher.group();
			if (ignoreParams.contains(key)) {
				continue;
			}
			String details = paramMap.get(key);
			map.put(key, StringUtils.isBlank(details) ? key : details);
		}
		return map;
	}
}
