package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Ticket;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class StoreSessionDAO extends BaseStoreSessionDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public StoreSessionDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void loadStoreSession(PaginationSupport model) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(StoreSession.class);
			criteria.addOrder(Order.asc(StoreSession.PROP_ID));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<StoreSession> result = criteria.list();
			model.setRows(result);
		} finally {
			closeSession(session);
		}
	}

	public List<StoreSession> findSessions(Date startDate, Date endDate) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(Order.desc(StoreSession.PROP_OPEN_TIME));
			criteria.add(Restrictions.and(Restrictions.ge(StoreSession.PROP_OPEN_TIME, startDate), Restrictions.le(StoreSession.PROP_OPEN_TIME, endDate)));

			//@formatter:off
			criteria.add(
					Restrictions.or(
							Restrictions.isNull(StoreSession.PROP_CLOSE_TIME),
							Restrictions.and(Restrictions.ge(StoreSession.PROP_CLOSE_TIME, startDate), Restrictions.le(StoreSession.PROP_CLOSE_TIME, endDate))
							));
			//@formatter:on

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<StoreSession> findSessions(Outlet outlet, Date startDate, Date endDate) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.and(Restrictions.ge(StoreSession.PROP_OPEN_TIME, startDate), Restrictions.le(StoreSession.PROP_OPEN_TIME, endDate)));
			if (outlet != null) {
				criteria.add(Restrictions.eq(StoreSession.PROP_OUTLET_ID, outlet.getId()));
			}
			List openTimeList = criteria.list();
			criteria.add(Restrictions.and(Restrictions.ge(StoreSession.PROP_CLOSE_TIME, startDate), Restrictions.le(StoreSession.PROP_CLOSE_TIME, endDate)));
			List closeTimeList = criteria.list();

			if (!closeTimeList.isEmpty()) {
				for (Iterator iterator = closeTimeList.iterator(); iterator.hasNext();) {
					StoreSession storeSession = (StoreSession) iterator.next();
					if (!openTimeList.contains(storeSession))
						openTimeList.add(storeSession);

				}
			}

			return openTimeList;
		} finally {
			closeSession(session);
		}
	}

	public StoreSession findOpenSession(Outlet outlet) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.isNull(StoreSession.PROP_CLOSE_TIME));
			if (outlet != null) {
				criteria.add(Restrictions.eq(StoreSession.PROP_OUTLET_ID, outlet.getId()));
			}
			criteria.addOrder(Order.desc(StoreSession.PROP_OPEN_TIME));
			List openSessions = criteria.list();
			if (openSessions != null) {
				return (StoreSession) openSessions.get(0);
			}
		} finally {
			closeSession(session);
		}
		return null;
	}

	//	public List<String> getCashDrawerIds(String sessionID) {
	//		Session session = null;
	//		try {
	//			session = getSession();
	//			Criteria criteria = session.createCriteria(CashDrawer.class);
	//			criteria.add(Restrictions.eq(CashDrawer.PROP_ID, sessionID));
	//			criteria.setProjection(Projections.property(CashDrawer.))
	//			List<CashDrawer> list = criteria.list();
	//			for (CashDrawer cashDrawer : list) {
	//				Criteria ticketCriteria = session.createCriteria(Ticket.class);
	//				criteria.add(Restrictions.eq(Ticket.PROP_C, sessionID));
	//			}
	//			
	//			return list;
	//		} finally {
	//			closeSession(session);0
	//		}
	//
	//	}

	public Integer getStoreSessionTotalGuest(StoreSession storeSession) {
		if (storeSession == null)
			return 0;
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.eq(Ticket.PROP_STORE_SESSION_ID, storeSession.getId()));
			criteria.setProjection(Projections.sum(Ticket.PROP_NUMBER_OF_GUESTS));
			Long guestNumber = (Long) criteria.uniqueResult();
			if (guestNumber != null) {
				return guestNumber.intValue();
			}
			return 0;
		}
	}

	public List<StoreSession> getTransactionsByOutlet(Date from, Date to, Outlet outlet, Pagination pagination) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			if (pagination != null) {
				criteria.setFirstResult(pagination.getCurrentRowIndex());
				criteria.setMaxResults(pagination.getPageSize());
			}
			updateCriteria(criteria, from, to, outlet);
			List list = criteria.list();

			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			updateCriteria(criteria, from, to, outlet);
			if (pagination != null) {
				Number uniqueResult = (Number) criteria.uniqueResult();
				pagination.setNumRows(uniqueResult.intValue());
			}

			return list;
		} finally {
			closeSession(session);
		}
	}

	private void updateCriteria(Criteria criteria, Date from, Date to, Outlet outlet) {
		//		criteria.add(Restrictions.and(Restrictions.ge(StoreSession.PROP_OPEN_TIME, startDate), Restrictions.le(StoreSession.PROP_OPEN_TIME, endDate)));
		if (from != null) {
			criteria.add(Restrictions.ge(StoreSession.PROP_OPEN_TIME, from));
		}
		if (to != null) {
			//TODO: Update sync
			criteria.add(Restrictions.le(StoreSession.PROP_OPEN_TIME, to));
		}
		if (outlet != null) {
			criteria.add(Restrictions.eq(StoreSession.PROP_OUTLET_ID, outlet.getId()));
		}
	}

	public void saveOrUpdateStoreSessions(List<StoreSession> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				StoreSession item = (StoreSession) iterator.next();
				StoreSession existingItem = get(item.getId());
				String colseByUserId = item.getClosedByUserId();
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					if (colseByUserId != null) {
						existingItem.setClosedByUserId(colseByUserId);
					}
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public StoreSession getLastOpenSession() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.isNull(StoreSession.PROP_CLOSE_TIME));
			criteria.addOrder(Order.desc(StoreSession.PROP_OPEN_TIME));
			criteria.setMaxResults(1);
			return (StoreSession) criteria.uniqueResult();
		}
	}

	public StoreSession getLastStoreSession() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(Order.desc(StoreSession.PROP_OPEN_TIME));
			criteria.setMaxResults(1);
			return (StoreSession) criteria.uniqueResult();
		}
	}

}