package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.GiftCard;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class GiftCardDAO extends BaseGiftCardDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public GiftCardDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof GiftCard) {
			GiftCard giftCard = (GiftCard) obj;
			giftCard.setDeleted(Boolean.TRUE);
			super.update(giftCard, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<GiftCard> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveAsList(List<GiftCard> giftCardList) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (GiftCard giftCard : giftCardList) {
				String cardNumber = giftCard.getCardNumber();
				if (cardNumber == null) {
					throw new PosException(Messages.getString("GiftCardDAO.0")); //$NON-NLS-1$
				}
				GiftCard card = session.get(GiftCard.class, cardNumber);
				if (card == null) {
					cardNumber = cardNumber.replaceAll("-", ""); //$NON-NLS-1$ //$NON-NLS-2$
					giftCard.setCardNumber(cardNumber);
					save(giftCard, session);
				}
				else {
					card.setOwnerName(giftCard.getOwnerName());
					card.setBalance(giftCard.getBalance());
					card.setIssueDate(giftCard.getIssueDate());
					card.setActivationDate(giftCard.getActivationDate());
					card.setDeActivationDate(giftCard.getDeActivationDate());
					card.setExpiryDate(giftCard.getExpiryDate());
					card.setActive(giftCard.isActive());
					card.setDisable(giftCard.isDisable());
					card.setDurationType(giftCard.getDurationType());
					card.setDuration(giftCard.getDuration());
					card.setPinNumber(giftCard.getPinNumber());
					card.setPoint(giftCard.getPoint());
					card.setBatchNo(giftCard.getBatchNo());
					card.setEmail(giftCard.getEmail());
					card.setType(giftCard.getType());
					update(card, session);
				}
			}
			tx.commit();

		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public GiftCard findByCardNumber(Session session, String cardNumber) {
		Criteria criteria = session.createCriteria(GiftCard.class);
		this.addDeletedFilter(criteria);

		if (StringUtils.isNotEmpty(cardNumber)) {
			criteria.add(Restrictions.eq(GiftCard.PROP_CARD_NUMBER, cardNumber));
			if (criteria.list().isEmpty()) {
				return null;
			}
			return (GiftCard) criteria.list().get(0);
		}
		return null;
	}

	public GiftCard findByCardNumber(String cardNumber) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			if (StringUtils.isNotEmpty(cardNumber)) {
				criteria.add(Restrictions.eq(GiftCard.PROP_CARD_NUMBER, cardNumber));
				if (criteria.list().isEmpty()) {
					return null;
				}
				return (GiftCard) criteria.list().get(0);
			}
			return null;
		}
	}

	public void searchByCardAndIssueDate(String cardNumber, String batchNumber, Date fromDate, Date toDate, String status, PaginationSupport model) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			if (fromDate != null && toDate != null)
				criteria.add(Restrictions.between(GiftCard.PROP_ISSUE_DATE, fromDate, toDate));
			if (StringUtils.isNotEmpty(cardNumber)) {
				criteria.add(Restrictions.ilike(GiftCard.PROP_CARD_NUMBER, cardNumber, MatchMode.START));
			}
			if (StringUtils.isNotEmpty(batchNumber)) {
				criteria.add(Restrictions.eq(GiftCard.PROP_BATCH_NO, batchNumber));
			}

			if (status.equalsIgnoreCase("ACTIVE")) { //$NON-NLS-1$
				criteria.add(Restrictions.eq(GiftCard.PROP_ACTIVE, true));
			}
			else if (status.equalsIgnoreCase("DEACTIVE")) { //$NON-NLS-1$
				criteria.add(Restrictions.eq(GiftCard.PROP_ACTIVE, false));
			}
			else if (status.equalsIgnoreCase("DISABLED")) { //$NON-NLS-1$
				criteria.add(Restrictions.eq(GiftCard.PROP_DISABLE, true));
			}

			criteria.setProjection(Projections.rowCount());
			Number uniqueResult = (Number) criteria.uniqueResult();
			model.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			criteria.addOrder(Order.asc(GiftCard.PROP_ISSUE_DATE));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			model.setRows(criteria.list());
		}
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	public List<Date> findByDate() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			criteria.setProjection(Projections.distinct(Projections.property(GiftCard.PROP_ISSUE_DATE)));
			List list = criteria.list();
			if (list.isEmpty()) {
				return null;
			}
			return list;
		}
	}

	@SuppressWarnings("unchecked")
	public List<GiftCard> findExceptDisable() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(GiftCard.PROP_DISABLE, false));
			List<GiftCard> list = criteria.list();
			if (!list.isEmpty()) {
				return list;
			}
			return null;
		}
	}

	public GiftCard initialize(GiftCard giftCard) {
		if (giftCard == null)
			return giftCard;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(giftCard);

			return giftCard;
		} finally {
			closeSession(session);
		}
	}

	public boolean findActiveCardByBatchNumber(String batchNumber) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			if (StringUtils.isNotEmpty(batchNumber)) {
				criteria.add(Restrictions.eq(GiftCard.PROP_BATCH_NO, batchNumber));
				criteria.add(Restrictions.or(Restrictions.eq(GiftCard.PROP_ACTIVE, true),
						Restrictions.and(Restrictions.eq(GiftCard.PROP_ACTIVE, false), Restrictions.isNotNull(GiftCard.PROP_ACTIVATION_DATE))));
				return !criteria.list().isEmpty();
			}
			return false;
		}
	}

	@SuppressWarnings("unchecked")
	public boolean deleteCardListByBatchNumber(String batchNumber) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(GiftCard.PROP_DISABLE, false));
			List<GiftCard> giftCardList = null;
			if (StringUtils.isNotEmpty(batchNumber)) {
				criteria.add(Restrictions.eq(GiftCard.PROP_BATCH_NO, batchNumber));
				giftCardList = criteria.list();
				if (giftCardList != null) {
					for (GiftCard giftCard : giftCardList) {
						delete(giftCard);
					}
					return true;
				}
			}
			return false;
		}
	}

	@SuppressWarnings("unchecked")
	public List<GiftCard> findByBatchNumber(String batchNO) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(GiftCard.PROP_BATCH_NO, batchNO));
			List<GiftCard> list = criteria.list();
			if (!list.isEmpty()) {
				return list;
			}
			return null;
		}
	}

	public boolean hasBatchNo(String batchNO) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(GiftCard.PROP_BATCH_NO, batchNO));
			criteria.setMaxResults(1);
			return criteria.list().size() > 0;
		}
	}

	@SuppressWarnings("unchecked")
	public List<GiftCard> findGiftCards(Date fromDate, Date toDate, boolean isExpired, String sortOption) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(GiftCard.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.between(GiftCard.PROP_ACTIVATION_DATE, fromDate, toDate));
			if (isExpired) {
				criteria.add(Restrictions.lt(GiftCard.PROP_EXPIRY_DATE, new Date()));
			}
			else {
				criteria.add(Restrictions.ge(GiftCard.PROP_EXPIRY_DATE, new Date()));
			}
			if ("Owner".equals(sortOption)) { //$NON-NLS-1$
				criteria.addOrder(Order.asc(GiftCard.PROP_OWNER_NAME));
			}
			else {
				criteria.addOrder(Order.asc(GiftCard.PROP_ACTIVATION_DATE));
			}
			return criteria.list();
		}
	}

	//Gift card tables primary key is card no.
	public void updateItemsLastSyncTime(List<String> ids, String tableName) {
		if (ids == null || ids.isEmpty()) {
			return;
		}
		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator<String> iterator = ids.iterator(); iterator.hasNext();) {
			String id = iterator.next();
			whereCondition += "'" + id + "'"; //$NON-NLS-1$ //$NON-NLS-2$
			if (iterator.hasNext()) {
				whereCondition += ","; //$NON-NLS-1$
			}
		}
		whereCondition += ")"; //$NON-NLS-1$
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + tableName + " set %s=:lastSyncTime where %s in %s"; //$NON-NLS-1$ //$NON-NLS-2$
			hqlString = String.format(hqlString, GiftCard.PROP_LAST_SYNC_TIME, GiftCard.PROP_CARD_NUMBER, whereCondition);
			Query query = session.createQuery(hqlString);
			Date realTime = new Date();
			query.setParameter("lastSyncTime", realTime); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public void saveOrUpdateGiftCardList(List<GiftCard> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<GiftCard> iterator = dataList.iterator(); iterator.hasNext();) {
				GiftCard item = (GiftCard) iterator.next();
				GiftCard existingItem = get(item.getCardNumber());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getBatchNo() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					double existingBalance = existingItem.getBalance();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setBalance(existingBalance);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					//item.setBalance(0D);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}
}