package com.floreantpos.model.dao;

import java.util.Calendar;
import java.util.Date;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.OutdoorTicket;
import com.floreantpos.model.Patient;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.ext.OutdoorTicketStatus;
import com.floreantpos.swing.PaginatedListModel;

public class OutdoorTicketDAO extends BaseOutdoorTicketDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OutdoorTicketDAO() {
	}

	public void loadOutdoorTickets(PaginatedListModel<OutdoorTicket> dataModel, String orderId, String patientId, String doctorId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OutdoorTicket.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.like(OutdoorTicket.PROP_ID, orderId, MatchMode.ANYWHERE));
			}

			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_CUSTOMER_ID, patientId));
			}

			if (StringUtils.isNotBlank(doctorId)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_DOCTOR_ID, doctorId));
			}

			dataModel.setNumRows(rowCount(criteria));

			criteria.addOrder(Order.desc(OutdoorTicket.PROP_CREATE_DATE));
			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());
			dataModel.setData(criteria.list());
		}
	}

	public void saveWithPatient(OutdoorTicket outdoorTicket) {
		try (Session session = createNewSession()) {

			Transaction tx = session.beginTransaction();

			if (StringUtils.isBlank(outdoorTicket.getCustomerId())) {
				Patient patient = new Patient();

				patient.setFirstName(outdoorTicket.getCustomerName());
				patient.setName(outdoorTicket.getCustomerName());
				patient.setDateOfBirth(outdoorTicket.getDateOfBirth());
				patient.setMobileNo(outdoorTicket.getCustomerPhone());
				patient.putPatientGender(outdoorTicket.getCustomerGender());
				patient.setOutletId(outdoorTicket.getOutletId());

				PatientDAO.getInstance().save(patient, session);

				outdoorTicket.setCustomerId(patient.getId());
			}

			saveOrUpdate(outdoorTicket, session);

			tx.commit();

		}
	}

	public Ticket saveWithOutdoorOrder(OutdoorTicket outdoorTicket, Ticket ticket) {
		try (Session session = createNewSession()) {

			Transaction tx = session.beginTransaction();

			saveOrUpdate(outdoorTicket, session);

			ticket.setOutdoorTicketId(outdoorTicket.getId());
			TicketDAO.getInstance().saveOrUpdate(ticket, session);

			tx.commit();

			return ticket;
		}
	}

	public OutdoorTicket findByTransactionId(String transactionId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OutdoorTicket.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(OutdoorTicket.PROP_TRANSACTION_ID, transactionId));

			criteria.addOrder(Order.asc(OutdoorTicket.PROP_CREATE_DATE));
			criteria.setMaxResults(1);

			return (OutdoorTicket) criteria.uniqueResult();
		}
	}

//	public boolean isRevisitFeeApplicable(String customerId, String doctorId, String excludedCutrentId, int inDays) {
//		OutdoorTicket previousOutdoorVisit = getPreviousOutdoorVisit(customerId, doctorId, excludedCutrentId, inDays);
//		return previousOutdoorVisit != null;
//	}

	public OutdoorTicket getPreviousOutdoorVisit(String customerId, String doctorId, String excludedCutrentId, int inDays) {
		if (StringUtils.isBlank(customerId)) {
			return null;
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OutdoorTicket.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(excludedCutrentId)) {
				criteria.add(Restrictions.ne(OutdoorTicket.PROP_ID, excludedCutrentId));
			}

			criteria.add(Restrictions.eq(OutdoorTicket.PROP_STATUS, OutdoorTicketStatus.PAID.name()));
			criteria.add(Restrictions.eq(OutdoorTicket.PROP_CUSTOMER_ID, customerId));
			if (StringUtils.isNotBlank(doctorId)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_DOCTOR_ID, doctorId));
			}

			Calendar cal = Calendar.getInstance();
			cal.setTime(new Date());
			cal.set(Calendar.DAY_OF_YEAR, -inDays);
			Date cutoffDate = cal.getTime();
			criteria.add(Restrictions.ge(OutdoorTicket.PROP_APPOINTMENT_DATE, cutoffDate));

			criteria.addOrder(Order.desc(OutdoorTicket.PROP_APPOINTMENT_DATE));
			criteria.setMaxResults(1);

			return (OutdoorTicket) criteria.uniqueResult();
		}
	}
}