package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Attribute;
import com.floreantpos.model.AttributeGroup;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class AttributeGroupDAO extends BaseAttributeGroupDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public AttributeGroupDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		AttributeGroup attributeGroup = (AttributeGroup) obj;
		if (attributeGroup == null) {
			throw new PosException(Messages.getString("AttributeGroupDAO.0")); //$NON-NLS-1$
		}
		attributeGroup.setDeleted(true);
		updateAttributeList(attributeGroup);
		update(attributeGroup, session);
	}

	private void updateAttributeList(AttributeGroup attributeGroup) {
		List<Attribute> attributes = attributeGroup.getAttributes();
		if (attributes != null) {
			for (Attribute attribute : attributes) {
				attribute.setDeleted(true);
				AttributeDAO.getInstance().checkForeignRelation(attribute);
			}
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<AttributeGroup> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(Order.asc(AttributeGroup.PROP_NAME));
			this.addDeletedFilter(criteria);
			return this.cleanupDeletedAttribute(criteria.list());
		}
	}

	private List<AttributeGroup> cleanupDeletedAttribute(List<AttributeGroup> attributeGroups) {
		if (attributeGroups != null) {
			attributeGroups.forEach(group -> {
				List<Attribute> attributes = group.getAttributes();
				if (attributes != null) {
					Iterator<Attribute> iterator = attributes.iterator();
					while (iterator.hasNext()) {
						Attribute attribute = iterator.next();
						if (attribute.isDeleted()) {
							iterator.remove();
						}
					}
				}
			});
		}
		return attributeGroups;
	}

	public void saveOrUpdateAttributeGroupList(List<AttributeGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateAttributeGroupList(dataList, updateLastUpdateTime, updateSyncTime, false, false);
	}

	public void saveOrUpdateAttributeGroupList(List<AttributeGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean saveNewDataOnly,
			boolean forceUpdate) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<AttributeGroup> iterator = dataList.iterator(); iterator.hasNext();) {
				AttributeGroup item = (AttributeGroup) iterator.next();
				AttributeGroup existingItem = get(item.getId());
				if (existingItem != null) {
					if (!forceUpdate && (saveNewDataOnly || !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime()))) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

}