/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projection;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.POSConstants;
import com.floreantpos.PosLog;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PackagingUnit;
import com.floreantpos.model.Patient;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.Store;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.TicketSource;
import com.floreantpos.model.User;
import com.floreantpos.model.ext.LabWorkStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.ItemwiseRfReportData;
import com.floreantpos.report.ReportUtil;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class TicketItemDAO extends BaseTicketItemDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public TicketItemDAO() {
	}

	public boolean deleteTicketItemWithTicket(Integer itemID) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);
			criteria.add(Restrictions.eq(TicketItem.PROP_MENU_ITEM_ID, itemID));
			List<TicketItem> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return false;
			}
			for (TicketItem ticketItem : result) {
				ticketItem.setTicket(null);
				super.delete(ticketItem, session);
			}
			return true;
		}
	}

	@SuppressWarnings("unchecked")
	public List<TicketItem> getSalesItems(Date startDate, Date endDate, Terminal terminal) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);
			criteria.createAlias(TicketItem.PROP_TICKET, "t");

			criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, DateUtil.getUTCStartOfDay(startDate)));
			criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, DateUtil.getUTCEndOfDay(endDate)));
			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE));
			if (terminal != null) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_TERMINAL_ID, terminal.getId()));
			}

			return criteria.list();
		}
	}

	public Double getItemsCountWithinHour(Session session, String outletId, Integer hourDuration) {
		try {
			Calendar calendar = Calendar.getInstance();
			Date toDate = calendar.getTime();
			calendar.add(Calendar.HOUR, -(hourDuration == null ? 24 : hourDuration));
			Date fromDate = calendar.getTime();
			Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.between("ticket." + Ticket.PROP_CREATE_DATE, DateUtil.getUTCStartOfDay(fromDate), DateUtil.getUTCEndOfDay(toDate))); //$NON-NLS-1$
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$
			}
			criteria.add(Restrictions.eqOrIsNull(TicketItem.PROP_VOIDED, false));
			ProjectionList projectionList = Projections.projectionList();
			Projection projection = Projections.sum(TicketItem.PROP_QUANTITY);
			projectionList.add(projection);
			criteria.setProjection(projectionList);
			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				return number.doubleValue();
			}
		} catch (Exception e0) {
			PosLog.error(this.getReferenceClass(), e0);
		}
		return 0D;
	}

	public List<TicketItem> findTicketItemWithinDate(Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups, Boolean isInventoryItem) {
		return findTicketItemWithinDate(startDate, endDate, terminal, groups, isInventoryItem, null, Boolean.FALSE);
	}

	public List<TicketItem> findTicketItemWithinDate(Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups, Boolean isInventoryItem,
			Outlet outlet, Boolean isShowOnlineTicket) {
		return findTicketItemWithinDate(startDate, endDate, terminal, groups, isInventoryItem, outlet, isShowOnlineTicket, null);
	}

	public List<TicketItem> findTicketItemWithinDate(Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups, Boolean isInventoryItem,
			Outlet outlet, Boolean isShowOnlineTicket, List<User> users) {
		return findTicketItemWithinDate(startDate, endDate, terminal, groups, isInventoryItem, outlet, isShowOnlineTicket, users, "");
	}

	public List<TicketItem> findTicketItemWithinDate(Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups, Boolean isInventoryItem,
			Outlet outlet, Boolean isShowOnlineTicket, List<User> users, String orderId) {
		try (Session session = this.createNewSession()) {
			List<String> groupIdList = POSUtil.getStringIds(groups, MenuGroup.class);

			Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$

			if (outlet != null) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$
			}

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_ID, orderId)); //$NON-NLS-1$
			}
			else {
				if (startDate != null) {
					criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, startDate));
				}
				if (endDate != null) {
					criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, endDate));
				}

				if (terminal != null) {
					criteria.add(Restrictions.eq("ticket." + Ticket.PROP_TERMINAL_ID, terminal.getId())); //$NON-NLS-1$
				}
				if (!groupIdList.isEmpty()) {
					criteria.add(Restrictions.in(TicketItem.PROP_GROUP_ID, groupIdList));
				}
				if (isInventoryItem != null) {
					criteria.add(Restrictions.eq(TicketItem.PROP_INVENTORY_ITEM, isInventoryItem));
				}

				if (users != null) {
					List<String> allUserIds = new ArrayList<String>();
					for (User user : users) {
						allUserIds.addAll(user.getRoleIds());
					}
					criteria.add(Restrictions.in("ticket." + Ticket.PROP_OWNER_ID, allUserIds)); //$NON-NLS-1$
				}
			}
			if (isShowOnlineTicket) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_SOURCE, TicketSource.Online.name())); //$NON-NLS-1$
			}
			_RootDAO.addSalesAdminCriteria(session, criteria, "ticket." + Ticket.PROP_ID, startDate, endDate, outlet.getId(), null);

			criteria.addOrder(Order.asc(TicketItem.PROP_NAME));
			List<TicketItem> list = criteria.list();
			return list;
		}
	}

	public List<TicketItemModifier> findTicketItemModifierWithinDate(String outletId, Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups,
			Boolean isInventoryItem) {
		try (Session session = this.createNewSession()) {
			List groupIdList = new ArrayList<>();
			if (groups != null) {
				for (MenuGroup menuGroup : groups) {
					groupIdList.add(menuGroup.getId());
				}
			}
			Criteria criteria = session.createCriteria(TicketItemModifier.class).createAlias(TicketItemModifier.PROP_TICKET_ITEM, "ticketItem") //$NON-NLS-1$
					.createAlias("ticketItem." + TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.between("ticket." + Ticket.PROP_CREATE_DATE, startDate, endDate)); //$NON-NLS-1$
			if (isInventoryItem != null) {
				criteria.add(Restrictions.eq("ticketItem." + TicketItem.PROP_INVENTORY_ITEM, isInventoryItem)); //$NON-NLS-1$
			}
			if (!groupIdList.isEmpty()) {
				criteria.add(Restrictions.in("ticketItem." + TicketItem.PROP_GROUP_ID, groupIdList)); //$NON-NLS-1$
			}
			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.property(TicketItemModifier.PROP_ITEM_ID), TicketItemModifier.PROP_ITEM_ID);
			pList.add(Projections.property(TicketItemModifier.PROP_NAME), TicketItemModifier.PROP_NAME);
			pList.add(Projections.property(TicketItemModifier.PROP_MODIFIER_TYPE), TicketItemModifier.PROP_MODIFIER_TYPE);
			pList.add(Projections.property(TicketItemModifier.PROP_UNIT_PRICE), TicketItemModifier.PROP_UNIT_PRICE);
			pList.add(Projections.property(TicketItemModifier.PROP_ITEM_QUANTITY), TicketItemModifier.PROP_ITEM_QUANTITY);
			pList.add(Projections.property(TicketItemModifier.PROP_SERVICE_CHARGE), TicketItemModifier.PROP_SERVICE_CHARGE);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_UNIT_PRICE), TicketItemModifier.PROP_ADJUSTED_UNIT_PRICE);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_DISCOUNT), TicketItemModifier.PROP_ADJUSTED_DISCOUNT);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_SUBTOTAL), TicketItemModifier.PROP_ADJUSTED_SUBTOTAL);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_TOTAL), TicketItemModifier.PROP_ADJUSTED_TOTAL);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_TAX), TicketItemModifier.PROP_ADJUSTED_TAX);
			pList.add(Projections.property(TicketItemModifier.PROP_TAX_INCLUDED), TicketItemModifier.PROP_TAX_INCLUDED);
			pList.add(Projections.property("ticketItem." + TicketItem.PROP_QUANTITY), TicketItemModifier.TRANSIENT_PROP_TICKET_ITEM_QUANTITY); //$NON-NLS-1$
			criteria.setProjection(pList);

			criteria.setResultTransformer(Transformers.aliasToBean(TicketItemModifier.class));
			criteria.addOrder(Order.asc(TicketItemModifier.PROP_NAME));
			List<TicketItemModifier> list = criteria.list();
			return list;
		}
	}

	public void findLabTestHistoryItems(String patientId, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);
			criteria.createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Property.forName("ticket." + Ticket.PROP_CUSTOMER_ID).eq(patientId)); //$NON-NLS-1$
			criteria.add(Restrictions.or(Restrictions.isNull(TicketItem.PROP_PRODUCT_TYPE),
					Restrictions.eq(TicketItem.PROP_PRODUCT_TYPE, ProductType.PATHOLOGY.name()).ignoreCase()));
			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
			criteria.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, LabWorkStatus.APPROVED.name()));

			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.setResultTransformer(Transformers.aliasToBean(TicketItem.class));
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			criteria.addOrder(Order.desc(TicketItem.PROP_CREATE_DATE));
			criteria.addOrder(Order.asc("ticket." + Ticket.PROP_ID));

			tableModel.setRows(criteria.list());
		}
	}

	public List<TicketItem> findTicketItemWithinDateForDashboard(String outletId, Date startDate, Date endDate, Integer numberOfItems, Boolean sortAsc,
			Session session) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
		criteria.add(Restrictions.between(TicketItem.PROP_CREATE_DATE, DateUtil.getUTCStartOfDay(startDate), DateUtil.getUTCEndOfDay(endDate)));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$
		}
		criteria.add(Restrictions.or(Restrictions.isNull(TicketItem.PROP_VOIDED), Restrictions.eq(TicketItem.PROP_VOIDED, false)));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(TicketItem.PROP_NAME), TicketItem.PROP_NAME);
		projectionList.add(Projections.property(TicketItem.PROP_QUANTITY), TicketItem.PROP_QUANTITY);
		criteria.setProjection(projectionList);
		criteria.setMaxResults(numberOfItems == null ? 20 : numberOfItems);
		criteria.setResultTransformer(Transformers.aliasToBean(TicketItem.class));
		if (sortAsc != null) {
			if (sortAsc) {
				criteria.addOrder(Order.asc(TicketItem.PROP_QUANTITY));
			}
			else {
				criteria.addOrder(Order.desc(TicketItem.PROP_QUANTITY));
			}
		}
		return criteria.list();
	}

	public void findLabTestItems(String orderId, String patientId, String sampleID, String itemName, LabWorkStatus labWorkStatus,
			PaginationSupport tableModel) {
		findLabTestItems(orderId, null, sampleID, itemName, patientId, labWorkStatus, tableModel);
	}

	public void findLabTestItems(String orderId, String patientName, String sampleID, String itemName, String patientId, LabWorkStatus labWorkStatus,
			PaginationSupport tableModel) {
		try (Session session = createNewSession()) {

			Criteria criteria = createCriteriaForFindLabTestItems(orderId, patientName, patientId, null, null, null, session);

			if (labWorkStatus != null) {
				if (labWorkStatus == LabWorkStatus.PENDING_SAMPLE) {
					criteria.add(Restrictions.isNull(TicketItem.PROP_KITCHEN_STATUS));
				}
				else {
					criteria.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, labWorkStatus.name()));
				}
			}

			if (StringUtils.isNotBlank(sampleID)) {
				criteria.add(Restrictions.eq(TicketItem.PROP_LAB_TEST_ID, sampleID));
			}

			if (StringUtils.isNotBlank(itemName)) {
				criteria.add(Restrictions.ilike(TicketItem.PROP_NAME, itemName, MatchMode.ANYWHERE));
			}

			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			criteria.addOrder(Order.desc(TicketItem.PROP_CREATE_DATE));
			criteria.addOrder(Order.asc("ticket." + Ticket.PROP_ID));

			tableModel.setRows(criteria.list());
		}
	}

	private Criteria createCriteriaForFindLabTestItems(String orderId, String patientNameOrPhone, String patientId, Customer referrer,
			Doctor labDoctor, Doctor consultant, Session session) {
		
		Criteria criteria = session.createCriteria(TicketItem.class);
		criteria.createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
		
		if (StringUtils.isNotBlank(patientId)) {
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_CUSTOMER_ID, patientId));
		}
		else if (StringUtils.isNotBlank(patientNameOrPhone)) {
			DetachedCriteria patientCriteria = DetachedCriteria.forClass(Patient.class);
			patientCriteria.setProjection(Property.forName(Patient.PROP_ID));
			
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, patientNameOrPhone, MatchMode.ANYWHERE),
					Restrictions.ilike(Customer.PROP_MOBILE_NO, patientNameOrPhone, MatchMode.ANYWHERE)));
			patientCriteria.add(disjunction);

			criteria.add(Property.forName("ticket." + Ticket.PROP_CUSTOMER_ID).in(patientCriteria)); //$NON-NLS-1$
		}

		criteria.add(Restrictions.or(Restrictions.isNull(TicketItem.PROP_PRODUCT_TYPE),
				Restrictions.eq(TicketItem.PROP_PRODUCT_TYPE, ProductType.PATHOLOGY.name()).ignoreCase()));

		criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$

		if (StringUtils.isNotBlank(orderId)) {
			criteria.add(Restrictions.ilike("ticket." + Ticket.PROP_ID, orderId, MatchMode.END)); //$NON-NLS-1$
		}

		if (referrer != null) {
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_REFERRER_ID, referrer.getId()));
		}

		if (labDoctor != null) {
			criteria.add(Restrictions.eq(TicketItem.PROP_LAB_DOCTOR_ID, labDoctor.getId()));
		}

		if (consultant != null) {
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_DOCTOR_ID, consultant.getId()));
		}

		criteria.add(Restrictions.or(Restrictions.isNull(TicketItem.PROP_RETURNED_SOURCE), Restrictions.eq(TicketItem.PROP_RETURNED_SOURCE, Boolean.FALSE)));
		criteria.add(Restrictions.ne(TicketItem.PROP_ITEM_RETURNED, Boolean.TRUE));
		criteria.add(Restrictions.ne(TicketItem.PROP_VOIDED, Boolean.TRUE));
		return criteria;
	}

	public void findLabTestItemsForReportDelivery(String orderId, String patientNameOrPhone, String patientId, LabWorkStatus labWorkStatus,
			PaginationSupport tableModel) {
		findLabTestItemsForReportDelivery(orderId, patientNameOrPhone, patientId, labWorkStatus, tableModel, null);
	}

	public void findLabTestItemsForReportDelivery(String orderId, String patientNameOrPhone, String patientId, LabWorkStatus labWorkStatus,
			PaginationSupport tableModel, Customer referrer) {
		findLabTestItemsForReportDelivery(orderId, patientNameOrPhone, patientId, labWorkStatus, tableModel, referrer, null, null);
	}

	public void findLabTestItemsForReportDelivery(String orderId, String patientNameOrPhone, String patientId, LabWorkStatus labWorkStatus,
			PaginationSupport tableModel, Customer referrer, Doctor labDoctor, Doctor consultant) {
		try (Session session = createNewSession()) {

			Criteria criteria = createCriteriaForFindLabTestItems(orderId, patientNameOrPhone, patientId, referrer, labDoctor, consultant, session);

			if (labWorkStatus != null) {
				if (labWorkStatus == LabWorkStatus.ONGOINING) {
					Disjunction disjunction = Restrictions.disjunction();
					disjunction.add(Restrictions.isNull(TicketItem.PROP_KITCHEN_STATUS));
					disjunction.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, LabWorkStatus.PENDING_SAMPLE.name()));
					disjunction.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, LabWorkStatus.SAMPLE_COLLECTED.name()));
					disjunction.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, LabWorkStatus.RUN_TEST.name()));
					disjunction.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, LabWorkStatus.RESULT_RECORDED.name()));
					criteria.add(disjunction);

				}
				else if (labWorkStatus == LabWorkStatus.READY) {
					criteria.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, LabWorkStatus.APPROVED.name()));
					criteria.add(Restrictions.ne(TicketItem.PROP_REPORT_DELIVERED, true));
				}
				else if (labWorkStatus == LabWorkStatus.DELIVERED) {
					criteria.add(Restrictions.eq(TicketItem.PROP_REPORT_DELIVERED, true));
				}
				else if (labWorkStatus == LabWorkStatus.APPROVED) {
					criteria.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, labWorkStatus.name()));
				}
			}

			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			criteria.addOrder(Order.desc(TicketItem.PROP_CREATE_DATE));
			criteria.addOrder(Order.asc("ticket." + Ticket.PROP_ID));

			tableModel.setRows(criteria.list());
		}
	}

	public List<TicketItem> findVoidedOrReturnTicketItems(Date startDate, Date endDate, Outlet outlet) {
		try (Session session = this.createNewSession()) {

			Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.ge(TicketItem.PROP_VOID_DATE, startDate));
			criteria.add(Restrictions.lt(TicketItem.PROP_VOID_DATE, endDate));
			criteria.add(Restrictions.or(Restrictions.eq(TicketItem.PROP_VOIDED, true), Restrictions.eq(TicketItem.PROP_ITEM_RETURNED, true)));
			if (outlet != null) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$
			}

			criteria.addOrder(Order.desc(TicketItem.PROP_VOID_DATE));
			criteria.addOrder(Order.asc(TicketItem.PROP_NAME));
			List<TicketItem> list = criteria.list();
			return list;
		}
	}

	public List<TicketItem> findTicketItems(Date startDate, Date endDate, Boolean voided, Boolean returned, Outlet outlet, User user) {
		try (Session session = this.createNewSession()) {

			Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, endDate));
			if (voided != null) {
				criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, voided));
			}
			if (returned != null) {
				criteria.add(Restrictions.eq(TicketItem.PROP_ITEM_RETURNED, returned));
			}
			//criteria.add(Restrictions.eq("ticket." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
			if (outlet != null) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$
			}

			if (user != null) {
				criteria.add(Restrictions.in("ticket." + Ticket.PROP_OWNER_ID, user.getRoleIds())); //$NON-NLS-1$
			}

			criteria.addOrder(Order.asc(TicketItem.PROP_NAME));
			List<TicketItem> list = criteria.list();
			return list;
		}
	}

	public List<TicketItem> findTopItems(int itemSize) {
		try (Session session = createNewSession()) {
			Calendar calendar = Calendar.getInstance();
			Date endDate = calendar.getTime(); //DateUtil.convertServerTimeToBrowserTime(calendar.getTime());
			calendar.add(Calendar.MONTH, -1);
			Date startDate = calendar.getTime(); //DateUtil.convertServerTimeToBrowserTime(calendar.getTime());
			return findTopItems(session, null, startDate, endDate, itemSize, Arrays.asList(ProductType.PATHOLOGY));
		}
	}

	public List<TicketItem> findTopItems(Session session, String outletId, Date startDate, Date endDate, int itemCount) {
		return findTopItems(session, outletId, startDate, endDate, itemCount, TicketItem.PROP_QUANTITY, null);
	}

	public List<TicketItem> findTopItems(Session session, String outletId, Date startDate, Date endDate, int itemCount, List<ProductType> productType) {
		return findTopItems(session, outletId, startDate, endDate, itemCount, TicketItem.PROP_QUANTITY, productType);
	}

	public List<TicketItem> findTopItems(Session session, String outletId, Date startDate, Date endDate, int itemCount, String property,
			List<ProductType> productTypes) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
		criteria.add(Restrictions.between(TicketItem.PROP_CREATE_DATE, DateUtil.getUTCStartOfDay(startDate), DateUtil.getUTCEndOfDay(endDate)));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$
		}
		criteria.add(Restrictions.or(Restrictions.isNull(TicketItem.PROP_VOIDED), Restrictions.eq(TicketItem.PROP_VOIDED, false)));
		criteria.add(Restrictions.eq("ticket." + Ticket.PROP_VOIDED, Boolean.FALSE));//$NON-NLS-1$

		if (productTypes != null) {
			Disjunction disjunction = Restrictions.disjunction();
			for (ProductType productType : productTypes) {
				disjunction.add(Restrictions.eq(TicketItem.PROP_PRODUCT_TYPE, productType.name()));
			}
			criteria.add(disjunction);
		}

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(TicketItem.PROP_NAME), TicketItem.PROP_NAME);
		projectionList.add(Projections.property(TicketItem.PROP_MENU_ITEM_ID), TicketItem.PROP_MENU_ITEM_ID);
		projectionList.add(Projections.sum(property), property);
		projectionList.add(Projections.groupProperty(TicketItem.PROP_NAME));
		projectionList.add(Projections.groupProperty(TicketItem.PROP_MENU_ITEM_ID));
		criteria.setProjection(projectionList);

		criteria.setMaxResults(itemCount);
		criteria.setResultTransformer(Transformers.aliasToBean(TicketItem.class));
		criteria.addOrder(Order.desc(property));
		criteria.addOrder(Order.asc(TicketItem.PROP_NAME));
		return criteria.list();
	}

	public TicketItem findByLabTest(String labTestId) {
		try (Session session = this.createNewSession()) {
			return findByLabTest(labTestId, session);
		}
	}

	public TicketItem findByLabTest(String labTestId, Session session) {
		Criteria criteria = session.createCriteria(TicketItem.class);
		criteria.add(Restrictions.eq(TicketItem.PROP_LAB_TEST_ID, labTestId.trim())); //$NON-NLS-1$
		criteria.setMaxResults(1);
		return (TicketItem) criteria.uniqueResult();
	}

	public boolean hasDifferentLabTestByTestId(String ticketItemId, String labTestId) {
		try (Session session = this.createNewSession()) {
			return hasLabTestById(ticketItemId, labTestId, session);
		}
	}

	public boolean hasLabTestById(String labTestId) {
		try (Session session = this.createNewSession()) {
			return hasLabTestById(labTestId, session);
		}
	}

	public boolean hasLabTestById(String labTestId, Session session) {
		return hasLabTestById(null, labTestId, session);
	}

	public boolean hasLabTestById(String ticketItemId, String labTestId, Session session) {
		Criteria criteria = session.createCriteria(TicketItem.class);
		criteria.add(Restrictions.eq(TicketItem.PROP_LAB_TEST_ID, labTestId.trim())); //$NON-NLS-1$

		if (StringUtils.isNotEmpty(ticketItemId)) {
			criteria.add(Restrictions.ne(TicketItem.PROP_ID, ticketItemId)); //$NON-NLS-1$
		}

		int rowCount = rowCount(criteria);
		return rowCount > 0;
	}

	public List<TicketItem> findLabDoctorFeeOrders(String outletId, Date beginDate, Date toDate, Doctor labDoctor) {
		return findLabDoctorFeeOrders(outletId, beginDate, toDate, labDoctor, null);
	}

	public List<TicketItem> findLabDoctorFeeOrders(String outletId, Date beginDate, Date toDate, Doctor labDoctor, String orderId) {
		try (Session session = createNewSession()) {
			Criteria criteria = createLabDoctorFeeSearchCriteria(outletId, beginDate, toDate, labDoctor, false, orderId, session);
			criteria.addOrder(Order.desc(TicketItem.PROP_CREATE_DATE));
			return criteria.list();
		}
	}

	public double getUnpaidLabDoctorFee(String outletId, Date beginDate, Date toDate, Customer labDoctor) {
		return getUnpaidLabDoctorFee(outletId, beginDate, toDate, labDoctor, "");
	}

	public double getUnpaidLabDoctorFee(String outletId, Date beginDate, Date toDate, Customer labDoctor, String orderId) {
		try (Session session = createNewSession()) {
			Criteria criteria = createLabDoctorFeeSearchCriteria(outletId, beginDate, toDate, labDoctor, true, orderId, session);

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(TicketItem.PROP_LAB_DOCTOR_FEE));
			projectionList.add(Projections.sum(TicketItem.PROP_LAB_DOCTOR_FEE_PAID_AMOUNT));
			criteria.setProjection(projectionList);

			Object[] objects = (Object[]) criteria.uniqueResult();
			if (objects[0] == null) {
				return 0;
			}
			double payableLabDoctorFee = (objects[0] != null) ? ((Number) objects[0]).doubleValue() : 0;
			double paidLabDoctorFee = (objects[1] != null) ? ((Number) objects[1]).doubleValue() : 0;

			return NumberUtil.round(payableLabDoctorFee - paidLabDoctorFee);
		}
	}

	public List<TicketItem> getLabDoctorFeeUnpaidTicketItems(String outletId, Date beginDate, Date toDate, Customer labDoctor) {
		return getLabDoctorFeeUnpaidTicketItems(outletId, beginDate, toDate, labDoctor, "");
	}

	public List<TicketItem> getLabDoctorFeeUnpaidTicketItems(String outletId, Date beginDate, Date toDate, Customer labDoctor, String orderId) {
		try (Session session = createNewSession()) {
			Criteria criteria = createLabDoctorFeeSearchCriteria(outletId, beginDate, toDate, labDoctor, true, orderId, session);
			criteria.addOrder(Order.asc(TicketItem.PROP_CREATE_DATE));
			List<TicketItem> list = criteria.list();
			return list;
		}
	}

	private Criteria createLabDoctorFeeSearchCriteria(String outletId, Date beginDate, Date toDate, Customer labDoctor, boolean filterPaidOrders,
			String orderId, Session session) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "t");

		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq("t." + Ticket.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$
		}
		if (StringUtils.isNotBlank(orderId)) {
			criteria.add(Restrictions.eq("t." + Ticket.PROP_ID, orderId));
		}
		else {
			if (labDoctor != null) {
				criteria.add(Restrictions.eq(TicketItem.PROP_LAB_DOCTOR_ID, labDoctor.getId()));
			}

			if (filterPaidOrders) {
				criteria.add(Restrictions.eq(TicketItem.PROP_LAB_DOCTOR_FEE_PAID, false));
			}
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, LabWorkStatus.APPROVED.name()));
			disjunction.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, LabWorkStatus.DISPATCHED.name()));
			disjunction.add(Restrictions.eq(TicketItem.PROP_ALLOW_DR_FEE_BEFORE_LABWORK, true));
			criteria.add(disjunction);

			if (beginDate != null) {
				criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, beginDate));
			}
			if (toDate != null) {
				criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, toDate));
			}
		}

		criteria.add(Restrictions.or(Restrictions.isNull(TicketItem.PROP_RETURNED_SOURCE), Restrictions.eq(TicketItem.PROP_RETURNED_SOURCE, Boolean.FALSE)));
		criteria.add(Restrictions.eq(TicketItem.PROP_ITEM_RETURNED, Boolean.FALSE));
		criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
		criteria.add(Restrictions.isNotNull(TicketItem.PROP_LAB_DOCTOR_ID));

		criteria.add(Restrictions.gt(TicketItem.PROP_LAB_DOCTOR_FEE, Double.valueOf(0)));
		PosLog.debug(getClass(), "Lab doctor fee search criteria query: " + criteria.toString()); //$NON-NLS-1$
		return criteria;
	}

	public void adjustInventory(TicketItem ticketItem, InventoryTransactionType transactionType, String reason, InventoryLocation location) throws Exception {

		Transaction tx = null;

		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			adjustInventory(ticketItem, transactionType, reason, location, session);

			session.saveOrUpdate(ticketItem);

			tx.commit();
		}
	}

	public void adjustInventory(TicketItem ticketItem, InventoryTransactionType transactionType, String reason, InventoryLocation location, Session session)
			throws Exception {
		Store store = DataProvider.get().getStore();
		boolean isUpdateOnHandBlncForSale = store.isUpdateOnHandBlncForSale();
		boolean isUpdateAvailBlncForSale = store.isUpdateAvlBlncForSale();
		Ticket ticket = ticketItem.getTicket();
		Double unitQuantity = ticketItem.getQuantity();

		MenuItem menuItem = MenuItemDAO.getInstance().getMenuItemWithFields(session, ticketItem.getMenuItemId(), MenuItem.PROP_NAME, MenuItem.PROP_PRICE,
				MenuItem.PROP_SKU, MenuItem.PROP_BARCODE, MenuItem.PROP_UNIT_ID, MenuItem.PROP_COST, MenuItem.PROP_AVERAGE_UNIT_PURCHASE_PRICE,
				MenuItem.PROP_AVG_COST, MenuItem.PROP_DEFAULT_RECIPE_ID, MenuItem.PROP_INVENTORY_ITEM);
		if (menuItem == null) {
			return;
		}
		if (menuItem.isInventoryItem()) {
			InventoryTransaction invTrans = new InventoryTransaction();
			if (ticketItem.isItemReturned() && InventoryTransaction.REASON_VOID.equals(reason)) {
				reason = InventoryTransaction.REASON_RETURN;
			}
			if (transactionType == InventoryTransactionType.IN) {
				invTrans.setToInventoryLocation(location);
			}
			else {
				invTrans.setFromInventoryLocation(location);
			}
			invTrans.setReason(reason);
			invTrans.setTransactionDate(new Date());
			invTrans.setMenuItem(menuItem);
			invTrans.setType(transactionType.getType());

			invTrans.setTicketId(ticket.getId());
			invTrans.setUser(ticket.getOwner());

			IUnit transactionUnit = (IUnit) DataProvider.get().getInventoryUnitById(ticketItem.getUnitName());
			if (transactionUnit instanceof PackagingUnit) {
				session.refresh(menuItem);
				if (!Hibernate.isInitialized(menuItem.getStockUnits())) {
					Hibernate.initialize(menuItem.getStockUnits());
				}
			}
			invTrans.setDisplayUnit(transactionUnit, unitQuantity, menuItem);
			//				double baseUnitQuantity = menuItem.getBaseUnitQuantity(mapKey.getUnitCode());
			//				outTrans.setUnitPrice(menuItem.getPrice() * baseUnitQuantity);
			if (ticketItem != null) {
				invTrans.setUnitPrice(ticketItem.getAdjustedUnitPrice());
			}
			invTrans.setQuantity(unitQuantity);
			invTrans.setUnit(ticketItem.getUnitName());
			invTrans.setUnitType(ticketItem.getUnitType());
			invTrans.setUnitCost(menuItem.getCost());
			invTrans.calculateTotal();
			InventoryTransactionDAO.getInstance().adjustInventoryStock(invTrans, session, isUpdateAvailBlncForSale, isUpdateOnHandBlncForSale, false);
			ActionHistoryDAO.logJournalForInventoryTrans(invTrans);
		}
		//adjust inventory of recipe items
		if (StringUtils.isNotEmpty(menuItem.getDefaultRecipeId())) {
			Recepie recepie = RecepieDAO.getInstance().get(menuItem.getDefaultRecipeId(), session);
			RecepieDAO.getInstance().adjustRecipeItemsFromInventory(unitQuantity, Arrays.asList(recepie), session, ticketItem);
		}
	}

	public List<ItemwiseRfReportData> findItemWiseRFReportOrders(String outletId, Date beginDate, Date currentDate, String orderId) {
		List<ItemwiseRfReportData> datas = new ArrayList<ItemwiseRfReportData>();
		try (Session session = createNewSession()) {
			Criteria criteria = buildItemCommonCriteria(session, orderId, beginDate, currentDate, outletId, null);

			if (StringUtils.isBlank(orderId)) {
				//criteria.add(Restrictions.isNotNull("t." + Ticket.PROP_REFERRER_ID));
			}

			criteria.addOrder(Order.asc("t." + Ticket.PROP_ID));
			criteria.addOrder(getDefaultOrder());

			List<TicketItem> list = criteria.list();

			double totalItemPrice = 0;
			double totalItemDiscount = 0;
			double totalItemTax = 0;
			double totalRFAmount = 0;
			double totalRFOnNetSalesAmount = 0;
			double totalNonRFAmount = 0;
			double totalLDFee = 0;
			double totalNetSales = 0;

			double grandTotalItemPrice = 0;
			double grandTotalItemDiscount = 0;
			double grandTotalItemTax = 0;
			double grandTotalRFAmount = 0;
			double grandTotalRFOnNetSalesAmount = 0;
			double grandTotalNonRFAmount = 0;
			double grandTotalLDFee = 0;
			double grandTotalNetSales = 0;

			String previousTicketId = "";

			boolean firstItem = true;
			boolean hasRfItemInTicket = false;

			if (list != null && list.size() > 0) {
				for (TicketItem ticketItem : list) {
					if (ticketItem == null) {
						continue;
					}

					Ticket ticket = ticketItem.getTicket();
					String ticketId = ticket.getId();

					if (!previousTicketId.equals(ticketId)) {
						if (StringUtils.isNotBlank(previousTicketId) && hasRfItemInTicket) {
							datas.add(addTotalRow(totalItemPrice, totalItemDiscount, totalItemTax, totalRFAmount, totalRFOnNetSalesAmount, totalNonRFAmount,
									totalLDFee, totalNetSales));
						}

						totalItemPrice = 0;
						totalItemDiscount = 0;
						totalItemTax = 0;
						totalRFAmount = 0;
						totalRFOnNetSalesAmount = 0;
						totalNonRFAmount = 0;
						totalLDFee = 0;
						totalNetSales = 0;

						firstItem = true;
						hasRfItemInTicket = false;
						previousTicketId = ticketId;
					}

					PosLog.debug(getClass(), "Processing ticket: " + ticketId + ", item Id: " + ticketItem.getId());

					if (ticketItem.isRFCalculationApplicable()) {
						ItemwiseRfReportData ticketItemData = new ItemwiseRfReportData();
						ticketItemData.populateTicketItemData(ticketItem);
						if (firstItem) {
							ticketItemData.setOrderId(ticket.getId());
							firstItem = false;
						}
						datas.add(ticketItemData);

						totalItemPrice += ticketItemData.getItemPrice();
						totalItemDiscount += ticketItemData.getItemDiscount();
						totalItemTax += ticketItemData.getItemTax();
						totalRFAmount += ticketItemData.getRf();
						totalRFOnNetSalesAmount += ticketItemData.getRfOnNetSales();
						totalNonRFAmount += ticketItemData.getNonRf();
						totalLDFee += ticketItemData.getLabDoctorFee();
						totalNetSales += ticketItemData.getNet();

						grandTotalItemPrice += ticketItemData.getItemPrice();
						grandTotalItemDiscount += ticketItemData.getItemDiscount();
						grandTotalItemTax += ticketItemData.getItemTax();
						grandTotalRFAmount += ticketItemData.getRf();
						grandTotalRFOnNetSalesAmount += ticketItemData.getRfOnNetSales();
						grandTotalNonRFAmount += ticketItemData.getNonRf();
						grandTotalLDFee += ticketItemData.getLabDoctorFee();
						grandTotalNetSales += ticketItemData.getNet();

						hasRfItemInTicket = true;
					}
				}

				if (hasRfItemInTicket) {
					datas.add(addTotalRow(totalItemPrice, totalItemDiscount, totalItemTax, totalRFAmount, totalRFOnNetSalesAmount, totalNonRFAmount, totalLDFee,
							totalNetSales));
				}
			}

			ItemwiseRfReportData grandTotalData = new ItemwiseRfReportData();
			grandTotalData.setItemName(ReportUtil.getBoldText(POSConstants.GRAND_TOTAL));
			grandTotalData.setItemPrice(grandTotalItemPrice);
			grandTotalData.setItemDiscount(grandTotalItemDiscount);
			grandTotalData.setItemTax(grandTotalItemTax);
			grandTotalData.setRf(Math.round(grandTotalRFAmount));
			grandTotalData.setRfOnNetSales(Math.round(grandTotalRFOnNetSalesAmount));
			grandTotalData.setNonRf(Math.round(grandTotalNonRFAmount));
			grandTotalData.setNet(Math.round(grandTotalNetSales));
			grandTotalData.setLabDoctorFee(grandTotalLDFee);
			datas.add(grandTotalData);
		}
		return datas;
	}

	private ItemwiseRfReportData addTotalRow(double totalItemPrice, double totalItemDiscount, double totalItemTax, double totalRFAmount,
			double totalRFOnNetSalesAmount, double totalNonRFAmount, double totalLDFee, double totalNetSales) {
		ItemwiseRfReportData ticketData = new ItemwiseRfReportData();
		ticketData.setItemName(ReportUtil.getBoldText(POSConstants.TOTAL));
		ticketData.setItemPrice(totalItemPrice);
		ticketData.setItemDiscount(totalItemDiscount);
		ticketData.setItemTax(totalItemTax);
		ticketData.setRf(Math.round(totalRFAmount));
		ticketData.setRfOnNetSales(Math.round(totalRFOnNetSalesAmount));
		ticketData.setNonRf(Math.round(totalNonRFAmount));
		ticketData.setNet(Math.round(totalNetSales));
		ticketData.setLabDoctorFee(totalLDFee);
		return ticketData;
	}

	public Criteria buildItemCommonCriteria(Session session, String orderId, Date fromDate, Date toDate, String outletId, Boolean itemReturned) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "t"); //$NON-NLS-1$

		criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));

		criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$

		if (StringUtils.isNotBlank(orderId)) {
			criteria.add(Restrictions.eq("t." + Ticket.PROP_ID, orderId));
		}
		else {
			if (fromDate != null) {
				criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, fromDate));
			}
			if (toDate != null) {
				criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, toDate));
			}

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_OUTLET_ID, outletId));
			}

			if (itemReturned != null) {
				criteria.add(Restrictions.eq(TicketItem.PROP_ITEM_RETURNED, itemReturned));
			}
		}

		return criteria;
	}

	public boolean hasTicketItemBySerialNo(String serialNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(TicketItem.PROP_SERIAL_NO, serialNo));
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue() > 0;
			}
		}
		return false;
	}

}