/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;

import com.floreantpos.PosLog;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.util.NumberUtil;

public class UpdateDBTo413 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo413(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void updateUserPropertyToField() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		int totalUsers = rowCount();
		PosLog.info(UpdateDBTo413.class, "Found total users: " + totalUsers); //$NON-NLS-1$

		int increment = 10;
		int count = 0;
		while (count < totalUsers) {
			try (Session session = sessionFactory.openSession()) {
				Transaction transaction = session.beginTransaction();

				Pagination<User> pagination = new Pagination<User>(count, increment);
				pagination.setNumRows(totalUsers);
				findUsers(pagination, session);

				List<User> users = pagination.getDataList();
				for (User user : users) {
					String monthlySalaryString = user.getProperty("user.monthly_salary_amount"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(monthlySalaryString)) {
						PosLog.info(getClass(), "User: " + user.getFullName() + ", updated monthly salary from property to field: " + monthlySalaryString); //$NON-NLS-1$ //$NON-NLS-2$
						double monthlySalary = NumberUtil.parseOrGetZero(monthlySalaryString).doubleValue();
						user.setMonthlySalaryAmount(monthlySalary);
					}

					String amAmountString = user.getProperty("user.accounts_manager_amount"); //$NON-NLS-1$//$NON-NLS-2$
					if (StringUtils.isNotBlank(amAmountString)) {
						PosLog.info(getClass(), "User: " + user.getFullName() + ", updated accounts manager amount from property to field: " + amAmountString); //$NON-NLS-1$ //$NON-NLS-2$
						double amAmount = NumberUtil.parseOrGetZero(amAmountString).doubleValue();
						user.setAccountsManagerAmount(amAmount);
					}

					UserDAO.getInstance().saveOrUpdate(user, session);
				}
				transaction.commit();

				count += increment;

			}
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

	private void findUsers(Pagination<User> pagination, Session session) {
		Criteria criteria = createUserCriteria(session);
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());
		pagination.setRows(criteria.list());

	}

	public int rowCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createUserCriteria(session);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createUserCriteria(Session session) {
		Criteria criteria = session.createCriteria(User.class);
		return criteria;
	}
}