/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.util.List;

import org.apache.commons.lang.SerializationUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemCookingInstruction;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.ext.KitchenStatus;

public class KitchenTicketItemDAO extends BaseKitchenTicketItemDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public KitchenTicketItemDAO() {
	}

	public List<KitchenTicketItem> find(String itemId, boolean modifier) {
		Session session = null;
		try {
			session = getSession();
			return find(itemId, modifier, session);
		} finally {
			closeSession(session);
		}
	}

	public List<KitchenTicketItem> find(String itemId, boolean modifier, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(modifier ? KitchenTicketItem.PROP_TICKET_ITEM_MODIFIER_ID : KitchenTicketItem.PROP_TICKET_ITEM_ID, itemId));
		return criteria.list();
	}

	public void markVoided(String itemId, Double quantity, Session session) {
		Criteria criteria = session.createCriteria(KitchenTicketItem.class);
		criteria.add(Restrictions.eq(KitchenTicketItem.PROP_TICKET_ITEM_ID, itemId));
		criteria.add(Restrictions.eq(KitchenTicketItem.PROP_VOIDED, Boolean.FALSE));

		List list = criteria.list();
		if (list.isEmpty()) {
			return;
		}
		KitchenTicketItem kitchenTicketItem = (KitchenTicketItem) list.get(0);
		KitchenTicket kitchenTicket = kitchenTicketItem.getKitchenTicket();

		kitchenTicketItem.setVoided(true);
		KitchenTicketItemDAO.getInstance().update(kitchenTicketItem, session);

		if (kitchenTicketItem.getQuantity().compareTo(Math.abs(quantity)) > 0) {
			KitchenTicketItem newKitchenTicketItem = (KitchenTicketItem) SerializationUtils.clone(kitchenTicketItem);
			newKitchenTicketItem.setId(null);
			TicketItem ticketItem = TicketItemDAO.getInstance().get(newKitchenTicketItem.getTicketItemId());
			if (ticketItem != null) {
				ticketItem.setQuantity(newKitchenTicketItem.getQuantity() - Math.abs(quantity));
				newKitchenTicketItem.setMenuItemName(ticketItem.getNameDisplay());

				String nameDisplay = newKitchenTicketItem.getMenuItemName();
				List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
				if (ticketItemModifiers != null) {
					for (TicketItemModifier ticketItemModifier : ticketItem.getTicketItemModifiers()) {
						nameDisplay = nameDisplay.concat("\n" + ticketItemModifier.getNameDisplay(true)); //$NON-NLS-1$
					}
				}
				List<TicketItemCookingInstruction> cookingInstructions = ticketItem.getCookingInstructions();
				if (cookingInstructions != null) {
					for (TicketItemCookingInstruction ticketItemCookingInstruction : cookingInstructions) {
						nameDisplay = nameDisplay.concat("\n" + ticketItemCookingInstruction.getNameDisplay()); //$NON-NLS-1$
					}
				}
				newKitchenTicketItem.setMenuItemName(nameDisplay);
			}
			else {
				newKitchenTicketItem.setMenuItemName(newKitchenTicketItem.getQuantity() + " " + newKitchenTicketItem.getMenuItemNameDisplay()); //$NON-NLS-1$
			}
			newKitchenTicketItem.setQuantity(newKitchenTicketItem.getQuantity() - Math.abs(quantity));
			newKitchenTicketItem.setVoided(false);
			newKitchenTicketItem.setVoidedItemId(itemId);

			kitchenTicket.addToticketItems(newKitchenTicketItem);
			KitchenTicketDAO.getInstance().update(kitchenTicket, session);
		}
	}

	@SuppressWarnings("unchecked")
	public void bumpTicketItem(KitchenTicketItem kitchenTicketItem) {
		Transaction tx = null;

		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			Criteria criteria = session.createCriteria(KitchenTicketItem.class);
			criteria.add(Restrictions.eq(KitchenTicketItem.PROP_TICKET_ITEM_ID, kitchenTicketItem.getTicketItemId()));
			List<KitchenTicketItem> list = criteria.list();
			for (KitchenTicketItem kitchenTicketItem2 : list) {
				kitchenTicketItem2.setStatus(KitchenStatus.BUMP.name());
				update(kitchenTicketItem2, session);
			}

			criteria = session.createCriteria(TicketItem.class);
			criteria.add(Restrictions.eq(TicketItem.PROP_ID, kitchenTicketItem.getTicketItemId()));
			List<TicketItem> list2 = criteria.list();
			for (TicketItem ticketItem : list2) {
				ticketItem.setKitchenStatusValueWithChildren(KitchenStatus.BUMP);
				TicketItemDAO.getInstance().update(ticketItem, session);
			}
			kitchenTicketItem.setStatus(KitchenStatus.BUMP.name());

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}
}