/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceSubType;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.TransactionType;

public class UpdateDBTo382 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo382(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() throws Exception {
		PosLog.info(getClass(), "Updating " + schemaName + " schema.");

		try (Session session = sessionFactory.openSession()) {

			List<String> oldEmumTypesName = new ArrayList<String>();
			oldEmumTypesName.add(0, "BEGIN_CASH_DEPOSIT");
			oldEmumTypesName.add(1, "ACCOUNTS_MANAGER_PAYMENT");
			oldEmumTypesName.add(2, "DEPOSIT_TO_STORE");
			oldEmumTypesName.add(3, "RESELLER_PAYMENT");

			List<BalanceUpdateTransaction> transactions = findOldEnumTypeTransaction(session, oldEmumTypesName);
			if (transactions.size() > 0) {
				PosLog.info(getClass(), "Updating " + transactions.size() + " transactions.");

				Transaction tx = session.beginTransaction();

				for (Iterator<BalanceUpdateTransaction> iterator = transactions.iterator(); iterator.hasNext();) {
					BalanceUpdateTransaction transaction = (BalanceUpdateTransaction) iterator.next();

					String balanceTypeString = transaction.getBalanceTypeString();
					if (balanceTypeString.equals(oldEmumTypesName.get(0))) {
						transaction.setBalanceType(BalanceType.CASH_DRAWER_BALANCE);

						transaction.setTransactionSubType(BalanceSubType.BEGIN_CASH_DEPOSIT.name());

					}
					else if (balanceTypeString.equals(oldEmumTypesName.get(1))) {
						transaction.setBalanceType(BalanceType.ACCOUNTS_MANAGER_BALANCE);

						if (TransactionType.DEBIT.name().equals(transaction.getTransactionType())) {
							if (transaction.getDescription().equals("Transfer balance")) {
								transaction.setTransactionSubType(BalanceSubType.TRANSFER_OUT.name());
							}
							else {
								transaction.setTransactionSubType(BalanceSubType.EXPENSE.name());
							}
						}
						else {
							transaction.setTransactionSubType(BalanceSubType.ADD_BALANCE.name());
						}

					}
					else if (balanceTypeString.equals(oldEmumTypesName.get(2))) {
						transaction.setBalanceType(BalanceType.STORE_BALANCE);

						if (TransactionType.DEBIT.name().equals(transaction.getTransactionType())) {
							transaction.setTransactionSubType(BalanceSubType.WITHDRAW.name());
						}
						else {
							transaction.setTransactionSubType(BalanceSubType.TRANSFER_IN.name());
						}
					}
					else if (balanceTypeString.equals(oldEmumTypesName.get(3))) {
						transaction.setBalanceType(BalanceType.RESELLER_BALANCE);
					}
					else {
						PosLog.info(getClass(), "Updating " + balanceTypeString + " ignored.");
						continue;
					}

					session.saveOrUpdate(transaction);
				}
				tx.commit();
			}
		}

	}

	private List<BalanceUpdateTransaction> findOldEnumTypeTransaction(Session session, List<String> oldEnumTypesName) {
		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

		criteria.add(Restrictions.in(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, oldEnumTypesName));
		criteria.add(Restrictions.isNull(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE));

		return criteria.list();
	}

}