package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.OnlineCategory;
import com.floreantpos.model.OnlineItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.util.MenuSearchCriteria;
import com.floreantpos.swing.PaginationSupport;

public class OnlineItemDAO extends BaseOnlineItemDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OnlineItemDAO() {
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	public void saveOrUpdateOnlineItems(List<MenuItem> menuItems, Outlet outlet) {
		if (menuItems == null || menuItems.isEmpty()) {
			return;
		}
		List<String> ids = new ArrayList<>();
		for (MenuItem menuItem : menuItems) {
			ids.add(menuItem.getId());
		}
		Map<String, OnlineItem> existingDataMap = new HashMap<>();
		Map<String, OnlineCategory> existingCategoryMap = new HashMap<>();
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.in(OnlineItem.PROP_MENU_ITEM_ID, ids));
			criteria.add(Restrictions.eq(OnlineItem.PROP_OUTLET_ID, outlet.getId()));
			List list = criteria.list();
			if (list != null && list.size() > 0) {
				for (Iterator<OnlineItem> iterator = list.iterator(); iterator.hasNext();) {
					OnlineItem onlineItem = (OnlineItem) iterator.next();
					existingDataMap.put(onlineItem.getMenuItemId(), onlineItem);
				}
			}
			Transaction tx = session.beginTransaction();
			for (MenuItem menuItem : menuItems) {
				OnlineItem onlineItem = existingDataMap.get(menuItem.getId());
				if (onlineItem == null) {
					onlineItem = new OnlineItem();
					onlineItem.setMenuItemId(menuItem.getId());
					onlineItem.setActive(true);
				}
				onlineItem.setOutletId(outlet.getId());
				onlineItem.setName(menuItem.getName());
				saveOrUpdate(onlineItem, session);

				String menuCategoryId = menuItem.getMenuCategoryId();
				if (menuCategoryId == null) {
					continue;
				}
				OnlineCategory onlineCategory = existingCategoryMap.get(menuCategoryId);
				if (onlineCategory != null) {
					continue;
				}
				onlineCategory = OnlineCategoryDAO.getInstance().findByCategoryId(menuCategoryId, outlet, session);
				if (onlineCategory == null) {
					onlineCategory = new OnlineCategory();
					onlineCategory.setCategoryId(menuCategoryId);
					onlineCategory.setOutletId(outlet.getId());
					onlineCategory.setName(menuItem.getMenuCategoryName());
					OnlineCategoryDAO.getInstance().save(onlineCategory, session);
				}
				existingCategoryMap.put(menuCategoryId, onlineCategory);
			}
			tx.commit();
		}
	}

	public void saveOnlineItem(MenuItem menuItem, Outlet outlet, Session session) {
		if (menuItem == null) {
			return;
		}
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(OnlineItem.PROP_MENU_ITEM_ID, menuItem.getId()));
		criteria.add(Restrictions.eq(OnlineItem.PROP_OUTLET_ID, outlet.getId()));
		criteria.setMaxResults(1);

		OnlineItem onlineItem = (OnlineItem) criteria.uniqueResult();
		if (onlineItem == null) {
			onlineItem = new OnlineItem();
			onlineItem.setMenuItemId(menuItem.getId());
			onlineItem.setOutletId(outlet.getId());
			onlineItem.setActive(true);
		}
		onlineItem.setName(menuItem.getName());

		saveOrUpdate(onlineItem, session);

		String menuCategoryId = menuItem.getMenuCategoryId();
		if (menuCategoryId != null) {
			OnlineCategory onlineCategory = OnlineCategoryDAO.getInstance().findByCategoryId(menuCategoryId, outlet, session);
			if (onlineCategory == null) {
				onlineCategory = new OnlineCategory();
				onlineCategory.setCategoryId(menuCategoryId);
				onlineCategory.setOutletId(outlet.getId());
				onlineCategory.setName(menuItem.getMenuCategoryName());
				OnlineCategoryDAO.getInstance().save(onlineCategory, session);
			}
		}
	}

	public void removeOnlineItem(MenuItem menuItem, Outlet outlet) {
		OnlineItem onlineItem = findByMenuItemId(menuItem.getId(), outlet);
		if (onlineItem == null) {
			return;
		}
		delete(onlineItem);
	}

	public OnlineItem findByMenuItemId(String itemId, Outlet outlet) {
		try (Session session = createNewSession()) {
			return findByMenuItemId(itemId, session, outlet);
		}
	}

	public OnlineItem findByMenuItemId(String itemId, Session session, Outlet outlet) {
		if (StringUtils.isBlank(itemId)) {
			return null;
		}
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(OnlineItem.PROP_MENU_ITEM_ID, itemId));
		if (outlet != null) {
			criteria.add(Restrictions.eq(OnlineItem.PROP_OUTLET_ID, outlet.getId()));
		}
		criteria.setMaxResults(1);
		return (OnlineItem) criteria.uniqueResult();
	}

	public List<OnlineItem> findByMenuItemId(String itemId, Session session) {
		if (StringUtils.isBlank(itemId)) {
			return null;
		}
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(OnlineItem.PROP_MENU_ITEM_ID, itemId));
		return criteria.list();
	}

	public List<String> getOnlineMenuItemIds(List<MenuItem> existingItems) {
		if (existingItems == null || existingItems.isEmpty()) {
			return new ArrayList<>();
		}
		List<String> existingIds = new ArrayList<>();
		for (MenuItem menuItem : existingItems) {
			existingIds.add(menuItem.getId());
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(OnlineItem.PROP_MENU_ITEM_ID));
			criteria.add(Restrictions.in(OnlineItem.PROP_MENU_ITEM_ID, existingIds));
			return criteria.list();
		}
	}

	public List<String> getOnlineMenuItemIds(Outlet outlet) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(OnlineItem.PROP_MENU_ITEM_ID));
			criteria.add(Restrictions.eq(OnlineItem.PROP_OUTLET_ID, outlet.getId()));
			criteria.add(Restrictions.eq(OnlineItem.PROP_ACTIVE, Boolean.TRUE));
			return criteria.list();
		}
	}

	public List<String> getItemIdsByCategory(String categoryId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			criteria.setProjection(Projections.property(MenuItem.PROP_ID));
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, categoryId));
			return criteria.list();
		}
	}

	public List<OnlineItem> getOnlineItemsByCategory(String categoryId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OnlineItem.class);
			
			DetachedCriteria detachedCriteria = DetachedCriteria.forClass(MenuItem.class);
			detachedCriteria.setProjection(Projections.property(MenuItem.PROP_ID));
			detachedCriteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			detachedCriteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, categoryId));
			
			criteria.add(Property.forName(OnlineItem.PROP_MENU_ITEM_ID).in(detachedCriteria));
			return criteria.list();
		}
	}

	public void loadItems(PaginationSupport model, MenuSearchCriteria menuSearchCriteria, Outlet outlet) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			String itemName = menuSearchCriteria.getItemNameOrBarcode().trim();
			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(OnlineItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			if (outlet != null) {
				criteria.add(Restrictions.eq(OnlineItem.PROP_OUTLET_ID, outlet.getId()));
			}
			if (menuSearchCriteria.getMenuCategory() != null) {
				List<String> categoryMenuItemIds = getItemIdsByCategory(menuSearchCriteria.getMenuCategory().getId());
				if (categoryMenuItemIds == null || categoryMenuItemIds.isEmpty()) {
					model.setNumRows(0);
					model.setRows(new ArrayList<>());
					return;
				}
				criteria.add(Restrictions.in(OnlineItem.PROP_MENU_ITEM_ID, categoryMenuItemIds));
			}
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				model.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			List onlineItems = criteria.list();
			if (onlineItems == null || onlineItems.isEmpty()) {
				model.setRows(new ArrayList<>());
				return;
			}
			List<String> itemIds = new ArrayList<>();
			for (Iterator iterator = onlineItems.iterator(); iterator.hasNext();) {
				OnlineItem onlineItem = (OnlineItem) iterator.next();
				itemIds.add(onlineItem.getMenuItemId());
			}
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.in(MenuItem.PROP_ID, itemIds));
			criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME));
			List<MenuItem> result = criteria.list();
			model.setRows(result);
		}
	}

	public boolean isEnabledOnlineOrdering(MenuItem menuItem, Outlet outlet) {
		if (menuItem == null || menuItem.getId() == null) {
			return false;
		}
		return findByMenuItemId(menuItem.getId(), outlet) != null;
	}

	public boolean itemExists(String itemId, String outletId) {
		if (StringUtils.isBlank(itemId)) {
			return false;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(OnlineItem.PROP_MENU_ITEM_ID, itemId));
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(OnlineItem.PROP_OUTLET_ID, outletId));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		}
	}
}