package com.floreantpos.model;

import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseDoctorVisitMedicineItem;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "doctorVisit", "menuItem", "propertiesContainer", "medicineTimeNote", "medicationDurationType",
		"medicationReScheduled", "medicationScheduled", "medicationDuration", "durationContinue", "itemNameAsString", "afterNoon", "morning", "morningTime",
		"medTakingTime", "noon", "night", "takingQty", "takingNowTime", "afterNoonTime", "noonTime", "customName", "nightTime", "totalMedQty",
		"takingQtyUnit" })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement
public class DoctorVisitMedicineItem extends BaseDoctorVisitMedicineItem implements PropertyContainer2 {
	private static final long serialVersionUID = 1L;
	private transient MenuItem menuItem;
	private transient com.google.gson.JsonObject propertiesContainer;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public DoctorVisitMedicineItem() {
		setCreatedDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for primary key
	 */
	public DoctorVisitMedicineItem(java.lang.String id) {
		super(id);
		setCreatedDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for required fields
	 */
	public DoctorVisitMedicineItem(java.lang.String id, com.floreantpos.model.DoctorVisit doctorVisit) {

		super(id, doctorVisit);
		setCreatedDate(StoreDAO.getServerTimestamp());
	}

	/*[CONSTRUCTOR MARKER END]*/
	@XmlTransient
	public MenuItem getMenuItem() {
		return getMenuItem(false);
	}

	@XmlTransient
	public MenuItem getMenuItem(boolean forceUpdate) {
		String itemId = getItemId();
		if (StringUtils.isEmpty(itemId)) {
			return null;
		}

		if (menuItem == null) {
			return menuItem = MenuItemDAO.getInstance().loadInitialized(itemId);
		}

		if (forceUpdate) {
			menuItem = MenuItemDAO.getInstance().loadInitialized(itemId);
		}

		return menuItem;
	}

	public void setMenuItem(MenuItem menuItem) {
		this.menuItem = menuItem;
		String menuItemId = null;
		if (menuItem != null) {
			menuItemId = menuItem.getId();
		}
		super.setItemId(menuItemId);
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	private void initPropertyContainer() {
		if (propertiesContainer == null) {
			if (StringUtils.isBlank(super.getProperties())) {
				propertiesContainer = new com.google.gson.JsonObject();
			}
			else {
				propertiesContainer = new Gson().fromJson(super.getProperties(), com.google.gson.JsonObject.class);
			}
		}
	}

	@Override
	public void removeProperty(String propertyName) {
		initPropertyContainer();
		propertiesContainer.remove(propertyName);
		super.setProperties(propertiesContainer.toString());
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	public void putMedicationScheduled(boolean medicationScheduled) {
		addProperty("medication_scheduled", String.valueOf(medicationScheduled)); //$NON-NLS-1$
	}

	public Boolean isMedicationScheduled() {
		return getBooleanProperty("medication_scheduled", false); //$NON-NLS-1$
	}

	public Boolean isMedicationReScheduled() {
		if (!isMedicationScheduled()) {
			return false;
		}

		if (!isDurationContinue()) {
			return true;
		}

		Date createdDate = getCreatedDate();
		double medicineDosageInterval = getMedicationDuration();
		if (medicineDosageInterval == 0) {
			medicineDosageInterval = 7;
		}

		int daysBetween = DateUtil.getDaysBetween(createdDate, StoreDAO.getServerTimestamp());
		if (daysBetween < medicineDosageInterval) {
			return true;
		}

		return false;

	}

	public String getMedTakingTime() {
		return getProperty("medicine.time", "");
	}

	public void putMedTakingTime(String medTakingTime) {
		addProperty("medicine.time", medTakingTime); //$NON-NLS-1$
	}

	public boolean isMorning() {
		return getBooleanProperty("medicine.time.morning", false); //$NON-NLS-1$
	}

	public void putMorning(boolean isMorning) {
		addProperty("medicine.time.morning", String.valueOf(isMorning)); //$NON-NLS-1$
	}

	public boolean isNoon() {
		return getBooleanProperty("medicine.time.noon", false); //$NON-NLS-1$
	}

	public void putNoon(boolean isNoon) {
		addProperty("medicine.time.noon", String.valueOf(isNoon)); //$NON-NLS-1$
	}

	public boolean isAfterNoon() {
		return getBooleanProperty("medicine.time.after_noon", false); //$NON-NLS-1$
	}

	public void putAfterNoon(boolean isAfterNoon) {
		addProperty("medicine.time.after_noon", String.valueOf(isAfterNoon)); //$NON-NLS-1$
	}

	public boolean isNight() {
		return getBooleanProperty("medicine.time.night", false); //$NON-NLS-1$
	}

	public void putNight(boolean isNight) {
		addProperty("medicine.time.night", String.valueOf(isNight)); //$NON-NLS-1$
	}

	public String getMorningTime() {
		return getProperty("medicine.time.morning.time", "08:00");
	}

	public void putMorningTime(String morningTime) {
		addProperty("medicine.time.morning.time", morningTime); //$NON-NLS-1$
	}

	public String getNoonTime() {
		return getProperty("medicine.time.noon.time", "12:00");
	}

	public void putNoonTime(String time) {
		addProperty("medicine.time.noon.time", time); //$NON-NLS-1$
	}

	public String getAfterNoonTime() {
		return getProperty("medicine.time.after_noon.time", "16:00");
	}

	public void putAfterNoonTime(String time) {
		addProperty("medicine.time.after_noon.time", time); //$NON-NLS-1$
	}

	public String getNightTime() {
		return getProperty("medicine.time.night.time", "21:00");
	}

	public void putNightTime(String time) {
		addProperty("medicine.time.night.time", time); //$NON-NLS-1$
	}

	public String getMedicationDurationDisplay(boolean showBanglaTextOnPrescription) {
		double medicationDuration = getMedicationDuration();
		//if (medicationDuration <= 0) {
		//	return StringUtils.EMPTY;
		//}
		String displayDuration = StringUtils.EMPTY;
		if (medicationDuration > 0) {
			if (showBanglaTextOnPrescription) {
				displayDuration = " " + NumberUtil.convertToBanglaNumber(NumberUtil.formatNumberIfNeeded(medicationDuration));
			}
			else {
				displayDuration = " " + NumberUtil.formatNumberIfNeeded(medicationDuration);
			}
		}

		MedicationDurationType medDurationType = MedicationDurationType.fromString(getMedicationDurationType());
		if (medDurationType == null) {
			return displayDuration;
		}
		switch (medDurationType) {
			case TIME:
				displayDuration = showBanglaTextOnPrescription ? " এখনি" : " now";
				break;

			case DAY:
				if (medicationDuration > 0) {
					if (showBanglaTextOnPrescription) {
						displayDuration += " দিন";
					}
					else {
						displayDuration += (medicationDuration > 1 ? " Days" : " Day");
					}
				}
				break;

			case MONTH:
				if (medicationDuration > 0) {
					if (showBanglaTextOnPrescription) {
						displayDuration += " মাস";
					}
					else {
						displayDuration += (medicationDuration > 1 ? " Months" : " Month");
					}
				}
				break;

			case INFINITY:
				if (showBanglaTextOnPrescription) {
					displayDuration = " (চলবে)";
				}
				else {
					displayDuration = " (Continue)";
				}
				break;

			default:
				break;
		}
		return displayDuration;
	}

	public double getMedicationDuration() {
		return getDoubleProperty("medication.duration");
	}

	public void putMedicationDuration(double medicineDosageInterval) {
		addProperty("medication.duration", String.valueOf(medicineDosageInterval)); //$NON-NLS-1$
	}

	public String getMedicationDurationType() {
		return getProperty("medication.duration.type");
	}

	public void putMedicationDurationType(MedicationDurationType type) {
		addProperty("medication.duration.type", type != null ? type.name() : null); //$NON-NLS-1$
	}

	public double getTakingQty() {
		return getDoubleProperty("medicine.quantity");
	}

	public void putTakingQty(double takingQty) {
		addProperty("medicine.quantity", String.valueOf(takingQty)); //$NON-NLS-1$
	}

	public String getTakingQtyUnit() {
		return getProperty("medicine.quantity.unit", "");
	}

	public void putTakingQtyUnit(String takingQtyUnit) {
		addProperty("medicine.quantity.unit", takingQtyUnit); //$NON-NLS-1$
	}

	public String getCustomName() {
		return getProperty("custom.name", "");
	}

	public void putCustomName(String customName) {
		addProperty("custom.name", customName); //$NON-NLS-1$
	}

	public String getMedicineTimeNote() {
		return getProperty("medicine.time.notes", "");
	}

	public void putMedicineTimeNote(String medTimeNote) {
		addProperty("medicine.time.notes", medTimeNote); //$NON-NLS-1$
	}

	public double getTotalMedQty() {
		return getDoubleProperty("total_med_quantity");
	}

	public void putTotalMedQty(double takingQty) {
		addProperty("total_med_quantity", String.valueOf(takingQty)); //$NON-NLS-1$
	}

	public List<MedicationSchedule> createMedicationSchedules(List<MedicationSchedule> existingMS) {
		List<MedicationSchedule> medicationSchedules = new ArrayList<MedicationSchedule>();

		String medicationDurationType = getMedicationDurationType();
		MedicationDurationType medDurationType = MedicationDurationType.fromString(medicationDurationType);

		MedicationTakingInstruction takingTime = MedicationTakingInstruction.fromString(getMedTakingTime());
		if (MedicationTakingInstruction.IMMEDIATE == takingTime || medDurationType == MedicationDurationType.TIME) {
			putTotalMedQty(1);

			MedicationSchedule immediateSchedule = createMedicationSchedule(null, 0);
			if (immediateSchedule != null) {
				addToMedicationSchedules(existingMS, medicationSchedules, immediateSchedule);
			}

			return medicationSchedules;
		}

		boolean morning = isMorning();
		boolean noon = isNoon();
		boolean afterNoon = isAfterNoon();
		boolean night = isNight();

		double inDays = 0;

		if (morning) {
			inDays++;
		}
		if (noon) {
			inDays++;
		}
		if (afterNoon) {
			inDays++;
		}
		if (night) {
			inDays++;
		}

		double totalRequiredSchedule = 0;
		double duration = getMedicationDuration();
		if (medDurationType != null) {
			switch (medDurationType) {
				case TIME:
					totalRequiredSchedule = 1;
					break;

				case DAY:
					totalRequiredSchedule = inDays * duration;
					break;

				case MONTH:
					totalRequiredSchedule = inDays * duration * 30;
					break;

				case INFINITY:
					totalRequiredSchedule = inDays * duration * 7;
					break;

				default:
					break;
			}
		}

		putTotalMedQty(totalRequiredSchedule);

		if (totalRequiredSchedule > 0 && (morning || noon || afterNoon || night)) {

			int scheduleCount = 0;
			int dayOffset = 0;

			do {
				if (morning) {
					MedicationSchedule medicationSchedule = createMedicationSchedule(MedicationShift.MORNING, dayOffset);
					if (medicationSchedule != null) {
						scheduleCount++;
						addToMedicationSchedules(existingMS, medicationSchedules, medicationSchedule);
					}
					if (scheduleCount >= totalRequiredSchedule)
						break;
				}

				if (noon) {
					MedicationSchedule medicationSchedule = createMedicationSchedule(MedicationShift.NOON, dayOffset);
					if (medicationSchedule != null) {
						scheduleCount++;
						addToMedicationSchedules(existingMS, medicationSchedules, medicationSchedule);
					}
					if (scheduleCount >= totalRequiredSchedule)
						break;
				}
				if (afterNoon) {
					MedicationSchedule medicationSchedule = createMedicationSchedule(MedicationShift.AFTERNOON, dayOffset);
					if (medicationSchedule != null) {
						scheduleCount++;
						addToMedicationSchedules(existingMS, medicationSchedules, medicationSchedule);
					}
					if (scheduleCount >= totalRequiredSchedule)
						break;
				}
				if (night) {
					MedicationSchedule medicationSchedule = createMedicationSchedule(MedicationShift.NIGHT, dayOffset);
					if (medicationSchedule != null) {
						scheduleCount++;
						addToMedicationSchedules(existingMS, medicationSchedules, medicationSchedule);
					}
					if (scheduleCount >= totalRequiredSchedule)
						break;
				}
				dayOffset++; // Move to the next day
			} while (scheduleCount < totalRequiredSchedule);

		}
		else {
			PosLog.info(getClass(), "Total med quantity: " + totalRequiredSchedule + ", Item name: " + getMenuItem().getDisplayName()
					+ ", Schedule not created because there is no taking time.");
		}
		return medicationSchedules;
	}

	private void addToMedicationSchedules(List<MedicationSchedule> existingMS, List<MedicationSchedule> medicationSchedules,
			MedicationSchedule medicationSchedule) {
		boolean exists = false;
		for (Iterator<MedicationSchedule> iterator = existingMS.iterator(); iterator.hasNext();) {
			MedicationSchedule exSchedule = (MedicationSchedule) iterator.next();

			if (doCompareMedicationSchedule(medicationSchedule, exSchedule)) {
				exists = true;
				String existingDrVisitId = exSchedule.getDoctorVisitId();
				exSchedule.setDoctorVisitId(medicationSchedule.getDoctorVisitId());
				exSchedule.putActionDescription(
						String.format("Doctor visit ID has been changed from %s to %s.", existingDrVisitId, medicationSchedule.getDoctorVisitId()));
				medicationSchedules.add(exSchedule);

				iterator.remove();
				break;
			}
		}

		if (!exists) {
			medicationSchedules.add(medicationSchedule);
		}

	}

	//@formatter:off
	private boolean doCompareMedicationSchedule(MedicationSchedule medicationSchedule, MedicationSchedule exSchedule) {
		return exSchedule.getPatientId().equals(medicationSchedule.getPatientId())
		&& exSchedule.getAdmissionId().equals(medicationSchedule.getAdmissionId()) 
		&& exSchedule.getName().equals(medicationSchedule.getName())
		&& exSchedule.getMedicationTime().getTime() == (medicationSchedule.getMedicationTime().getTime())
		;
	}
	//@formatter:on

	private MedicationSchedule createMedicationSchedule(MedicationShift medicationShift, int day) {

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(StoreDAO.getServerTimestamp());

		calendar.add(Calendar.DAY_OF_MONTH, day);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);

		if (medicationShift != null) {

			LocalTime localTime = null;
			switch (medicationShift) {
				case MORNING:
					localTime = DateUtil.parse24HourLocalTime(getMorningTime());
					break;
				case NOON:
					localTime = DateUtil.parse24HourLocalTime(getNoonTime());
					break;
				case AFTERNOON:
					localTime = DateUtil.parse24HourLocalTime(getAfterNoonTime());
					break;
				case NIGHT:
					localTime = DateUtil.parse24HourLocalTime(getNightTime());

				default:
					break;
			}

			if (localTime != null) {
				calendar.set(Calendar.HOUR_OF_DAY, localTime.getHour());
				calendar.set(Calendar.MINUTE, localTime.getMinute());
			}
		}

		if (calendar.getTime().before(DateUtil.convertServerTimeToBrowserTime(getCreatedDate()))) {
			PosLog.debug(getClass(), "The medication schedule was ignored due to a past time.");
			return null;
		}

		MedicationSchedule medicationSchedule = new MedicationSchedule();
		medicationSchedule.setName(medicationShift == null ? MedicationTakingInstruction.IMMEDIATE.name() : medicationShift.name());
		medicationSchedule.setOutletId(DataProvider.get().getCurrentOutletId());
		medicationSchedule.setItemId(getItemId());
		medicationSchedule.setDoctorVisitMedicineItemId(getId());
		DoctorVisit doctorVisit = getDoctorVisit();
		if (doctorVisit != null) {
			medicationSchedule.setDoctorVisitId(doctorVisit.getId());
			medicationSchedule.setAdmissionId(doctorVisit.getAdmissionId());
			medicationSchedule.setPatientId(doctorVisit.getPatientId());
		}

		medicationSchedule.putMedTakingTime(getMedTakingTime());
		medicationSchedule.putTakingQty(getTakingQty());
		medicationSchedule.putTakingQtyUnit(getTakingQtyUnit());
		medicationSchedule.putCustomName(getCustomName());

		Date convertDateToBrowserTime;
		if (medicationShift != null) {
			convertDateToBrowserTime = DateUtil.convertDateToServerTime(calendar.getTime());
		}
		else {
			convertDateToBrowserTime = calendar.getTime();
		}

		medicationSchedule.setMedicationTime(convertDateToBrowserTime);
		medicationSchedule.setStatus(MedicationStatus.PENDING.name());

		return medicationSchedule;
	}

	public Date getTakingNowTime() {
		return getDateProperty("takingNowTime");
	}

	public void putTakingNowTime(Date currentTime) {
		addDateProperty("takingNowTime", currentTime);
	}

	public boolean isDurationContinue() {
		return getBooleanProperty("duration.continue", false);
	}

	public void putDurationContinue(boolean isDurationContinue) {
		addProperty("duration.continue", String.valueOf(isDurationContinue));
	}

	public String getItemNameAsString() {
		MenuItem menuItem = getMenuItem();
		return menuItem != null ? menuItem.getDisplayName() : getCustomName();
	}

	public String getItemNameDisplayWithInstruction(boolean showInDetailView, Boolean showBanglaTextOnPrescription) {
		String medicineNames = "";
		String menuItemName = getItemNameAsString();

		if (showInDetailView) {
			medicineNames += "<ul class=\"list_ul\">"; //$NON-NLS-1$
			medicineNames += "<li class=\"list_item\">" + menuItemName + "</li>"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		else {
			medicineNames += "<div>&#x2022; " + menuItemName + "</div>"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		return medicineNames += getInstructionDisplay(showInDetailView, showBanglaTextOnPrescription);
	}

	public String getInstructionDisplay(boolean showInDetailView, Boolean showBanglaTextOnPrescription) {
		String instructionTxt = StringUtils.EMPTY;

		Boolean morning = isMorning();
		Boolean noon = isNoon();
		Boolean afterNoon = isAfterNoon();
		Boolean night = isNight();

		String divStartTag;
		if (showInDetailView) {
			divStartTag = "<div class=\"list_item\">"; //$NON-NLS-1$
		}
		else {
			divStartTag = "<div>"; //$NON-NLS-1$
		}
		instructionTxt += divStartTag; //$NON-NLS-1$

		if (morning || noon || afterNoon || night) {
			StringBuilder medTimeBuilder = new StringBuilder();
			medTimeBuilder.append(showBanglaTextOnPrescription ? morning ? "১" : "০" : morning ? "1" : "0"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			medTimeBuilder.append(" + "); //$NON-NLS-1$

			medTimeBuilder.append(showBanglaTextOnPrescription ? noon ? "১" : "০" : noon ? "1" : "0"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			medTimeBuilder.append(" + "); //$NON-NLS-1$

			if (afterNoon) {
				medTimeBuilder.append(showBanglaTextOnPrescription ? afterNoon ? "১" : "০" : afterNoon ? "1" : "0"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				medTimeBuilder.append(" + "); //$NON-NLS-1$
			}
			medTimeBuilder.append(showBanglaTextOnPrescription ? night ? "১" : "০" : night ? "1" : "0"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			medTimeBuilder.append(" "); //$NON-NLS-1$
			instructionTxt += medTimeBuilder.toString();
		}
		MedicationTakingInstruction takingInstruction = MedicationTakingInstruction.fromString(getMedTakingTime());
		String medicineTime = takingInstruction == null ? StringUtils.EMPTY : takingInstruction.toString();
		if (StringUtils.isNotBlank(medicineTime) && MedicationTakingInstruction.ANY != takingInstruction) {
			if (showBanglaTextOnPrescription) {
				if (MedicationTakingInstruction.BEFORE_MEAL == takingInstruction) {
					medicineTime = "খাবারের আগে"; //$NON-NLS-1$
				}
				else if (MedicationTakingInstruction.AFTER_MEAL == takingInstruction) {
					medicineTime = "খাবারের পর"; //$NON-NLS-1$
				}
				else if (MedicationTakingInstruction.IMMEDIATE == takingInstruction) {
					medicineTime = "এখনি"; //$NON-NLS-1$
				}
			}

			instructionTxt += " " + medicineTime; //$NON-NLS-1$
		}

		if (!(MedicationTakingInstruction.IMMEDIATE == takingInstruction)) {
			String medicationDurationDisplay = getMedicationDurationDisplay(showBanglaTextOnPrescription);
			if (StringUtils.isNotBlank(medicationDurationDisplay)) {
				instructionTxt += (divStartTag.equals(instructionTxt) ? StringUtils.EMPTY : ", ") + medicationDurationDisplay;
			}
		}
		instructionTxt += "</div>"; //$NON-NLS-1$

		if (showInDetailView) {
			instructionTxt += "<div class=\"list_item\">"; //$NON-NLS-1$
		}
		else {
			instructionTxt += "<div>"; //$NON-NLS-1$
		}
		String medicineQuantity = NumberUtil.formatNumberIfNeeded(getTakingQty());
		if (StringUtils.isNotBlank(medicineQuantity) && !medicineQuantity.equals("0")) { //$NON-NLS-1$
			if (showBanglaTextOnPrescription) {
				instructionTxt += " " + NumberUtil.convertToBanglaNumber(medicineQuantity); //$NON-NLS-1$
			}
			else {
				instructionTxt += " " + medicineQuantity; //$NON-NLS-1$
			}

			String medicineQuantityUnit = getTakingQtyUnit();
			if (StringUtils.isNotBlank(medicineQuantityUnit)) {
				instructionTxt += " " + medicineQuantityUnit; //$NON-NLS-1$
			}
			else {
				if (showBanglaTextOnPrescription) {
					instructionTxt += " টি"; //$NON-NLS-1$
				}
				else {
					instructionTxt += " Piece"; //$NON-NLS-1$
				}
			}
			if (takingInstruction != null && !(MedicationTakingInstruction.IMMEDIATE == takingInstruction)) {
				if (showBanglaTextOnPrescription) {
					instructionTxt += " প্রতিবার";
				}
				else {
					instructionTxt += " each time";
				}
			}
		}
		instructionTxt += "</div>"; //$NON-NLS-1$

		if (showInDetailView) {
			instructionTxt += "</ul>"; //$NON-NLS-1$
		}

		String notes = getMedicineTimeNote();
		if (StringUtils.isNotBlank(notes)) {
			instructionTxt += "<div>" + notes + "</div>"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		instructionTxt += "</br>"; //$NON-NLS-1$
		return instructionTxt;
	}

}