/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.print;

import java.awt.print.PrinterAbortException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.PrinterIsNotConfiguredException;
import com.floreantpos.main.Application;
import com.floreantpos.model.CashBreakdown;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.DrawerType;
import com.floreantpos.model.Printer;
import com.floreantpos.model.Store;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.TipsCashoutReport;
import com.floreantpos.model.TipsCashoutReportTableModel;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.CashDrawerDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.VoidItemDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.ReceiptPrintService;
import com.floreantpos.report.ReportUtil;
import com.floreantpos.services.report.CashDrawerReportService;
import com.floreantpos.services.report.StoreSessionSummaryReportService;
import com.floreantpos.util.NumberUtil;

import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JREmptyDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRBeanCollectionDataSource;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;

public class PosPrintService {
	private static final SimpleDateFormat date_formatter = new SimpleDateFormat("dd MMM, yyyy hh:mm aaa"); //$NON-NLS-1$
	//	private static final SimpleDateFormat date_formatter_yy = new SimpleDateFormat("dd MMM, yy hh:mm aaa");
	private static Log logger = LogFactory.getLog(PosPrintService.class);

	public static void printDrawerPullReport(CashDrawer cashDrawer) throws Exception {
		try {
			JasperPrint jasperPrint = populateDrawerPullReportParameters(cashDrawer);
			printReportByJasperPrint(jasperPrint);
		} catch (PrinterIsNotConfiguredException e) {
			throw e;
		} catch (Exception e) {
			//
		}
	}

	public static void printCashDrawerSummaryReport(CashDrawer cashDrawer) {
		try {
			JasperPrint jasperPrint = populateStoreSessionSummaryReportParameters(cashDrawer);
			printReportByJasperPrint(jasperPrint);
		} catch (PrinterIsNotConfiguredException e) {
			throw e;
		} catch (PrinterAbortException e) {
		} catch (Exception e) {
			//PosLog.error(PosPrintService.class, e);
		}
	}

	/*	public static void printServerSummaryReport(CashDrawer cashDrawer) {
			try {
				JasperPrint jasperPrint = populateServerSummaryReportParameters(cashDrawer);
	
				JRPrintServiceExporter exporter = new JRPrintServiceExporter();
				exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
				exporter.setParameter(JRPrintServiceExporterParameter.PRINT_SERVICE,
						PrintServiceUtil.getPrintServiceForPrinter(jasperPrint.getProperty("printerName"))); //$NON-NLS-1$
				exporter.exportReport();
				
				PrintRequestAttributeSet printRequestAttributeSet = new HashPrintRequestAttributeSet();
				printRequestAttributeSet.add(MediaSizeName.ISO_A4);
	
				PrintServiceAttributeSet printServiceAttributeSet = new HashPrintServiceAttributeSet();
				PrintService printServiceForPrinter = PrintServiceUtil.getPrintServiceForPrinter(jasperPrint.getProperty("printerName"));
				
				printServiceAttributeSet.add(new PrinterName(printServiceForPrinter.getName(), null));
	
				JRPrintServiceExporter exporter = new JRPrintServiceExporter();
	
				exporter.setExporterInput(new SimpleExporterInput(jasperPrint));
				SimplePrintServiceExporterConfiguration configuration = new SimplePrintServiceExporterConfiguration();
				configuration.setPrintRequestAttributeSet(printRequestAttributeSet);
				configuration.setPrintServiceAttributeSet(printServiceAttributeSet);
				configuration.setDisplayPageDialog(false);
				configuration.setDisplayPrintDialog(true);
				exporter.setConfiguration(configuration);
				exporter.exportReport();
				
			} catch (PrinterAbortException e) {
			} catch (Exception e) {
				PosLog.error(PosPrintService.class, e);
			}
		}*/

	public static void printReportByJasperPrint(JasperPrint jasperPrint) {
		try {
			List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
			if (receiptPrinters == null || receiptPrinters.isEmpty()) {
				throw new PrinterIsNotConfiguredException(Messages.getString("PosPrintService.11")); //$NON-NLS-1$
			}

			boolean printerFound = false;
			for (Printer printer : receiptPrinters) {
				String printerName = printer.getDeviceName();
				if (StringUtils.isEmpty(printerName)) {
					continue;
				}
				printerFound = true;
				jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, printerName);
				ReceiptPrintService.printQuitely(jasperPrint);
			}
			if (!printerFound) {
				throw new PrinterIsNotConfiguredException(Messages.getString("PosPrintService.11")); //$NON-NLS-1$
			}
		} catch (PrinterIsNotConfiguredException e) {
			throw e;
		} catch (Exception e) {
			PosLog.error(PosPrintService.class, e);
		}
	}

	public static void printServerTipsReport(TipsCashoutReport report) {
		try {
			JasperPrint jasperPrint = createServerTipsReport(report);
			printReportByJasperPrint(jasperPrint);
		} catch (Exception e) {
			PosLog.error(PosPrintService.class, e);
			logger.error("error print tips report", e); //$NON-NLS-1$
		}
	}

	public static JasperPrint createServerTipsReport(TipsCashoutReport report) throws Exception {
		try {
			HashMap<String, Object> parameters = new HashMap<String, Object>();
			parameters.put("server", report.getServer()); //$NON-NLS-1$
			if (report.getFromDate() != null)
				parameters.put("fromDate", Application.formatDate(report.getFromDate())); //$NON-NLS-1$
			if (report.getToDate() != null)
				parameters.put("toDate", Application.formatDate(report.getToDate())); //$NON-NLS-1$
			parameters.put("reportDate", new SimpleDateFormat("dd MMM yyyy, hh:mm aaa").format(report.getReportTime())); //$NON-NLS-1$ //$NON-NLS-2$
			parameters.put("transactionCount", report.getDatas() == null ? "0" : "" + report.getDatas().size()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			parameters.put("cashTips", NumberUtil.formatNumber(report.getCashTipsAmount())); //$NON-NLS-1$
			parameters.put("chargedTips", NumberUtil.formatNumber(report.getChargedTipsAmount())); //$NON-NLS-1$
			parameters.put("declaredTips", report.getDeclaredTipsAmount()); //$NON-NLS-1$
			parameters.put("tipsDue", report.getTipsDue()); //$NON-NLS-1$

			Store store = StoreDAO.getRestaurant();

			//ReportUtil.populatePatternExpression(parameters, store);
			parameters.put("headerLine1", store.getName()); //$NON-NLS-1$

			JasperReport mainReport = ReportUtil.getReport("ServerTipsReport"); //$NON-NLS-1$
			JRDataSource dataSource = new JRTableModelDataSource(new TipsCashoutReportTableModel(report.getDatas(),
					new String[] { "ticketId", "saleType", "ticketTotal", "tips", "chargedTips", "tipsPaid" })); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
			return JasperFillManager.fillReport(mainReport, parameters, dataSource);
		} catch (Exception e) {
			throw e;
		}
	}

	public static JasperPrint populateDrawerPullReportParameters(CashDrawer cashDrawer) throws Exception {
		boolean staffBankReport = cashDrawer.getDrawerType() == DrawerType.STAFF_BANK;
		HashMap<String, Object> parameters = new HashMap<String, Object>();
		Store store = StoreDAO.getRestaurant();
		//ReportUtil.populatePatternExpression(parameters, store);
		parameters.put("headerLine1", store.getName()); //$NON-NLS-1$
		parameters.put("reportTitle", staffBankReport ? "Staff bank report" : "Cash drawer report"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		populateTerminalStatusReportParams(parameters);
		parameters.put(JRParameter.IS_IGNORE_PAGINATION, true);
		User assignedUser = cashDrawer.getAssignedUser();
		if (assignedUser != null) {
			String userInfo = assignedUser.getFullName();
			parameters.put("user", (staffBankReport ? "Staff: " : Messages.getString("PosPrintService.4") + " ") + userInfo); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (cashDrawer.getStartTime() != null)
			parameters.put("startTime", Messages.getString("PosPrintService.12") + date_formatter.format(cashDrawer.getStartTime())); //$NON-NLS-1$ //$NON-NLS-2$
		parameters.put("date", Messages.getString("PosPrintService.13") + date_formatter.format(new Date())); //$NON-NLS-1$ //$NON-NLS-2$
		parameters.put("totalVoid", cashDrawer.getTotalVoid()); //$NON-NLS-1$

		parameters.put("declaredTips", cashDrawer.getDeclaredTips()); //$NON-NLS-1$

		JasperReport subReportCurrencyBalance = ReportUtil.getReport("drawer-currency-balance"); //$NON-NLS-1$
		JasperReport subReport = ReportUtil.getReport("drawer-pull-void-veport"); //$NON-NLS-1$

		parameters.put("currencyBalanceReport", subReportCurrencyBalance); //$NON-NLS-1$
		parameters.put("subreportParameter", subReport); //$NON-NLS-1$

		JasperReport mainReport = ReportUtil.getReport("terminal_status_report"); //$NON-NLS-1$
		JRBeanCollectionDataSource dataSource = new JRBeanCollectionDataSource(Arrays.asList(new CashDrawer[] { cashDrawer }));
		JasperPrint jasperPrint = JasperFillManager.fillReport(mainReport, parameters, dataSource);
		jasperPrint.setProperty("printerName", DataProvider.get().getPrinters().getReceiptPrinter()); //$NON-NLS-1$
		jasperPrint.setName("DrawerPullReport" + cashDrawer.getId()); //$NON-NLS-1$
		return jasperPrint;
	}

	public static JasperPrint populateStoreSessionSummaryReportParameters(CashDrawer cashDrawer) throws Exception {
		HashMap<String, Object> map = new HashMap<String, Object>();
		Store store = StoreDAO.getRestaurant();
		//ReportUtil.populatePatternExpression(map, store);
		map.put("headerLine1", store.getName()); //$NON-NLS-1$
		map.put("reportTitle", ReportUtil.reportLabelWithBoldTag(Messages.getString("StoreSessionReportTitle"))); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("salesBalance", Messages.getString("SALES_BALANCE")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowNetSales", POSConstants.NET_SALES); //$NON-NLS-1$ 
		map.put("rowDiscount", "-" + POSConstants.DISCOUNT); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowSalesTax", "+" + POSConstants.SALES_TAX); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowSC", "+" + POSConstants.SERVICE_CHARGE); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowTotal", "=" + POSConstants.TOTAL); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowTips", "+" + POSConstants.TIPS); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowGReceivable", "=" + Messages.getString("grossReceivable")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowCReceipts", "-" + Messages.getString("cashReceipts")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowCCards", "-" + Messages.getString("creditCards")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowDCards", "-" + Messages.getString("debitCards")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowMPayments", "-" + Messages.getString("memberPayments")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowCPayments", "-" + Messages.getString("customPayments")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowGCert", "-" + Messages.getString("giftCertificate")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowRefundPlus", "+" + POSConstants.REFUND_BUTTON_TEXT); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowTolerance", "+" + Messages.getString("CurrencyExplorer.8")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowReceiptDiff", "=" + Messages.getString("receiptDifferential")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowCashTips", "+" + Messages.getString("TipsCashoutReportTableModel.9")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowChargedTips", "+" + Messages.getString("chargedTips")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowTipsPaid", "-" + Messages.getString("TipsCashoutReportTableModel.5")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowTipsDiff", "=" + Messages.getString("tipsDiff")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("cashBalance", Messages.getString("CASH_BALANCE")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowCash", POSConstants.CASH); //$NON-NLS-1$
		map.put("rowTipsPaid", "-" + Messages.getString("TipsCashoutReportTableModel.5")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowPayOut", "-" + Messages.getString("UserPermission.13")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowRefundMinus", "-" + POSConstants.REFUND_BUTTON_TEXT); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put("rowBeginCash", "+" + Messages.getString("beginCash")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  
		map.put("rowDBleed", "-" + Messages.getString("CashDropDialog.2")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  
		map.put("rowDAcc", "=" + Messages.getString("drawerAccountable")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  
		map.put("rowDTips", ">" + Messages.getString("declaredTips")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  

		map.put(JRParameter.IS_IGNORE_PAGINATION, true);
		map.put("startTime", Messages.getString("PosPrintService.14") + date_formatter.format(cashDrawer.getStartTime())); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("user", Messages.getString("PosPrintService.16") + cashDrawer.getAssignedBy()); //$NON-NLS-1$ //$NON-NLS-2$
		if (cashDrawer.getReportTime() != null) {
			map.put("date", Messages.getString("PosPrintService.17") + date_formatter.format(cashDrawer.getReportTime())); //$NON-NLS-1$ //$NON-NLS-2$
			map.put("reportUser", Messages.getString("PosPrintService.18") + cashDrawer.getClosedBy()); //$NON-NLS-1$ //$NON-NLS-2$
		}

		map.put("totalVoid", cashDrawer.getTotalVoid()); //$NON-NLS-1$
		map.put("declaredTips", cashDrawer.getDeclaredTips()); //$NON-NLS-1$

		JasperReport subReportCurrencyBalance = ReportUtil.getReport("drawer-currency-balance"); //$NON-NLS-1$
		JasperReport subReport = ReportUtil.getReport("drawer-pull-void-veport"); //$NON-NLS-1$

		map.put("currencyBalanceReport", subReportCurrencyBalance); //$NON-NLS-1$
		map.put("subreportParameter", subReport); //$NON-NLS-1$

		JasperReport mainReport = ReportUtil.getReport("store-session-summary-report_72"); //$NON-NLS-1$

		JRBeanCollectionDataSource dataSource = new JRBeanCollectionDataSource(Arrays.asList(new CashDrawer[] { cashDrawer }));
		JasperPrint jasperPrint = JasperFillManager.fillReport(mainReport, map, dataSource);
		//TODO: handle exception
		jasperPrint.setProperty("printerName", DataProvider.get().getPrinters().getReceiptPrinter()); //$NON-NLS-1$
		jasperPrint.setName("StoreSessionSummaryReport"); //$NON-NLS-1$
		return jasperPrint;
	}

	public static JasperPrint populateTerminalStatusReport(CashDrawer cashDrawer) throws Exception {
		boolean staffBankReport = cashDrawer.getDrawerType() == DrawerType.STAFF_BANK;
		HashMap<String, Object> map = new HashMap<String, Object>();
		Store store = StoreDAO.getRestaurant();
		//ReportUtil.populatePatternExpression(map, store);
		map.put("headerLine1", store.getName()); //$NON-NLS-1$
		map.put("reportTitle", staffBankReport ? Messages.getString("StaffBankReportTitle") : Messages.getString("CashDrawerReportTitle")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		populateTerminalStatusReportParams(map);

		JasperPrint jasperPrint = populateTerminalStatusReport(cashDrawer, staffBankReport, map);
		return jasperPrint;
	}

	private static void populateTerminalStatusReportParams(HashMap<String, Object> map) {
		map.put("salesBalance", Messages.getString("SALES_BALANCE")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowGrossReceipts", Messages.getString("grossReceipts")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowCReceipts", "-" + Messages.getString("cashReceipts")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowCCards", "-" + Messages.getString("creditCards")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowDCards", "-" + Messages.getString("debitCards")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowMPayments", "-" + Messages.getString("memberPayments")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowCPayments", "-" + Messages.getString("customPayments")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowGReturns", "-" + Messages.getString("giftReurns")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowGCChange", "-" + Messages.getString("giftCertChange")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowRefundPlus", "+" + POSConstants.REFUND_BUTTON_TEXT); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowReceiptDiff", "=" + Messages.getString("receiptDifferential")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowCashTips", "+" + Messages.getString("TipsCashoutReportTableModel.9")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowChargedTips", "+" + Messages.getString("chargedTips")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowTipsPaid", "-" + Messages.getString("TipsCashoutReportTableModel.5")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowTipsDiff", "=" + Messages.getString("tipsDiff")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("cashBalance", Messages.getString("CASH_BALANCE")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowCash", POSConstants.CASH); //$NON-NLS-1$
		map.put("rowTipsPaid", "-" + Messages.getString("TipsCashoutReportTableModel.5")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowPayOut", "-" + Messages.getString("UserPermission.13")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowRefundMinus", "-" + POSConstants.REFUND_BUTTON_TEXT); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put("rowBeginCash", "+" + Messages.getString("beginCash")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  
		map.put("rowDBleed", "-" + Messages.getString("CashDropDialog.2")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  
		map.put("rowDAcc", "=" + Messages.getString("drawerAccountable")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  
		map.put("rowDTips", ">" + Messages.getString("declaredTips")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  
		map.put("rowCToDipo", ">" + Messages.getString("cashToDeposit")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$  
		map.put("rowCBreakD", Messages.getString("CBreakDown")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowCSRAmount", POSConstants.AMOUNT); //$NON-NLS-1$
		map.put("rowVRExceptions", Messages.getString("VRExceptions")); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put("rowVRTax", Messages.getString("VRTax")); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put("rowVRAmount", POSConstants.AMOUNT); //$NON-NLS-1$ 
		map.put("rowVRTotal", POSConstants.TOTAL); //$NON-NLS-1$ 
	}

	public static JasperPrint populateServerSummaryReportParameters(CashDrawer cashDrawer) throws Exception {
		HashMap<String, Object> map = new HashMap<String, Object>();
		Store store = StoreDAO.getRestaurant();
		//ReportUtil.populatePatternExpression(map, store);
		map.put("headerLine1", store.getName()); //$NON-NLS-1$
		map.put("SSReportTitle", Messages.getString("SSReportTitle")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("salesBalance", Messages.getString("SALES_BALANCE")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowNetSales", POSConstants.NET_SALES); //$NON-NLS-1$ 
		map.put("rowDiscount", "-" + POSConstants.DISCOUNT); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowSalesTax", "+" + POSConstants.SALES_TAX); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowSC", "+" + POSConstants.SERVICE_CHARGE); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowTotal", "=" + POSConstants.TOTAL); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowTips", "+" + POSConstants.TIPS); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowGReceivable", "=" + Messages.getString("grossReceivable")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowCReceipts", "-" + Messages.getString("cashReceipts")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowCCards", "-" + Messages.getString("creditCards")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowDCards", "-" + Messages.getString("debitCards")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowMPayments", "-" + Messages.getString("memberPayments")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowCPayments", "-" + Messages.getString("customPayments")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowGCert", "-" + Messages.getString("giftCertificate")); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
		map.put("rowRefundPlus", "+" + POSConstants.REFUND_BUTTON_TEXT); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("rowTolerance", "+" + Messages.getString("CurrencyExplorer.8")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowReceiptDiff", "=" + Messages.getString("receiptDifferential")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("rowCashTips", Messages.getString("TipsCashoutReportTableModel.9")); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put("rowChargedTips", Messages.getString("chargedTips")); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put(JRParameter.IS_IGNORE_PAGINATION, true);
		User assignedUser = cashDrawer.getAssignedUser();
		if (assignedUser != null) {
			String userInfo = assignedUser.getFullName();
			map.put("user", Messages.getString("PosPrintService.27") + userInfo); //$NON-NLS-1$ //$NON-NLS-2$
		}

		String storeSessionStartTime = DateUtil.formatFullDateAndTimeAsString(cashDrawer.getStartTime());
		if (cashDrawer.getReportTime() != null) {
			map.put("date", //$NON-NLS-1$
					"<b>" + Messages.getString("FROM") + "</b> " + storeSessionStartTime + " <b>" + Messages.getString("TO") + "</b> " //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
							+ DateUtil.formatFullDateAndTimeAsString(cashDrawer.getReportTime()));
		}
		else {
			map.put("date", "<b>" + Messages.getString("FROM") + "</b> " + storeSessionStartTime + "&#10;" + Messages.getString("TO") + "</b> " //$NON-NLS-6$//$NON-NLS-7$ //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ 
					+ DateUtil.formatFullDateAndTimeAsString(new Date()));
		}

		map.put("totalVoid", cashDrawer.getTotalVoid()); //$NON-NLS-1$
		map.put("declaredTips", cashDrawer.getDeclaredTips()); //$NON-NLS-1$

		JasperReport subReportCurrencyBalance = ReportUtil.getReport("drawer-currency-balance"); //$NON-NLS-1$
		JasperReport subReport = ReportUtil.getReport("drawer-pull-void-veport"); //$NON-NLS-1$
		JasperReport catBreakOutSubReport = ReportUtil.getReport("category-break-out-sub-report");//$NON-NLS-1$

		map.put("currencyBalanceReport", subReportCurrencyBalance); //$NON-NLS-1$
		map.put("subreportParameter", subReport); //$NON-NLS-1$
		map.put("catBreakOutSubReport", catBreakOutSubReport); //$NON-NLS-1$
		map.put("TitleCatBreakOut", Messages.getString("CategoryBreakout"));//$NON-NLS-1$ //$NON-NLS-2$

		JasperReport mainReport = ReportUtil.getReport("server-summary-report_72"); //$NON-NLS-1$

		JRBeanCollectionDataSource dataSource = new JRBeanCollectionDataSource(Arrays.asList(new CashDrawer[] { cashDrawer }));
		JasperPrint jasperPrint = JasperFillManager.fillReport(mainReport, map, dataSource);
		//TODO: handle exception
		jasperPrint.setProperty("printerName", DataProvider.get().getPrinters().getReceiptPrinter()); //$NON-NLS-1$
		jasperPrint.setName("ServerSummaryReport"); //$NON-NLS-1$
		return jasperPrint;
	}

	public static JasperPrint populateTerminalStatusReport(CashDrawer cashDrawer, boolean staffBankReport, HashMap<String, Object> parameters)
			throws JRException {
		parameters.put(JRParameter.IS_IGNORE_PAGINATION, true);
		User assignedUser = cashDrawer.getAssignedUser();
		if (assignedUser != null) {
			String userInfo = assignedUser.getFullName();
			parameters.put("user", (staffBankReport ? Messages.getString("PosPrintService.21") : Messages.getString("PosPrintService.4") + " ") + userInfo); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (cashDrawer.getStartTime() != null)
			parameters.put("startTime", Messages.getString("PosPrintService.24") + date_formatter.format(cashDrawer.getStartTime())); //$NON-NLS-1$ //$NON-NLS-2$
		parameters.put("date", Messages.getString("PosPrintService.25") + date_formatter.format(new Date())); //$NON-NLS-1$ //$NON-NLS-2$
		parameters.put("totalVoid", cashDrawer.getTotalVoid()); //$NON-NLS-1$

		parameters.put("declaredTips", cashDrawer.getDeclaredTips()); //$NON-NLS-1$

		JasperReport subReportCurrencyBalance = ReportUtil.getReport("drawer-currency-balance"); //$NON-NLS-1$
		JasperReport subReport = ReportUtil.getReport("drawer-pull-void-veport"); //$NON-NLS-1$

		parameters.put("currencyBalanceReport", subReportCurrencyBalance); //$NON-NLS-1$
		parameters.put("subreportParameter", subReport); //$NON-NLS-1$

		JasperReport mainReport = ReportUtil.getReport("terminal_status_report_72"); //$NON-NLS-1$

		JRBeanCollectionDataSource dataSource = new JRBeanCollectionDataSource(Arrays.asList(new CashDrawer[] { cashDrawer }));
		JasperPrint jasperPrint = JasperFillManager.fillReport(mainReport, parameters, dataSource);
		jasperPrint.setProperty("printerName", DataProvider.get().getPrinters().getReceiptPrinter()); //$NON-NLS-1$
		jasperPrint.setName("TerminalStatusReport" + cashDrawer.getTerminalId()); //$NON-NLS-1$
		return jasperPrint;
	}

	public static CashDrawer populateCashDrawerReportSummary(List<CashDrawer> reportList) {
		CashDrawer cashDrawersReportSummary = new CashDrawer();
		List<String> cashDrawerIds = new ArrayList<>();
		if (reportList != null) {
			for (CashDrawer report : reportList) {
				if (report.getReportTime() == null) {
					try {
						CashDrawerReportService reportService2 = new CashDrawerReportService(report);
						reportService2.populateReport();
					} catch (Exception e) {
					}
				}
				cashDrawerIds.add(report.getId());

				cashDrawersReportSummary.setBeginCash(cashDrawersReportSummary.getBeginCash() + report.getBeginCash());
				cashDrawersReportSummary.setNetSales(cashDrawersReportSummary.getNetSales() + report.getNetSales());
				cashDrawersReportSummary.setSalesTax(cashDrawersReportSummary.getSalesTax() + report.getSalesTax());
				cashDrawersReportSummary.setSalesDeliveryCharge(cashDrawersReportSummary.getSalesDeliveryCharge() + report.getSalesDeliveryCharge());

				cashDrawersReportSummary.setTotalVoid(cashDrawersReportSummary.getTotalVoid() + report.getTotalVoid());
				cashDrawersReportSummary.setTotalVoidWst(cashDrawersReportSummary.getTotalVoidWst() + report.getTotalVoidWst());
				cashDrawersReportSummary.setTipsPaid(cashDrawersReportSummary.getTipsPaid() + report.getTipsPaid());
				cashDrawersReportSummary.setTotalDiscountCount(cashDrawersReportSummary.getTotalDiscountCount() + report.getTotalDiscountCount());
				cashDrawersReportSummary.setTotalDiscountAmount(cashDrawersReportSummary.getTotalDiscountAmount() + report.getTotalDiscountAmount());

				cashDrawersReportSummary.setCashReceiptCount(cashDrawersReportSummary.getCashReceiptCount() + report.getCashReceiptCount());
				cashDrawersReportSummary.setCashReceiptAmount(cashDrawersReportSummary.getCashReceiptAmount() + report.getCashReceiptAmount());
				cashDrawersReportSummary.setCreditCardReceiptCount(cashDrawersReportSummary.getCreditCardReceiptCount() + report.getCreditCardReceiptCount());
				cashDrawersReportSummary
						.setCreditCardReceiptAmount(cashDrawersReportSummary.getCreditCardReceiptAmount() + report.getCreditCardReceiptAmount());
				cashDrawersReportSummary.setDebitCardReceiptCount(cashDrawersReportSummary.getDebitCardReceiptCount() + report.getDebitCardReceiptCount());
				cashDrawersReportSummary.setDebitCardReceiptAmount(cashDrawersReportSummary.getDebitCardReceiptAmount() + report.getDebitCardReceiptAmount());
				cashDrawersReportSummary.setGiftCertReturnCount(cashDrawersReportSummary.getGiftCertReturnCount() + report.getGiftCertReturnCount());
				cashDrawersReportSummary.setGiftCertReturnAmount(cashDrawersReportSummary.getGiftCertReturnAmount() + report.getGiftCertReturnAmount());
				cashDrawersReportSummary.setGiftCertChangeAmount(cashDrawersReportSummary.getGiftCertChangeAmount() + report.getGiftCertChangeAmount());
				cashDrawersReportSummary.setCustomPaymentCount(cashDrawersReportSummary.getCustomPaymentCount() + report.getCustomPaymentCount());
				cashDrawersReportSummary.setCustomPaymentAmount(cashDrawersReportSummary.getCustomPaymentAmount() + report.getCustomPaymentAmount());
				cashDrawersReportSummary.setAdvancePaymentAmount(cashDrawersReportSummary.getAdvancePaymentAmount() + report.getAdvancePaymentAmount());
				cashDrawersReportSummary.setRefundReceiptCount(cashDrawersReportSummary.getRefundReceiptCount() + report.getRefundReceiptCount());
				cashDrawersReportSummary.setRefundAmount(cashDrawersReportSummary.getRefundAmount() + report.getRefundAmount());
				cashDrawersReportSummary.setPayOutCount(cashDrawersReportSummary.getPayOutCount() + report.getPayOutCount());
				cashDrawersReportSummary.setPayOutAmount(cashDrawersReportSummary.getPayOutAmount() + report.getPayOutAmount());
				cashDrawersReportSummary.setDrawerBleedCount(cashDrawersReportSummary.getDrawerBleedCount() + report.getDrawerBleedCount());
				cashDrawersReportSummary.setDrawerBleedAmount(cashDrawersReportSummary.getDrawerBleedAmount() + report.getDrawerBleedAmount());
				cashDrawersReportSummary.setCashTips(cashDrawersReportSummary.getCashTips() + report.getCashTips());
				cashDrawersReportSummary.setChargedTips(cashDrawersReportSummary.getChargedTips() + report.getChargedTips());
				cashDrawersReportSummary.setCashBack(cashDrawersReportSummary.getCashBack() + report.getCashBack());

				cashDrawersReportSummary.setCashToDeposit(cashDrawersReportSummary.getCashToDeposit() + report.getCashToDeposit());
				//				cashDrawersReportSummary.setStoreSession(report.getStoreSession());
			}
		}
		cashDrawersReportSummary.calculate();
		cashDrawersReportSummary.setVoidEntries(new HashSet<>(VoidItemDAO.getInstance().getVoidEntries(cashDrawerIds)));
		return cashDrawersReportSummary;
	}

	@Deprecated
	public static void printCashDrawerReportSummary(StoreSession currentData) {
		List<CashDrawer> drawerReports = CashDrawerDAO.getInstance().findByStoreOperationData(currentData, false);
		CashDrawer cashDrawersReportSummary = populateCashDrawerReportSummary(drawerReports);
		cashDrawersReportSummary.setStartTime(currentData.getOpenTime());
		cashDrawersReportSummary.setAssignedBy(currentData.getOpenedBy());
		cashDrawersReportSummary.setReportTime(currentData.getCloseTime());
		cashDrawersReportSummary.setClosedBy(currentData.getClosedBy());
		printCashDrawerSummaryReport(cashDrawersReportSummary);
	}

	public static JasperPrint createSessionSummaryReportPrint(StoreSession currentSession) throws Exception {
		CashDrawer sessionSummaryReport = StoreSessionSummaryReportService.createSessionSummaryReport(currentSession);
		JasperPrint jasperPrint = PosPrintService.populateStoreSessionSummaryReportParameters(sessionSummaryReport);

		return jasperPrint;
	}

	public static void printStoreSessionSummaryReport(StoreSession currentSession) throws Exception {
		JasperPrint jasperPrint = createSessionSummaryReportPrint(currentSession);
		try {
			PosPrintService.printReportByJasperPrint(jasperPrint);
		} catch (PrinterIsNotConfiguredException e) {
			throw new PosException(Messages.getString("DrawerAndStaffBankReportDialog.14") + e.getMessage()); //$NON-NLS-1$
		}
	}

	public static CashDrawer populateCloudCashDrawerReportSummary(List<CashDrawer> reportList) {
		CashDrawer cashDrawersReportSummary = new CashDrawer();
		Map<String, CashBreakdown> summaryCashBreakdownMap = new HashMap<>();
		List<String> cashDrawerIds = new ArrayList<>();
		if (reportList != null) {
			for (CashDrawer report : reportList) {
				cashDrawerIds.add(report.getId());
				cashDrawersReportSummary.setBeginCash(cashDrawersReportSummary.getBeginCash() + report.getBeginCash());
				cashDrawersReportSummary.setNetSales(cashDrawersReportSummary.getNetSales() + report.getNetSales());
				cashDrawersReportSummary.setSalesTax(cashDrawersReportSummary.getSalesTax() + report.getSalesTax());
				cashDrawersReportSummary.setSalesDeliveryCharge(cashDrawersReportSummary.getSalesDeliveryCharge() + report.getSalesDeliveryCharge());
				//				cashDrawersReportSummary.setGratuityAmount(cashDrawersReportSummary.getGratuityAmount() + report.getCashTips()+ report.getChargedTips());

				cashDrawersReportSummary.setTotalVoid(cashDrawersReportSummary.getTotalVoid() + report.getTotalVoid());
				cashDrawersReportSummary.setTotalVoidWst(cashDrawersReportSummary.getTotalVoidWst() + report.getTotalVoidWst());
				cashDrawersReportSummary.setTipsPaid(cashDrawersReportSummary.getTipsPaid() + report.getTipsPaid());
				cashDrawersReportSummary.setTotalDiscountCount(cashDrawersReportSummary.getTotalDiscountCount() + report.getTotalDiscountCount());
				cashDrawersReportSummary.setTotalDiscountAmount(cashDrawersReportSummary.getTotalDiscountAmount() + report.getTotalDiscountAmount());

				cashDrawersReportSummary.setCashReceiptCount(cashDrawersReportSummary.getCashReceiptCount() + report.getCashReceiptCount());
				cashDrawersReportSummary.setCashReceiptAmount(cashDrawersReportSummary.getCashReceiptAmount() + report.getCashReceiptAmount());
				cashDrawersReportSummary.setCreditCardReceiptCount(cashDrawersReportSummary.getCreditCardReceiptCount() + report.getCreditCardReceiptCount());
				cashDrawersReportSummary
						.setCreditCardReceiptAmount(cashDrawersReportSummary.getCreditCardReceiptAmount() + report.getCreditCardReceiptAmount());
				cashDrawersReportSummary.setDebitCardReceiptCount(cashDrawersReportSummary.getDebitCardReceiptCount() + report.getDebitCardReceiptCount());
				cashDrawersReportSummary.setDebitCardReceiptAmount(cashDrawersReportSummary.getDebitCardReceiptAmount() + report.getDebitCardReceiptAmount());
				cashDrawersReportSummary.setGiftCertReturnCount(cashDrawersReportSummary.getGiftCertReturnCount() + report.getGiftCertReturnCount());
				cashDrawersReportSummary.setGiftCertReturnAmount(cashDrawersReportSummary.getGiftCertReturnAmount() + report.getGiftCertReturnAmount());
				cashDrawersReportSummary.setGiftCertChangeAmount(cashDrawersReportSummary.getGiftCertChangeAmount() + report.getGiftCertChangeAmount());
				cashDrawersReportSummary.setCustomPaymentCount(cashDrawersReportSummary.getCustomPaymentCount() + report.getCustomPaymentCount());
				cashDrawersReportSummary.setCustomPaymentAmount(cashDrawersReportSummary.getCustomPaymentAmount() + report.getCustomPaymentAmount());
				cashDrawersReportSummary.setAdvancePaymentAmount(cashDrawersReportSummary.getAdvancePaymentAmount() + report.getAdvancePaymentAmount());
				cashDrawersReportSummary.setRefundReceiptCount(cashDrawersReportSummary.getRefundReceiptCount() + report.getRefundReceiptCount());
				cashDrawersReportSummary.setRefundAmount(cashDrawersReportSummary.getRefundAmount() + report.getRefundAmount());
				cashDrawersReportSummary.setPayOutCount(cashDrawersReportSummary.getPayOutCount() + report.getPayOutCount());
				cashDrawersReportSummary.setPayOutAmount(cashDrawersReportSummary.getPayOutAmount() + report.getPayOutAmount());
				cashDrawersReportSummary.setDrawerBleedCount(cashDrawersReportSummary.getDrawerBleedCount() + report.getDrawerBleedCount());
				cashDrawersReportSummary.setDrawerBleedAmount(cashDrawersReportSummary.getDrawerBleedAmount() + report.getDrawerBleedAmount());
				cashDrawersReportSummary.setCashTips(cashDrawersReportSummary.getCashTips() + report.getCashTips());
				cashDrawersReportSummary.setChargedTips(cashDrawersReportSummary.getChargedTips() + report.getChargedTips());
				cashDrawersReportSummary.setCashBack(cashDrawersReportSummary.getCashBack() + report.getCashBack());

				cashDrawersReportSummary.setCashToDeposit(cashDrawersReportSummary.getCashToDeposit() + report.getCashToDeposit());

				List<CashBreakdown> cashBreakdowns = report.getCashBreakdownList();
				if (cashBreakdowns != null) {
					for (CashBreakdown cashBreakdown : cashBreakdowns) {
						CashBreakdown breakdown = summaryCashBreakdownMap.get(cashBreakdown.getCurrency().getName());
						if (breakdown == null) {
							breakdown = new CashBreakdown();
							summaryCashBreakdownMap.put(cashBreakdown.getCurrency().getName(), breakdown);
						}
						breakdown.setBalance(breakdown.getBalance() + cashBreakdown.getBalance());
						breakdown.setBalance(breakdown.getTotalAmount() + cashBreakdown.getTotalAmount());
					}
				}
				//				cashDrawersReportSummary.setStoreSession(report.getStoreSession());
			}
		}
		cashDrawersReportSummary.calculate();
		cashDrawersReportSummary.setVoidEntries(new HashSet<>(VoidItemDAO.getInstance().getVoidEntries(cashDrawerIds)));
		return cashDrawersReportSummary;
	}

	public static void printClockInOutReceipt(User user) throws Exception {
		Date currentTime = new Date();
		Store store = DataProvider.get().getStore();

		SimpleDateFormat dateFormat = new SimpleDateFormat("dd MMM, yyyy"); //$NON-NLS-1$
		SimpleDateFormat dateFormat2 = new SimpleDateFormat("dd MMM, hh:mm aaa"); //$NON-NLS-1$
		SimpleDateFormat timeFormat = new SimpleDateFormat("hh:mm:ss aaa"); //$NON-NLS-1$

		String message = ""; //$NON-NLS-1$
		if (user.isClockedIn()) {
			message = Messages.getString("PosPrintService.36") + timeFormat.format(currentTime); //$NON-NLS-1$
		}
		else {
			message = Messages.getString("PosPrintService.37") + timeFormat.format(currentTime); //$NON-NLS-1$
		}

		String sep = "\n......................................................................."; //$NON-NLS-1$
		String data = "\n" + store.getName(); //$NON-NLS-1$
		data += "\n" + store.getAddressLine1(); //$NON-NLS-1$
		data += sep;
		data += "\n-" + user.getFullName() + " (#" + user.getId() + ")" + (user.isClockedIn() ? " clock in" : " clock out") + "-"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
		data += sep;
		data += Messages.getString("PosPrintService.47") + dateFormat.format(currentTime); //$NON-NLS-1$
		data += "\n" + message; //$NON-NLS-1$
		data += Messages.getString("PosPrintService.49") + dateFormat2.format(new Date()); //$NON-NLS-1$
		data += sep;

		HashMap<String, Object> map = new HashMap<String, Object>(2);
		map.put("title", ""); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("data", data); //$NON-NLS-1$
		JasperReport report = ReportUtil.getReport("generic-receipt");//$NON-NLS-1$
		JasperPrint jasperPrint = JasperFillManager.fillReport(report, map, new JREmptyDataSource());
		printReportByJasperPrint(jasperPrint);
	}

}
