/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuShift;
import com.floreantpos.model.PriceRule;
import com.floreantpos.model.PriceShift;
import com.floreantpos.model.Shift;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ShiftDAO extends BaseShiftDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ShiftDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		if (obj instanceof Shift) {
			Shift shift = (Shift) obj;
			if (shift instanceof MenuShift) {
				List<MenuCategory> categories = MenuCategoryDAO.getInstance().findByMenuShiftId(shift.getId(), session);
				if (categories != null && !categories.isEmpty()) {
					String details = ""; //$NON-NLS-1$
					details = Messages.getString("MenuItemDAO.222"); //$NON-NLS-1$
					int count = 1;
					for (MenuCategory category : categories) {
						details += "\n" + count + ". " + category.getName(); //$NON-NLS-1$ //$NON-NLS-2$
						count++;
					}
					throw new PosException(Messages.getString("ShiftDAO.0"), details); //$NON-NLS-1$
				}
			}
			else if (shift instanceof PriceShift) {
				List<PriceRule> priceRules = PriceRuleDAO.getInstance().findByPriceShiftId(shift.getId(), session, Boolean.TRUE);
				if (priceRules != null && !priceRules.isEmpty()) {
					String details = ""; //$NON-NLS-1$
					details = Messages.getString("MenuItemDAO.333"); //$NON-NLS-1$
					int count = 1;
					for (PriceRule priceRule : priceRules) {
						details += "\n" + count + ". " + priceRule.getName(); //$NON-NLS-1$ //$NON-NLS-2$
						count++;
					}
					throw new PosException(Messages.getString("ShiftDAO.1"), details); //$NON-NLS-1$
				}
			}
			shift.setDeleted(Boolean.TRUE);
			super.update(shift, session);
		}
		else {
			throw new PosException(Messages.getString("ShiftDAO.2")); //$NON-NLS-1$
		}
	}

	public List<Shift> findAllActive() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(PriceShift.PROP_ENABLE, Boolean.TRUE));
			criteria.addOrder(Order.asc(PriceShift.PROP_PRIORITY));
			return criteria.list();

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public boolean exists(String shiftName) throws PosException {
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Shift.PROP_NAME, shiftName));
			List list = criteria.list();
			if (list != null && list.size() > 0) {
				return true;
			}
			return false;
		} catch (Exception e) {
			throw new PosException(Messages.getString("ShiftDAO.14"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
				}
			}
		}
	}

	public Shift getByName(String shiftName) throws PosException {
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Shift.PROP_NAME, shiftName));
			List<Shift> list = criteria.list();
			if (list != null && list.size() > 0) {
				return list.get(0);
			}
			return null;
		} catch (Exception e) {
			throw new PosException(Messages.getString("ShiftDAO.15"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
				}
			}
		}
	}

	public void refresh(Shift shift) throws PosException {
		Session session = null;

		try {
			session = createNewSession();
			session.refresh(shift);
		} catch (Exception e) {
			throw new PosException(Messages.getString("ShiftDAO.16"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
				}
			}
		}
	}

	public Shift findByName(String name) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Shift.PROP_NAME, name));
			return (Shift) criteria.uniqueResult();
		} catch (Exception e) {
			throw e;
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
				}
			}
		}
	}

	public void saveOrUpdateShifts(List<Shift> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<Shift> iterator = dataList.iterator(); iterator.hasNext();) {
				Shift item = (Shift) iterator.next();
				Shift existingItem = get(item.getId(), item.getOutletId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public Shift get(String shiftId, String outletId) {
		return super.get(new Shift(shiftId, outletId));
	}
}