package com.floreantpos.report;

import java.io.Serializable;
import java.util.Date;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;

import com.floreantpos.model.Agent;
import com.floreantpos.model.AgentTypeEnum;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.util.DateUtil;

public class ReferralCommissionReportData implements Serializable {

	private String id;
	private String orderId;
	private String orderDate;
	private String outletId;
	private String referralType;

	private String patientName;
	private String patientId;
	private String patientMobileNo;
	private String referralId;
	private String referralBy;
	private double grossSale;
	private double discount;
	private double receivable;
	private double due;
	private double dueColl;
	private double received;
	private String reportDoctor;
	private double rfOnNetSales;
	private double rfOnReport;
	private double totalRF;
	private double rfPayableAmount;
	private double taxAmount;
	private double agentCommissionTotal;
	private double labDoctorFee;
	private double rfPaid;
	private double labDoctorFeePaidAmount;
	private boolean isHeaderRow;
	private double officeNet;
	private double officeTotal;

	public ReferralCommissionReportData() {

	}

	public ReferralCommissionReportData(Ticket ticket) {
		double receivablelAmount = ticket.getTotalAmount();
		setOrderId(ticket.getId());
		setOrderDate(DateUtil.formatAsShortDate(DateUtil.convertServerTimeToBrowserTime(ticket.getCreateDate())));
		setOutletId(ticket.getOutletId());
		setGrossSale(ticket.getSubtotalAmount());
		setDiscount(ticket.getDiscountAmount());
		setReceivable(receivablelAmount);
		setDue(ticket.getDueAmount());
		setReceived(ticket.getPaidAmountAfterRefund());
		setTaxAmount(ticket.getTaxAmount());
		setLabDoctorFee(ticket.getLabDoctorFee());
		setRfPaid(ticket.getReferrerFeePaidAmount());
		setLabDoctorFeePaidAmount(ticket.getLabDoctorFeePaidAmount());

		double rfOnReport = ticket.getReferrerFeeOnReport();
		double rfOnNetSales = ticket.getReferrerFeeOnNetSales();
		setRfOnNetSales(rfOnNetSales);
		setRfOnReport(rfOnReport);
		setTotalRF(rfOnNetSales + rfOnReport);
		setRfPayableAmount(ticket.getRfPayableAmount());
		setAgentCommissionTotal(rfOnNetSales + rfOnReport);

		calculateDueCollectionAmount(ticket);

		String agentId = ticket.getReferrerId();
		setReferralId(agentId);
		if (StringUtils.isNotBlank(agentId)) {
			Customer referrer = CustomerDAO.getInstance().get(agentId);
			if (referrer != null) {
				setReferralBy(referrer.getName());
				if (referrer instanceof Agent) {
					String agentType = referrer.getAgentType();
					if (StringUtils.isNotBlank(agentType)) {
						setReferralType(AgentTypeEnum.fromString(agentType).getDisplayString());
					}
					else {
						setReferralType("Other");
					}
				}
				else if (referrer instanceof Doctor) {
					if (referrer.getIndorDoctor()) {
						setReferralType(AgentTypeEnum.INDOOR_DOCTOR.getDisplayString());
					}
					else {
						setReferralType(AgentTypeEnum.OUTDOOR_DOCTOR.getDisplayString());
					}
				}

				calculateDueCollectionAmount(ticket);

			}
			else {
				setReferralType("Self");
				setReferralBy(""); //$NON-NLS-1$
			}
		}
		else {
			setReferralType("Self");
			setReferralBy(""); //$NON-NLS-1$
		}

		String doctorId = ticket.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			Doctor doctor = DoctorDAO.getInstance().get(doctorId);
			if (doctor != null) {
				setReportDoctor(doctor.getName());
			}
		}

		Customer customer = ticket.getCustomer();
		if (customer != null) {
			setPatientName(customer.getName());
			setPatientMobileNo(customer.getMobileNo());
			setPatientId(customer.getId());
		}

		calculateOfficeAmount();
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getReferralType() {
		return referralType;
	}

	public void setReferralType(String referralType) {
		this.referralType = referralType;
	}

	public String getPatientName() {
		return patientName;
	}

	public void setPatientName(String patientName) {
		this.patientName = patientName;
	}

	public String getPatientId() {
		return patientId;
	}

	public void setPatientId(String patientId) {
		this.patientId = patientId;
	}

	public String getReferralId() {
		return referralId;
	}

	public void setReferralId(String referralId) {
		this.referralId = referralId;
	}

	public String getReferralBy() {
		return referralBy;
	}

	public void setReferralBy(String referralBy) {
		this.referralBy = referralBy;
	}

	public double getGrossSale() {
		return grossSale;
	}

	public void setGrossSale(double subtotal) {
		this.grossSale = subtotal;
	}

	public double getDiscount() {
		return discount;
	}

	public void setDiscount(double discount) {
		this.discount = discount;
	}

	public double getReceivable() {
		return receivable;
	}

	public void setReceivable(double receivable) {
		this.receivable = receivable;
	}

	public double getDue() {
		return due;
	}

	public void setDue(double due) {
		this.due = due;
	}

	public double getDueColl() {
		return dueColl;
	}

	public void setDueColl(double previousDueColl) {
		this.dueColl = previousDueColl;
	}

	public double getReceived() {
		return received;
	}

	public void setReceived(double received) {
		this.received = received;
	}

	public String getReportDoctor() {
		return reportDoctor;
	}

	public void setReportDoctor(String reportDoctor) {
		this.reportDoctor = reportDoctor;
	}

	public double getRfOnNetSales() {
		return rfOnNetSales;
	}

	public void setRfOnNetSales(double agentCommissionOnNetSales) {
		this.rfOnNetSales = agentCommissionOnNetSales;
	}

	public double getRfOnReport() {
		return rfOnReport;
	}

	public void setRfOnReport(double agentCommissionOnReport) {
		this.rfOnReport = agentCommissionOnReport;
	}

	public double getTotalRF() {
		return totalRF;
	}

	public void setTotalRF(double totalRF) {
		this.totalRF = totalRF;
	}

	public double getRfPayableAmount() {
		return rfPayableAmount;
	}

	public void setRfPayableAmount(double rfPayableAmount) {
		this.rfPayableAmount = rfPayableAmount;
	}

	public double getTaxAmount() {
		return taxAmount;
	}

	public void setTaxAmount(double taxAmount) {
		this.taxAmount = taxAmount;
	}

	public double getAgentCommissionTotal() {
		return agentCommissionTotal;
	}

	public void setAgentCommissionTotal(double agentFee) {
		this.agentCommissionTotal = agentFee;
	}

	public double getLabDoctorFee() {
		return labDoctorFee;
	}

	public void setLabDoctorFee(double labDoctorFee) {
		this.labDoctorFee = labDoctorFee;
	}

	public double getRfPaid() {
		return rfPaid;
	}

	public void setRfPaid(double rfPaid) {
		this.rfPaid = rfPaid;
	}

	public double getRfFeeDue() {
		return rfPayableAmount - rfPaid;
	}

	public double getLabDoctorFeePaidAmount() {
		return labDoctorFeePaidAmount;
	}

	public void setLabDoctorFeePaidAmount(double labDoctorFeePaidAmount) {
		this.labDoctorFeePaidAmount = labDoctorFeePaidAmount;
	}

	public double getLabDoctorFeeDueAmount() {
		return labDoctorFee - labDoctorFeePaidAmount;
	}

	public String getOrderId() {
		return orderId;
	}

	public void setOrderId(String id) {
		this.orderId = id;
	}

	public String getOutletId() {
		return outletId;
	}

	public void setOutletId(String outletId) {
		this.outletId = outletId;
	}

	public String getOrderDate() {
		return orderDate;
	}

	public void setOrderDate(String orderDate) {
		this.orderDate = orderDate;
	}

	public boolean isHeaderRow() {
		return isHeaderRow;
	}

	public void setHeaderRow(boolean isheaderRow) {
		this.isHeaderRow = isheaderRow;
	}

	public String getPatientMobileNo() {
		return patientMobileNo;
	}

	public void setPatientMobileNo(String patientMobileNo) {
		this.patientMobileNo = patientMobileNo;
	}

	public double getOfficeNet() {
		return officeNet;
	}

	public void setOfficeNet(double officeNet) {
		this.officeNet = officeNet;
	}

	public double getOfficeTotal() {
		return officeTotal;
	}

	public void setOfficeTotal(double officeTotal) {
		this.officeTotal = officeTotal;
	}

	public void calculateOfficeAmount() {
		officeNet = receivable - (labDoctorFee + rfOnReport + rfOnNetSales + taxAmount);
		//		officeTotal = receivable + getRfFeeDue();
		//		officeTotal = received - (labDoctorFee + rfOnReport + rfOnNetSales + taxAmount) + getRfFeeDue();

		officeTotal = officeNet + getRfFeeDue();

		if (officeTotal > received) {
			officeTotal = received;
		}
	}

	public void calculateDueCollectionAmount(Ticket ticket) {
		Set<PosTransaction> transactions = ticket.getTransactions();
		if (transactions == null || transactions.isEmpty()) {
			return;
		}
		Date tDate = ticket.getCreateDate();
		double dueCollectionAmount = 0;
		for (PosTransaction posTransaction : transactions) {
			if (posTransaction.isVoided()) {
				continue;
			}
			Date startOfDay = DateUtil.startOfDay(posTransaction.getTransactionTime());
			if (tDate.before(startOfDay)) {
				if (posTransaction instanceof RefundTransaction) {
					dueCollectionAmount += (-1) * posTransaction.getAmount();
				}
				else {
					dueCollectionAmount += posTransaction.getAmount();
				}
			}
		}
		setDueColl(dueCollectionAmount);
	}

}
