/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Set;

import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseUser;
import com.floreantpos.model.dao.CashDrawerDAO;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.AESencrp;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "image", "status", "parentUser", "linkedUser", "roles", "roleIds", "currentCashDrawer",
		"activeDrawerPullReport", "type", "typeName", "currentShift", "manager", "administrator" })
@XmlRootElement(name = "user")
public class User extends BaseUser implements TimedModel {
	private static final long serialVersionUID = 1L;
	private transient JsonObject propertiesContainer;
	private String typeName;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public User() {
		super();
	}

	/**
	 * Constructor for primary key
	 */
	public User(java.lang.String id, java.lang.String outletId) {

		super(id, outletId);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public final static String USER_TYPE_MANAGER = Messages.getString("User.0"); //$NON-NLS-1$
	public final static String USER_TYPE_CASHIER = Messages.getString("User.1"); //$NON-NLS-1$
	public final static String USER_TYPE_SERVER = Messages.getString("User.2"); //$NON-NLS-1$
	public static final String PROP_ID = "id"; //$NON-NLS-1$
	public static final String PROP_OUTLET_ID = "outletId"; //$NON-NLS-1$
	private static final String JSON_PROP_REGULAR_WORK_HOUR = "regular_work_hour"; //$NON-NLS-1$
	private static final String JSON_PROP_REGULAR_WORK_HOUR_WEEKLY = "regular_work_hour_weekly"; //$NON-NLS-1$

	private CashDrawer currentCashDrawer;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@XmlTransient
	public UserType getType() {
		String userTypeId2 = getUserTypeId();
		if (userTypeId2 != null) {
			return DataProvider.get().getUserType(userTypeId2);
		}
		return null;
	}

	public void setType(UserType userType) {
		if (userType != null) {
			super.setUserTypeId(userType.getId());
		}
		else {
			super.setUserTypeId(null);
		}
	}

	@XmlTransient
	public Shift getCurrentShift() {
		return DataProvider.get().getShiftById(getCurrentShiftId(), getOutletId());
	}

	public void setCurrentShift(Shift shift) {
		if (shift != null) {
			super.setCurrentShiftId(shift.getId());
		}
		else {
			super.setCurrentShiftId(null);
		}
	}

	/**
	 * Return the value associated with the column: ACTIVE
	 */
	public java.lang.Boolean isActive() {
		return super.isActive() == null ? Boolean.TRUE : super.isActive();
	}

	public boolean hasPermission(UserPermission permission) {
		UserType userType = getType();
		if (userType == null) {
			return false;
		}
		return userType.hasPermission(permission);
	}

	public boolean hasPermissionInAnyRole(UserPermission permission) {
		User userRoleWithPermission = getUserRoleWithPermission(permission);
		return userRoleWithPermission != null;
	}

	public User getUserRoleWithPermission(UserPermission permission) {
		UserType userType = getType();
		if (userType != null && userType.hasPermission(permission)) {
			return this;
		}
		List<User> users = getRoles();
		for (User roledUser : users) {
			UserType roledUserType = roledUser.getType();
			if (roledUserType == null) {
				continue;
			}
			if (roledUserType.hasPermission(permission)) {
				return roledUser;
			}
		}
		return null;
	}

	@XmlTransient
	public CashDrawer getCurrentCashDrawer() {
		String currentCashDrawerId = super.getCurrentCashDrawerId();
		if (StringUtils.isBlank(currentCashDrawerId)) {
			return currentCashDrawer = null;
		}
		if (currentCashDrawer != null && currentCashDrawerId.equals(currentCashDrawer.getId()))
			return currentCashDrawer;
		return currentCashDrawer = CashDrawerDAO.getInstance().get(currentCashDrawerId);
	}

	public boolean isCashDrawerAssigned() {
		return getCurrentCashDrawer() != null;
	}

	public void setCurrentCashDrawer(CashDrawer currentCashDrawer) {
		this.currentCashDrawer = currentCashDrawer;
		setCurrentCashDrawerId(currentCashDrawer == null ? null : currentCashDrawer.getId());
	}

	public void doClockIn(Terminal terminal, Shift shift, Calendar currentTime) {
		UserDAO.getInstance().refresh(this);
		if (isClockedIn()) {
			throw new PosException(Messages.getString("User.3") + getFullName()); //$NON-NLS-1$
		}
		List<User> linkedUsers = getLinkedUser();
		if (linkedUsers != null) {
			for (User user : linkedUsers) {
				if (user.isDeleted()) {
					continue;
				}
				if (user.isClockedIn()) {
					throw new PosException(Messages.getString("User.3") + getFullName()); //$NON-NLS-1$
				}
			}
		}

		setClockedIn(true);
		setCurrentShift(shift);
		setLastClockInTime(currentTime.getTime());
		if (isDriver()) {
			setAvailableForDelivery(true);
		}

		AttendenceHistory attendenceHistory = new AttendenceHistory();
		attendenceHistory.setClockInTime(currentTime.getTime());
		attendenceHistory.setClockInHour(Short.valueOf((short) currentTime.get(Calendar.HOUR_OF_DAY)));
		attendenceHistory.setUser(this);
		attendenceHistory.setTerminal(terminal);
		attendenceHistory.setShift(shift);
		attendenceHistory.setOutletId(getOutletId());

		UserDAO.getInstance().saveClockIn(this, attendenceHistory, shift, currentTime);
	}

	public void doClockOut(AttendenceHistory attendenceHistory, Shift shift, Calendar currentTime) {
		setClockedIn(false);
		setCurrentShift(null);
		setLastClockInTime(null);
		setLastClockOutTime(null);
		if (isDriver()) {
			setAvailableForDelivery(false);
		}

		attendenceHistory.setClockedOut(true);
		attendenceHistory.setClockOutTime(currentTime.getTime());
		attendenceHistory.setClockOutHour(Short.valueOf((short) currentTime.get(Calendar.HOUR_OF_DAY)));

		UserDAO.getInstance().saveClockOut(this, attendenceHistory, shift);
	}

	public boolean canViewAllOpenTickets() {
		UserType userType = getType();
		if (userType == null) {
			return false;
		}

		Set<UserPermission> permissions = userType.getPermissions();
		if (permissions == null) {
			return false;
		}

		for (UserPermission permission : permissions) {
			if (permission.equals(UserPermission.EDIT_OTHER_USERS_TICKETS)) {
				return true;
			}
		}
		return false;
	}

	public boolean canViewAllCloseTickets() {
		UserType userType = getType();
		if (userType == null) {
			return false;
		}

		Set<UserPermission> permissions = userType.getPermissions();
		if (permissions == null) {
			return false;
		}

		for (UserPermission permission : permissions) {
			if (permission.equals(UserPermission.EDIT_OTHER_USERS_TICKETS)) {
				return true;
			}
		}

		return false;
	}

	public void setFullName(String str) {
	}

	public String getStatus() {
		if (isClockedIn()) {
			if (isAvailableForDelivery()) {
				return Messages.getString("User.5"); //$NON-NLS-1$
			}
			else {
				return Messages.getString("User.6"); //$NON-NLS-1$
			}
		}
		else {
			return Messages.getString("User.7"); //$NON-NLS-1$
		}
	}

	public String getFullName() {
		String name = ""; //$NON-NLS-1$
		if (StringUtils.isNotEmpty(getFirstName())) {
			name = getFirstName() + " "; //$NON-NLS-1$
		}

		if (StringUtils.isNotEmpty(getLastName())) {
			name += getLastName();
		}
		return name;
	}

	@Override
	public String toString() {
		return getFullName();
	}

	@XmlTransient
	public boolean isManager() {
		return hasPermission(UserPermission.PERFORM_MANAGER_TASK);
	}

	@XmlTransient
	public boolean isAdministrator() {
		return hasPermission(UserPermission.PERFORM_ADMINISTRATIVE_TASK);
	}

	@XmlTransient
	public User getParentUser() {
		String parentUserId = super.getParentUserId();
		if (StringUtils.isEmpty(parentUserId)) {
			return null;
		}
		return DataProvider.get().getUserById(parentUserId, getOutletId());
	}

	public void setParentUser(User user) {
		if (user != null) {
			setParentUserId(user.getId());
		}
		else {
			setParentUserId(null);
		}
	}

	public boolean hasLinkedUser() {
		List<User> linkedUser = getLinkedUser();
		if (linkedUser != null && linkedUser.size() > 0) {
			return true;
		}
		if (getParentUser() == null) {
			return false;
		}
		return getParentUser().hasLinkedUser();
	}

	/**
	 * This method is used to get all active role.
	 * (Deleted and de-active Role user should not be added in this list).
	 * 
	 */
	@XmlTransient
	public List<User> getRoles() {
		List<User> roleList = new ArrayList<>();
		List<User> linkedUser = getLinkedUser();

		if (!isRoot()) {
			return getParentUser().getRoles();
		}

		if (linkedUser == null || linkedUser.isEmpty()) {
			roleList.add(this);
			return roleList;
		}

		if (this.getParentUserId() == null && !roleList.contains(this)) {
			roleList.add(this);
		}

		for (User user : linkedUser) {
			if (!user.isDeleted() && user.isActive()) {
				roleList.add(user);
			}
		}
		return roleList;
	}

	public void setEncryptedPassword(String plainText) {
		try {
			super.setPassword(AESencrp.encrypt(plainText));
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	@JsonIgnoreProperties
	@JsonIgnore
	public String getPasswordAsPlainText() {
		String secretKey = super.getPassword();
		if (StringUtils.isNotEmpty(secretKey)) {
			try {
				return AESencrp.decrypt(secretKey);
			} catch (Exception e) {
				//PosLog.error(getClass(), e);
			}
		}
		return secretKey;
	}

	public void setEncryptedSignatureApprovalCode(String PIN) {
		try {
			super.setSignatureApprovalCode(AESencrp.encrypt(PIN));
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	@JsonIgnoreProperties
	@JsonIgnore
	public String getSignatureApprovalCodeAsPlainText() {
		String PIN = super.getSignatureApprovalCode();
		if (StringUtils.isNotEmpty(PIN)) {
			try {
				return AESencrp.decrypt(PIN);
			} catch (Exception e) {
				//PosLog.error(getClass(), e);
			}
		}
		return PIN;
	}

	@JsonIgnoreProperties
	@JsonIgnore
	public String getLongPasswordAsPlainText() {
		String secretKey = super.getLongPassword();
		if (StringUtils.isNotEmpty(secretKey)) {
			try {
				return AESencrp.decrypt(secretKey);
			} catch (Exception e) {
				//PosLog.error(getClass(), e);
			}
		}
		return secretKey;
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	public void addProperty(String key, String value) {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		propertiesContainer.addProperty(key, value);
	}

	public String getProperty(String key) {
		if (propertiesContainer == null) {
			return null;
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	@XmlTransient
	public String getTypeName() {
		if (typeName != null) {
			return typeName;
		}
		UserType userType = getType();
		if (userType == null) {
			return null;
		}
		return userType.getName();
	}

	public void setTypeName(String typeName) {
		this.typeName = typeName;
	}

	/**
	 * This method is used to get  user id list including linkedUser id.
	 * 
	  */
	@XmlTransient
	public List<String> getRoleIds() {
		List<String> roleIdList = new ArrayList<>();
		List<User> linkedUserList = null;
		if (isRoot()) {
			roleIdList.add(getId());
			linkedUserList = getLinkedUser();
		}
		else {
			User rootUser = getParentUser();
			roleIdList.add(rootUser.getId());
			linkedUserList = rootUser.getLinkedUser();
		}
		if (linkedUserList != null) {
			for (User linkedUser : linkedUserList) {
				if (linkedUser.isActive()) {
					roleIdList.add(linkedUser.getId());
				}
			}
		}
		return roleIdList;
	}

	public void setRegularWorkHour(Double wHour) {
		addProperty(JSON_PROP_REGULAR_WORK_HOUR, String.valueOf(wHour));
	}

	@XmlTransient
	@JsonIgnore
	public int getRegularWorkHour() {
		String property = getProperty(JSON_PROP_REGULAR_WORK_HOUR);
		if (!isRoot() && StringUtils.isBlank(property)) {
			User parentUser = getParentUser();
			if (parentUser != null) {
				return parentUser.getRegularWorkHour();
			}
			return 8;
		}

		Number parseDouble;
		try {
			parseDouble = NumberUtil.parse(property);
			if (NumberUtil.isZero(parseDouble)) {
				return 8;
			}
			return parseDouble.intValue();
		} catch (ParseException e) {
			return 8;
		}
	}

	public void setRegularWorkHourWeekly(Double wHour) {
		addProperty(JSON_PROP_REGULAR_WORK_HOUR_WEEKLY, String.valueOf(wHour));
	}

	@XmlTransient
	@JsonIgnore
	public int getRegularWorkHourWeekly() {
		String property = getProperty(JSON_PROP_REGULAR_WORK_HOUR_WEEKLY);
		if (!isRoot() && StringUtils.isBlank(property)) {
			User parentUser = getParentUser();
			if (parentUser != null) {
				return parentUser.getRegularWorkHour();
			}
			return 40;
		}

		Number parseDouble;
		try {
			parseDouble = NumberUtil.parse(property);
			if (NumberUtil.isZero(parseDouble)) {
				return 40;
			}
			return parseDouble.intValue();
		} catch (ParseException e) {
			return 40;
		}
	}

	public double getCostPerHourRate() {
		Double perHour = getCostPerHour();
		if (!isRoot() && NumberUtil.isZero(perHour)) {
			User parentUser = getParentUser();
			if (parentUser != null) {
				return parentUser.getCostPerHour();
			}
		}
		return perHour;
	}

	public double getOvertimePerHourRate() {
		Double perHour = getOvertimeRatePerHour();
		if (!isRoot() && NumberUtil.isZero(perHour)) {
			User parentUser = getParentUser();
			if (parentUser != null) {
				return parentUser.getOvertimeRatePerHour();
			}
		}
		return perHour;
	}

	public void putDesignation(String doctorDesignation) {
		addProperty("user.designation", doctorDesignation); //$NON-NLS-1$
	}

	public String getDesignation() {
		String designation = getProperty("user.designation");
		return StringUtils.isNotBlank(designation) ? designation : "";
	}

	//	public void putLabStuff(boolean isLabStuff) {
	//		addProperty("user.lab_stuff", String.valueOf(isLabStuff)); //$NON-NLS-1$
	//	}
	//
	//	public Boolean isLabStuff() {
	//		return POSUtil.getBoolean(getProperty("user.lab_stuff"), false); //$NON-NLS-1$
	//	}

	//	public void putAccountsManager(boolean isAccountsManager) {
	//		addProperty("user.accounts_manager", String.valueOf(isAccountsManager)); //$NON-NLS-1$
	//	}
	//
	//	public Boolean isAccountsManager() {
	//		return POSUtil.getBoolean(getProperty("user.accounts_manager"), false); //$NON-NLS-1$
	//	}

	//	public void putExpenseOfficer(boolean isExpenseOfficer) {
	//		addProperty("user.expense_officer", String.valueOf(isExpenseOfficer)); //$NON-NLS-1$
	//	}
	//
	//	public Boolean isExpenseOfficer() {
	//		return POSUtil.getBoolean(getProperty("user.expense_officer"), false); //$NON-NLS-1$
	//	}

	//	public void putAccountsOwner(boolean isAccountsOwner) {
	//		addProperty("user.accounts_owner", String.valueOf(isAccountsOwner)); //$NON-NLS-1$
	//	}
	//
	//	public boolean isAccountsOwner() {
	//		return POSUtil.getBoolean(getProperty("user.accounts_owner"), false); //$NON-NLS-1$
	//	}

	public boolean isStoreOwner(Store store) {
		return store != null && store.getOwnerId() != null && store.getOwnerId().equals(getId());
	}

	public String getSignatureImageId() {
		return getProperty("signatureImageId"); //$NON-NLS-1$
	}

	public void setSignatureImageId(String imageId) {
		addProperty("signatureImageId", imageId); //$NON-NLS-1$
	}

	public void setAllowDigitalSignature(boolean allowDigitalSignature) {
		addProperty("allowDigitalSignature", String.valueOf(allowDigitalSignature)); //$NON-NLS-1$
	}

	public boolean isAllowDigitalSignature() {
		return POSUtil.getBoolean(getProperty("allowDigitalSignature"), false); //$NON-NLS-1$
	}

	public void putDeductionGsonData(String deductionGsonData) {
		addProperty("salary.deduction_list_data", deductionGsonData);
	}

	public String getDeductionGsonData() {
		String property = getProperty("salary.deduction_list_data");
		return StringUtils.isBlank(property) ? "" : property;
	}

	public void putSalesAdmin(boolean isSalesAdmin) {
		addProperty("user.sales_admin", String.valueOf(isSalesAdmin)); //$NON-NLS-1$
	}

	public boolean isSalesAdmin() {
		return POSUtil.getBoolean(getProperty("user.sales_admin"), false); //$NON-NLS-1$
	}

	public boolean isAccountsRoleActive() {
		boolean active = POSUtil.getBoolean(getProperty("isAccountsRoleActive"), true);
		return active && (isAccountsOwner() || isAccountsManager() || isExpenseOfficer());
	}

	public void putAccountsRoleActive(Boolean b) {
		addProperty("isAccountsRoleActive", b == null ? String.valueOf(false) : String.valueOf(b));
	}
}