package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Customer;
import com.floreantpos.model.CustomerGroup;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class CustomerGroupDAO extends BaseCustomerGroupDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public CustomerGroupDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		CustomerGroup customerGroup = (CustomerGroup) obj;
		if (customerGroup == null) {
			throw new PosException(Messages.getString("CustomerGroupDAO.0")); //$NON-NLS-1$
		}

		List<Customer> customers = CustomerDAO.getInstance().getCustomerByCustomerGroup(customerGroup, session);
		if (customers != null && !customers.isEmpty()) {
			String message = customerGroup.getName() + Messages.getString("CustomerGroupDAO.1"); //$NON-NLS-1$
			String details = this.constructExceptionDetailsByCustomer(customerGroup, customers);
			throw new PosException(message, details);
		}
		customerGroup.setDeleted(true);
		update(customerGroup, session);
	}

	@Override
	public List<CustomerGroup> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(Order.asc(CustomerGroup.PROP_NAME));
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	private String constructExceptionDetailsByCustomer(CustomerGroup customerGroup, List<Customer> customers) {
		if (customers != null && !customers.isEmpty()) {
			StringBuilder builder = new StringBuilder(customerGroup.getName() + " " + Messages.getString("CustomerGroupDAO.3")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < customers.size(); i++) {
				String message = (i + 1) + "." + " " + customers.get(i).getName(); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	public void initialize(CustomerGroup customerGroup) {
		if (customerGroup == null || customerGroup.getId() == null)
			return;

		if (Hibernate.isInitialized(customerGroup.getCustomers())) {
			return;
		}
		Session session = null;

		try {
			session = createNewSession();
			session.refresh(customerGroup);

			Hibernate.initialize(customerGroup.getCustomers());
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateCustomerGroupList(List<CustomerGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<CustomerGroup> iterator = dataList.iterator(); iterator.hasNext();) {
				CustomerGroup item = (CustomerGroup) iterator.next();
				CustomerGroup existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}
}