package com.orocube.rest.service.mqtt;

import java.util.Set;
import java.util.UUID;

import org.apache.commons.lang.StringUtils;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttCallbackExtended;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;
import org.json.JSONObject;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.hl7.model.Result;
import com.floreantpos.hl7.model.Test;
import com.floreantpos.model.Store;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.MqttCommand;
import com.floreantpos.util.POSUtil;

public class OroMqttClient extends OroMqttMessageReceiver implements MqttCallbackExtended {
	private static final String MQTT_SERVER_URL = "tcp://siiopa.com:1883"; //$NON-NLS-1$
	private MqttClient mqttClient;
	private int qos = 0;

	private static OroMqttClient instance;
	private String deviceId;

	private OroMqttClient() {
	}

	public static OroMqttClient getInstance() {
		if (instance == null) {
			instance = new OroMqttClient();
		}
		return instance;
	}

	public MqttClient getMqttClient() {
		return getInstance().mqttClient;
	}

	public boolean isConnectedToServer() {
		if (mqttClient == null) {
			return false;
		}
		if (mqttClient.isConnected())
			return true;
		return false;
	}

	public void initMqttClient(String deviceIdStr) throws Exception {
		this.deviceId = deviceIdStr;

		mqttClient = new MqttClient(MQTT_SERVER_URL, deviceId, new MemoryPersistence());
		MqttConnectOptions connOpts = new MqttConnectOptions();
		connOpts.setCleanSession(true);
		connOpts.setConnectionTimeout(2000);
		connOpts.setAutomaticReconnect(true);
		mqttClient.setCallback(this);
		mqttClient.connect(connOpts);
	}

	public void disconnect() {
		try {
			mqttClient.disconnect();
		} catch (MqttException e) {
		}
	}

	//	private void subscribeToPublicTopic() throws Exception {
	//		subscribe(getStoreUuid() + MqttCommand.TOPIC_PUBLIC, qos);
	//	}
	//	
	//	private void subscribeToStore(String topic, int qos) throws Exception {
	//		subscribe(getStoreUuid() + topic, qos);
	//	}

	//	private void subscribe(String topic, int qos) throws Exception {
	//		if (mqttClient == null) {
	//			return;
	//		}
	//		if (!mqttClient.isConnected()) {
	//			throw new PosException(Messages.getString("OroMqttClient.4")); //$NON-NLS-1$
	//		}
	//		mqttClient.subscribe(topic, 2, this);
	//		
	//		PosLog.debug(getClass(), "subscribed to topic: " + topic); //$NON-NLS-1$
	//	}

	protected void showMqttNotification(String msg) {
		PosLog.info(getClass(), msg);
	}

	public void publishOnThread(final String topic, final String msg) {
		//		new Thread(new Runnable() {
		//
		//			@Override
		//			public void run() {
		//				publishData(topic, msg);
		//			}
		//		}).start();
		publishData(topic, msg);
	}

	public void publishData(final String topic, final String msg) {
		publishData(topic, msg, false);
	}

	public void publishData(final String topic, final String msg, boolean compress) {
		publishData(getStoreUuid(), topic, msg, compress);
	}

	public void publishData(final String storeId, final String topic, final String msg, boolean compress) {
		try {
			if (mqttClient == null) {
				return;
			}
			if (!isConnectedToServer())
				return;

			PosLog.debug(getClass(), "publishing to topic: " + storeId + topic); //$NON-NLS-1$
			mqttClient.publish(storeId + topic, compress ? POSUtil.compress(msg) : msg.getBytes(), qos, false);
		} catch (Exception me) {
			PosLog.error(getClass(), "Notification service error. " + me.getMessage()); //$NON-NLS-1$
		}
	}

	public String getDeviceId() {
		return deviceId;
	}

	private String getStoreUuid() {
		Store store = DataProvider.get().getStore();
		if (store != null) {
			return store.getUuid() + "/"; //$NON-NLS-1$
		}

		return UUID.randomUUID().toString() + "/"; //$NON-NLS-1$
	}

	public void publishToOutlet(String topic, String message, String outletId) {
		try {
			if (StringUtils.isBlank(outletId)) {
				return;
			}
			OroMqttClient.getInstance().getMqttClient().publish(getStoreUuid() + outletId + "/" + topic, message.getBytes(), qos, false);
		} catch (Exception e) {
			PosLog.error(OroMqttMessagePublisher.class, e.getMessage());
		}
	}

	@Override
	public void connectionLost(Throwable arg0) {
		PosLog.debug(getClass(), "mqtt connection lost. client id: " + mqttClient.getClientId()); //$NON-NLS-1$
	}

	@Override
	public void deliveryComplete(IMqttDeliveryToken arg0) {
	}

	@Override
	public void connectComplete(boolean reconnect, String serverURI) {
		PosLog.debug(getClass(), "connected to mqtt server. client id: " + mqttClient.getClientId());
	}

	public void notifyDataUpdated(Class<?> beanClass) {
		if (beanClass == null) {
			return;
		}
		if (DataProvider.get().isCaching(beanClass)) {
			try {
				JSONObject jsonObject = new JSONObject();
				jsonObject.put("terminalKey", DataProvider.get().getMqttDeviceId()); //$NON-NLS-1$
				OroMqttClient.getInstance().publishData(MqttCommand.TOPIC_REFRESH_CACHE, jsonObject.toString());
			} catch (Exception e) {
				PosLog.error(getClass(), e);
			}
		}
	}

	public void notifyStoreClosed() {
		try {
			JSONObject jsonObject = new JSONObject();
			Terminal terminal = DataProvider.get().getCurrentTerminal();
			jsonObject.put("terminalKey", DataProvider.get().getMqttDeviceId()); //$NON-NLS-1$
			jsonObject.put("terminalName", terminal.getName()); //$NON-NLS-1$
			OroMqttClient.getInstance().publishData(MqttCommand.TOPIC_STORE_CLOSED, jsonObject.toString());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void subscribeToMedlogicsData(String storeId) {
		try {
			subscribe(storeId + "/" + MqttCommand.TOPIC_MEDLOGICS_DATA, qos);
		} catch (Exception e) {
			//PosLog.error(getClass(), "Mqtt connection failed"); //$NON-NLS-1$
		}
	}

	public void subscribeToMedlogicsGlobalTopics() {
		try {
			subscribe(MqttCommand.TOPIC_MEDLOGICS_NOTIFICATION, qos);
		} catch (Exception e) {
			//PosLog.error(getClass(), "Mqtt connection failed"); //$NON-NLS-1$
		}
	}

	private void subscribe(String topic, int qos) throws Exception {
		if (mqttClient == null || topic == null) {
			return;
		}
		if (!mqttClient.isConnected()) {
			throw new PosException(Messages.getString("OroMqttClient.4")); //$NON-NLS-1$
		}
		mqttClient.subscribe(topic, 2, this);

		PosLog.debug(getClass(), "subscribed to topic: " + topic); //$NON-NLS-1$
	}

	public void publishHl7Data(Test test, String msg) {
		StringBuilder builder = new StringBuilder();
		builder.append(msg + "\n"); //$NON-NLS-1$

		Set<String> keySet = test.resultmap.keySet();
		for (String key : keySet) {
			Result result = test.resultmap.get(key);
			builder.append(key + ":" + result.result + "\n"); //$NON-NLS-1$ //$NON-NLS-2$
		}

		publishData(MqttCommand.TOPIC_MEDLOGICS_DATA, builder.toString());
	}

}