package com.floreantpos.hl7;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Session;

import com.floreantpos.PosLog;
import com.floreantpos.hl7.model.Result;
import com.floreantpos.hl7.model.Test;
import com.floreantpos.jdbc.OroMultitenantConnectionProvider;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MachineResult;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemResult;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.GlobalConfigDAO;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.MachineResultDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.TestItemDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.dao.TicketItemDAO;
import com.floreantpos.model.ext.LabWorkStatus;
import com.floreantpos.model.util.InventoryUnitConversionRule;
import com.google.gson.Gson;

public class HL7Util {

	private final static SimpleDateFormat mythicFormatter = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss");
	private Session session;
	private String storeId;
	private String outletId;

	public HL7Util() {
	}

	public HL7Util(Session session, String storeId, String outletId) {
		this.session = session;
		this.storeId = storeId;
		this.outletId = outletId;

	}

	public synchronized void addResultToTicketItem(Test test, TicketItem ticketItem) throws Exception {
		Map<String, Result> machineResultmap = test.resultmap;
		List<TestItem> menuItemTestItems = getMenuItemTestItems(ticketItem.getMenuItemId());
		for (TestItem testItem : menuItemTestItems) {
			String machineCode = testItem.getMachineCode();
			if (StringUtils.isBlank(machineCode)) {
				continue;
			}

			Result machineResult = machineResultmap.get(machineCode);
			if (machineResult == null) {
				continue;
			}
			testItem.putOriginalMachineResult(machineResult.result);
			if (StringUtils.isBlank(testItem.getUnit())) {
				testItem.setUnit(machineResult.unit);
			}
			else {
				convertMachineResultToItemUnit(testItem, machineResult);
			}
			testItem.setResult(machineResult.result);
			testItem.putMachineName(test.machine);
			//mythic data
			if (StringUtils.isNotBlank(machineResult.normalLow) || StringUtils.isNotBlank(machineResult.normalHigh)) {
				testItem.setNormalValue(machineResult.normalLow + " - " + machineResult.normalHigh);
			}
			machineResultmap.remove(machineCode);
		}

		int sortOrder = menuItemTestItems.size();
		List<Result> remainResults = new ArrayList<Result>(machineResultmap.values());
		for (int i = 0; i < remainResults.size(); i++) {
			Result machineResult = remainResults.get(i);

			String machineCode = machineResult.testName;

			TestItem testItem = TestItemDAO.getInstance().findByMachineCode(machineCode, session);
			if (testItem == null) {
				testItem = new TestItem();
				testItem.setName(machineCode);
				testItem.setMachineCode(machineCode);
			}
			else {
				testItem.setLastSyncTime(null);
				testItem.setLastUpdateTime(null);
			}

			testItem.putMachineName(test.machine);
			testItem.setId(machineResult.resultNo);
			testItem.putOriginalMachineResult(machineResult.result);
			String testItemUnit = testItem.getUnit();
			if (StringUtils.isBlank(testItemUnit)) {
				testItem.setUnit(machineResult.unit);
			}
			else {
				convertMachineResultToItemUnit(testItem, machineResult);
			}
			testItem.setResult(machineResult.result);

			//mythic data
			if (StringUtils.isNotBlank(machineResult.normalLow) || StringUtils.isNotBlank(machineResult.normalHigh)) {
				testItem.setNormalValue(machineResult.normalLow + " - " + machineResult.normalHigh);
			}

			testItem.putShowTestItem(false);
			testItem.setSortOrder(++sortOrder);
			menuItemTestItems.add(testItem);
			//parsedResult += String.format("Result no: %s, name=%s, value: %s, unit: %s", result.resultNo, result.testName, result.result, result.unit);
			//parsedResult += "\n";
		}

		List<TestItemResult> testItemResults = ticketItem.getTestItemResults();
		testItemResults.clear();
		TestItemResult testItemResult = new TestItemResult();
		testItemResult.setSortOrder(testItemResults.size());
		testItemResult.setTestItems(menuItemTestItems);
		testItemResults.add(testItemResult);
		ticketItem.setTestItemResults(testItemResults);

		ticketItem.setTestItems(menuItemTestItems);
		ticketItem.putMachineResultReceived(true);
		ticketItem.putMachineResultTime(new Date());
		ticketItem.setLabWorkStatusValue(LabWorkStatus.RESULT_RECORDED);
	}

	public synchronized void addResultToTicketItem(String testId, TicketItem ticketItem) throws Exception {
		MachineResult machineResult = MachineResultDAO.getInstance().getOrderHistoryBySampleId(testId, session);
		addResultToTicketItem(ticketItem, machineResult);
	}

	public void addResultToTicketItem(TicketItem ticketItem, MachineResult machineResult) throws Exception {
		ticketItem.setTestItemResults(Collections.emptyList());
		if (machineResult != null && ticketItem != null) {
			String resultJson = machineResult.getResultJson();
			Test test = new Gson().fromJson(resultJson, Test.class);
			addResultToTicketItem(test, ticketItem);
			ticketItem.putMachineResultTime(machineResult.getResultDate());
		}
		else {
			ticketItem.putMachineResultReceived(false);
			ticketItem.putMachineResultTime(null);
		}
	}

	public synchronized List<TestItem> getMenuItemTestItems(String menuItemId) throws Exception {
		OroMultitenantConnectionProvider.initTenant(storeId);
		
		try (Session session = MenuItemDAO.getInstance().createNewSession()) {
			MenuItem menuItem = MenuItemDAO.getInstance().loadInitialized(menuItemId, session);

			List<TestItem> testItems = menuItem.getTestItems();
			if (testItems != null) {
				int sortorder = 1;
				for (TestItem testItem : testItems) {
					if (testItem == null) {
						continue;
					}
					testItem.setSortOrder(sortorder);
					sortorder++;
				}
			}
			return testItems;
		}
	}

	public static String cleanTestName(String testName) {
		//String[] chars = { "%", "#", "^" };
		String[] chars = { "^" }; //$NON-NLS-1$
		//System.out.println("cleaning: " + testName);
		for (int i = 0; i < chars.length; i++) {
			int index = testName.indexOf(chars[i]);
			if (index > 0) {
				return testName.substring(0, index);
			}
		}

		return testName;
	}

	public synchronized void convertMachineResultToItemUnit(TestItem testItem, Result result) {
		try {
			String unitid = testItem.getUnit();
			if (org.apache.commons.lang.StringUtils.isBlank(unitid)) {
				return;
			}
			InventoryUnit inventoryUnit = InventoryUnitDAO.getInstance().get(unitid, session);
			if (inventoryUnit == null) {
				return;
			}
			if (inventoryUnit.getConversionRate() > 0) {
				double newRsult = Double.valueOf(result.result);

				String inventoryUnitConversionRuleName = inventoryUnit.getProperty(InventoryUnit.PROP_CONVERSION_RULE,
						InventoryUnitConversionRule.DIVISION.getName());
				InventoryUnitConversionRule inventoryUnitConversionRule = InventoryUnitConversionRule.fromName(inventoryUnitConversionRuleName);
				if (inventoryUnitConversionRule == InventoryUnitConversionRule.MULTIPLICATION) {
					newRsult = newRsult * 1D / inventoryUnit.getConversionRate();
				}
				else {
					newRsult = newRsult / inventoryUnit.getConversionRate();
				}
				result.result = String.valueOf(newRsult);
			}
		} catch (Exception e) {
			//ignore
		}
	}

	public synchronized void updateUnit(Map<String, InventoryUnit> unitsMap, Result result, String unitName) throws Exception {
		if (StringUtils.isNotBlank(unitName)) {
			InventoryUnit inventoryUnit = unitsMap.get(unitName);
			if (inventoryUnit == null) {
				inventoryUnit = InventoryUnitDAO.getInstance().findByName(unitName, session);
				if (inventoryUnit == null) {
					inventoryUnit = new InventoryUnit();
					inventoryUnit.setConversionRate(1.0);
					inventoryUnit.setName(unitName);
					if (unitName.length() > 10) {
						inventoryUnit.setCode(String.format("%.10s", unitName)); //$NON-NLS-1$
					}
					else {
						inventoryUnit.setCode(unitName);
					}

					InventoryUnitDAO.getInstance().save(inventoryUnit, session);
				}
				unitsMap.put(unitName, inventoryUnit);
			}

			result.unit = inventoryUnit.getId();
			result.inventoryUnit = inventoryUnit;
		}
	}

	public synchronized Test saveMachineResult(Test test, String postData) throws Exception {
		String sampleId = test.sampleId;
		if (StringUtils.isBlank(sampleId)) {
			PosLog.error(HL7Util.class, "Sample id not found for machine data");
			sampleId = "00000";
		}
		PosLog.debug(HL7Util.class, String.format("Sample id: %s", sampleId));

		MachineResult machineResult = new MachineResult();
		machineResult.setStoreId(storeId);

		if (StringUtils.isNotBlank(test.date)) {
			machineResult.setResultDate(convertToMithicFormatterDate(test.date, test.time));
		}
		else {
			machineResult.setResultDate(new Date());
		}
		machineResult.setSampleId(sampleId);
		machineResult.setOutletId(outletId);
		machineResult.setResultJson(new Gson().toJson(test));
		machineResult.putMachineRawData(postData);

		MachineResultDAO.getInstance().saveOrUpdate(machineResult, session);

		PosLog.debug(HL7Util.class, "Result has been stored.");

		TicketItem ticketItemByID = null;

		boolean isSampleIdMatchWithInvoice = GlobalConfigDAO.getInstance().isSampleIdMatchWithInvoice(session);
		if (isSampleIdMatchWithInvoice) {
			Ticket ticket = TicketDAO.getInstance().loadFullTicket(sampleId, outletId, session);
			if (ticket != null) {
				List<TicketItem> ticketItems = ticket.getTicketItems();
				if (ticketItems != null) {
					for (TicketItem ticketItem : ticketItems) {
						if (ticketItem.getName().toLowerCase().startsWith("cbc")) {
							ticketItemByID = ticketItem;
							PosLog.debug(HL7Util.class, String.format("CBC item %s found from invoice %s.", ticketItem.getId(), sampleId));
							break;
						}
					}
				}
			}

			if (ticketItemByID == null) {
				PosLog.debug(HL7Util.class, String.format("CBC item not found from invoice %s.", sampleId));
				return test;
			}
		}
		else {
			ticketItemByID = TicketItemDAO.getInstance().findByLabTest(sampleId, session);
			if (ticketItemByID == null) {
				PosLog.debug(HL7Util.class, String.format("Ticket item for sample %s not found", sampleId));
				return test;
			}
		}

		addResultToTicketItem(test, ticketItemByID);

		TicketItemDAO.getInstance().saveOrUpdate(ticketItemByID, session);

		return test;
	}

	private Date convertToMithicFormatterDate(String date, String time) {
		if (StringUtils.isNotBlank(date)) {
			String dateTimeString = date + " " + time;
			try {
				TimeZone timeZone = TimeZone.getTimeZone("Asia/Dhaka"); //$NON-NLS-1$
				//				if (StringUtils.isNotBlank(outlet.getProperty(WorkingHours.TIME_ZONE))) {
				//					timeZone = TimeZone.getTimeZone(outlet.getProperty(WorkingHours.TIME_ZONE));
				//				}
				//				else {
				//					timeZone = TimeZone.getTimeZone("Asia/Dhaka"); //$NON-NLS-1$
				//				}
				mythicFormatter.setTimeZone(timeZone);
				return mythicFormatter.parse(dateTimeString.trim());
			} catch (Exception e) {
			}
		}
		return new Date();
	}

	public Session getSession() {
		return session;
	}

	public void setSession(Session session) {
		this.session = session;
	}

	public String getStoreId() {
		return storeId;
	}

	public void setStoreId(String storeId) {
		this.storeId = storeId;
	}

	public String getOutletId() {
		return outletId;
	}

	public void setOutletId(String outletId) {
		this.outletId = outletId;
	}

}
