package com.floreantpos.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import javax.script.SimpleScriptContext;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;

import com.floreantpos.PosException;
import com.floreantpos.model.TestItem;

public class TestResultUtil {

	public static void doCheckFormulaValidation(String formulaStr, List<TestItem> testItems) throws PosException {
		getResult(formulaStr, getVariableWithValueMap(formulaStr, testItems));
	}

	public static Map<String, TestItemDto> getVariableWithValueMap(String formulaStr, List<TestItem> testItems) throws PosException {
		return getVariableWithValueMap(formulaStr, testItems, null);
	}

	public static Map<String, TestItemDto> getVariableWithValueMap(String formulaStr, List<TestItem> testItems, Object defaultValue) throws PosException {
		Map<String, TestItemDto> machineCodesWithValues = createTestItemDtoMap(testItems);
		return doCalculateResult(formulaStr, machineCodesWithValues, testItems);
	}

	public static Map<String, TestItemDto> createTestItemDtoMap(List<TestItem> testItems) {
		TestItemMapper mapper = new TestItemMapper();
		Map<String, TestItemDto> machineCodesWithValues = testItems.stream().filter(t -> StringUtils.isNotBlank(t.getMachineCode()))
				.collect(Collectors.toMap(key -> key.getMachineCode(), value -> mapper.apply(value), (oldValue, newValue) -> oldValue));
		return machineCodesWithValues;
	}

	public static Map<String, TestItemDto> doCalculateResult(String formulaStr, Map<String, TestItemDto> machineCodesWithValues, List<TestItem> testItems)
			throws PosException {
		String formulaStr2 = formulaStr;

		Pattern pattern = Pattern.compile("round", Pattern.CASE_INSENSITIVE);
		Matcher matcher = pattern.matcher(formulaStr2);
		formulaStr2 = matcher.replaceAll("");

		pattern = Pattern.compile("floor", Pattern.CASE_INSENSITIVE);
		matcher = pattern.matcher(formulaStr2);
		formulaStr2 = matcher.replaceAll("");

		pattern = Pattern.compile("ceil", Pattern.CASE_INSENSITIVE);
		matcher = pattern.matcher(formulaStr2);
		formulaStr2 = matcher.replaceAll("");

		Map<String, TestItemDto> variableMachineCodeList = new HashMap<>();

		formulaStr2 = formulaStr2.replaceAll("[\\(\\)]", StringUtils.EMPTY); //$NON-NLS-1$

		String regex = "\\[([^\\[\\]]+)\\]";
		pattern = Pattern.compile(regex);
		matcher = pattern.matcher(formulaStr2);

		while (matcher.find()) {
			String machineCode = matcher.group(1);
			machineCode = machineCode.trim();
//			if (!machineCodesWithValues.containsKey(machineCode)) {
//				throw new PosException("Machine code '" + machineCode + "' not found");
//			}

			TestItemDto itemDto = (TestItemDto) machineCodesWithValues.get(machineCode);
			if (itemDto == null) {
				if (testItems != null) {
					for (TestItem tItem : testItems) {
						if (StringUtils.isNotBlank(tItem.getMachineCode()) && tItem.getMachineCode().equals(machineCode)) {
							itemDto = new TestItemDto(tItem.getName(), machineCode, tItem.getFormulaString(), tItem.getResult(), tItem.isFormulaEnable(),
									false);
							break;
						}
					}
				}
			}

			if (itemDto != null && itemDto.isFormulaEnable()) {
				String formulaString = itemDto.getFormula();
				if (!(itemDto != null && itemDto.isResultCalculated())) {
					Object result = getResult(formulaString, machineCodesWithValues);
					if (itemDto != null) {
						itemDto.setResultCalculated(true);
						itemDto.setResult(result);
						machineCodesWithValues.put(machineCode, itemDto);
					}
				}
			}

			variableMachineCodeList.put(matcher.group(), machineCodesWithValues.get(machineCode));
		}
		return variableMachineCodeList;
	}

	//	public static Object getResult(String formulaStr, List<TestItem> testItems) throws PosException {
	//		return getResult(formulaStr, getVariableWithValueMap(formulaStr, testItems));
	//	}

	public static Object getResult(String formulaStr, Map<String, TestItemDto> variableWithValueMap) throws PosException {
		SimpleScriptContext context = new SimpleScriptContext();
		for (Map.Entry<String, TestItemDto> entry : variableWithValueMap.entrySet()) {
			String key = entry.getKey();
			TestItemDto itemDto = entry.getValue();
			Object val = itemDto.result;
			//formulaStr = formulaStr.replaceAll(key, val.toString());
			if (val != null && val instanceof String) {
				String value = ((String) val).trim();
				if (value.startsWith("[") && value.endsWith("]")) {
					val = value.trim().replaceAll("[\\[\\]]", StringUtils.EMPTY); //$NON-NLS-1$
				}

				if (NumberUtils.isNumber(value)) {
					val = NumberUtils.toDouble(value);
				}
			}
			if (key != null && key instanceof String) {
				String string = ((String) key).trim();
				if (string.startsWith("[") && string.endsWith("]")) {
					key = string.trim().replaceAll("[\\[\\]]", StringUtils.EMPTY); //$NON-NLS-1$
				}
				key = replaceAllSpecialChar(key); //$NON-NLS-1$
			}
			context.setAttribute(key, val, ScriptContext.ENGINE_SCOPE);
		}

		formulaStr = getFormulaStringEscaped(formulaStr);

		Pattern pattern;
		Matcher matcher;
		if (formulaStr.toLowerCase().contains("round")) {
			pattern = Pattern.compile("round", Pattern.CASE_INSENSITIVE);
			matcher = pattern.matcher(formulaStr);
			formulaStr = matcher.replaceAll(isPresentNonNumberValue("round", formulaStr, context) ? StringUtils.EMPTY : "Math.round");
		}

		if (formulaStr.toLowerCase().contains("floor")) {
			pattern = Pattern.compile("floor", Pattern.CASE_INSENSITIVE);
			matcher = pattern.matcher(formulaStr);
			formulaStr = matcher.replaceAll(isPresentNonNumberValue("floor", formulaStr, context) ? StringUtils.EMPTY : "Math.floor");
		}

		if (formulaStr.toLowerCase().contains("ceil")) {
			pattern = Pattern.compile("ceil", Pattern.CASE_INSENSITIVE);
			matcher = pattern.matcher(formulaStr);
			formulaStr = matcher.replaceAll(isPresentNonNumberValue("ceil", formulaStr, context) ? StringUtils.EMPTY : "Math.ceil");
		}

		try {
			ScriptEngine engineByName = new ScriptEngineManager().getEngineByName("JS");
			Object result = engineByName.eval(formulaStr, context);
			if (result instanceof jdk.nashorn.api.scripting.ScriptObjectMirror) {
				jdk.nashorn.api.scripting.ScriptObjectMirror mirror = (jdk.nashorn.api.scripting.ScriptObjectMirror) result;
				for (String key : mirror.keySet()) {
					return mirror.get(key);
				}
			}
			else if (result != null && (result.equals(Double.POSITIVE_INFINITY) || result.equals(Double.NEGATIVE_INFINITY))) {
				return 0;
			}
			return result; //$NON-NLS-1$
		} catch (ScriptException e) {
			throw new PosException(String.format("Invalid formula %s", formulaStr));
		}
	}

	private static boolean isPresentNonNumberValue(String functionName, String formulaStr, SimpleScriptContext context) {
		String regex = functionName + "\\((.*?)\\)";

		Pattern pattern = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
		Matcher matcher = pattern.matcher(formulaStr);

		Set<String> variables = new HashSet<>();
		while (matcher.find()) {
			String innerString = matcher.group(1);

			pattern = Pattern.compile("([+\\-*/]+)");
			if (pattern.matcher(innerString).find()) {
				throw new PosException("Invalid formula");
			}
			variables.add(innerString);
		}

		if (variables.isEmpty()) {
			throw new PosException("Invalid formula");
		}

		boolean foundNonNumberValue = false;
		pattern = Pattern.compile(functionName, Pattern.CASE_INSENSITIVE);
		matcher = pattern.matcher(formulaStr);
		for (String variable : variables) {
			Object attr = context.getAttribute(variable.trim().replaceAll("[\\[\\]]", StringUtils.EMPTY));
			if (attr == null || (attr instanceof String && !NumberUtils.isNumber((String) attr))) {
				formulaStr = matcher.replaceAll(StringUtils.EMPTY);
				foundNonNumberValue = true;
				break;
			}
		}

		return foundNonNumberValue;
	}

	private static String getFormulaStringEscaped(String formulaStr) {
		String regexPattern = "\\[([^\\[\\]]+)\\]";
		Pattern pattern = Pattern.compile(regexPattern);
		Matcher matcher = pattern.matcher(formulaStr);

		while (matcher.find()) {
			String group = matcher.group(1);
			formulaStr = formulaStr.replace(group, replaceAllSpecialChar(group));
		}

		formulaStr = formulaStr.trim().replaceAll("[\\[\\]]", StringUtils.EMPTY); //$NON-NLS-1$
		return formulaStr;
	}

	private static String replaceAllSpecialChar(String str) {
		if (str == null) {
			return null;
		}
		// @formatter:off
		
		return str.replace("#", "HASH")
				.replace("%", "PERCENT")
				.replaceAll("[^a-zA-Z0-9]", "_");
		
		// @formatter:on
	}

	public static class TestItemDto {
		private String name;
		private String code;
		private String formula;
		private Object result;
		private boolean isFormulaEnable;
		private boolean isResultCalculated;

		public TestItemDto(String name, String code, String formula, Object result, boolean isFormulaEnable, boolean isResultCalculated) {
			this.name = name;
			this.code = code;
			this.formula = formula;
			this.result = result;
			this.setFormulaEnable(isFormulaEnable);
			this.isResultCalculated = isResultCalculated;
		}

		@Override
		public boolean equals(Object obj) {
			TestItemDto dto = (TestItemDto) obj;
			return Objects.equals(name, dto.name) || Objects.equals(code, dto.code);
		}

		public String getFormula() {
			return formula;
		}

		public void setFormula(String formula) {
			this.formula = formula;
		}

		public Object getResult() {
			return result;
		}

		public void setResult(Object result) {
			this.result = result;
		}

		public boolean isFormulaEnable() {
			return isFormulaEnable;
		}

		public void setFormulaEnable(boolean isFormulaEnable) {
			this.isFormulaEnable = isFormulaEnable;
		}

		public boolean isResultCalculated() {
			return isResultCalculated;
		}

		public void setResultCalculated(boolean isResultCalculated) {
			this.isResultCalculated = isResultCalculated;
		}

	}

	private static class TestItemMapper implements Function<TestItem, TestItemDto> {

		@Override
		public TestItemDto apply(TestItem t) {
			return new TestItemDto(t.getName(), t.getMachineCode(), t.isFormulaEnable() ? t.getFormulaString() : null, t.getResult(), t.isFormulaEnable(),
					false);
		}

	}

	public static void main(String[] args) {
		//invalid formula
		String formula = "floor(a + b + ceil([ +a#b-c%]))";

		//valid formula
		formula = "a + b + ceil([+a#b-c@~&^`_!%])";

		Map<String, Object> maps = new HashMap<>();
		maps.put("a", 1.6);
		maps.put("b", "3");
		maps.put("[+a#b-c@~&^`_!%]", "2.2");

		//Object result = TestResultUtil.getResult(formula, maps);
		//System.out.println("Result: " + result);
	}
}
