package com.floreantpos.model;

import java.text.ParseException;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseQuotation;
import com.floreantpos.model.util.DateUtil;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.orocube.rest.service.ServiceUtils;

@XmlRootElement
@JsonIgnoreProperties(ignoreUnknown = true)
@XmlAccessorType(XmlAccessType.FIELD)
public class Quotation extends BaseQuotation implements PropertyContainer2 {
	private static final long serialVersionUID = 1L;
	private transient JsonObject propertiesContainer;

	public static final int ORDER_PENDING = 0;
	public static final int ORDER_SENT = 1;
	public static final int ORDER_CONFIMED = 2;
	public static final int APPROVED = 3;
	public static final int DECLINED = 4;
	public static final int RESENT = 5;

	public static final String[] ORDER_STATUS = { "Open", "Sent", "Confirmed", "Approved", "Declined", "Re-sent" }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
	private Ticket ticket;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public Quotation() {
	}

	/**
	 * Constructor for primary key
	 */
	public Quotation(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@XmlTransient
	@JsonIgnore
	public Ticket getTicket() {
		String ticketJson = super.getTicketJson();
		if (ticket == null) {
			if (StringUtils.isNotBlank(ticketJson)) {
				try {
					ticket = ServiceUtils.convertJsonToTicket(ticketJson);
					if (StringUtils.isBlank(ticket.getId())) {
						ticket.setOrderStatus(getOrderStatus());
						ticket.putCustomerEditable(!isMailSent());
					}
					return ticket;
				} catch (Exception e) {
					PosLog.error(getClass(), e);
				}
			}
			return new Ticket();
		}
		return ticket;
	}

	public void setTicket(Ticket ticket) {
		this.ticket = ticket;
	}

	@Override
	public void setOrderStatus(Integer orderStatus) {
		if (ORDER_CONFIMED != orderStatus && (getOrderStatus() > orderStatus && getOrderStatus() != RESENT)) {
			return;
		}
		super.setOrderStatus(orderStatus);
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	public List<Comment> getComments() {
		String commentsJson = getProperty("comments", "[]"); //$NON-NLS-1$ //$NON-NLS-2$
		List<Comment> comments = new Gson().fromJson(commentsJson, new TypeToken<List<Comment>>() {
		}.getType());
		comments.sort(Comparator.comparing(Comment::getTime));
		return comments;
	}

	public void putComments(List<Comment> comments) {
		addProperty("comments", new Gson().toJson(comments)); //$NON-NLS-1$
	}

	public void addComment(Comment comment) {
		List<Comment> comments = getComments();
		comments.add(comment);
		addProperty("comments", new Gson().toJson(comments)); //$NON-NLS-1$
	}

	public void putApproved(boolean b) {
		addProperty("isApproved", String.valueOf(b)); //$NON-NLS-1$
	}

	public boolean isApproved() {
		return getBooleanProperty("isApproved", false); //$NON-NLS-1$
	}

	public void putDeclient(boolean b) {
		addProperty("isDeclient", String.valueOf(b)); //$NON-NLS-1$
	}

	public boolean isDeclient() {
		return getBooleanProperty("isDeclient", false); //$NON-NLS-1$
	}

	public Date getEmailSentDate() {
		try {
			String dateStr = getProperty("email.sent.date"); //$NON-NLS-1$
			if (StringUtils.isBlank(dateStr)) {
				return null;
			}
			return DateUtil.parseSyncTime(dateStr);
		} catch (ParseException e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public void putEmailSentDate(Date date) {
		addProperty("email.sent.date", DateUtil.formatSyncTime(date)); //$NON-NLS-1$
	}

	public boolean isMailSent() {
		return getEmailSentDate() != null || getOrderStatus() >= ORDER_SENT;
	}

	@Override
	public Date getDueDate() {
		Date dueDate = super.getDueDate();
		if (dueDate != null) {
			return DateUtil.endOfDay(dueDate);
		}
		return dueDate;
	}
}