/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.POSConstants;
import com.floreantpos.PosLog;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.ActionHistory.ActionType;
import com.floreantpos.model.Discount;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderItem;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketDiscount;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemDiscount;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ActionHistoryDAO extends BaseActionHistoryDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ActionHistoryDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveOrUpdateActionHistory(List<ActionHistory> actionHistoryList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (actionHistoryList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<ActionHistory> iterator = actionHistoryList.iterator(); iterator.hasNext();) {
				ActionHistory item = (ActionHistory) iterator.next();
				ActionHistory existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getActionName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public static void saveHistory(User performer, Ticket ticket, PosTransaction posTransaction, String actionName, String description) {
		Transaction transaction = null;
		try (Session session = ActionHistoryDAO.getInstance().createNewSession()) {
			transaction = session.beginTransaction();
			saveHistory(performer, ticket, posTransaction, actionName, description, session);
			transaction.commit();
		} catch (Exception e) {
			if (transaction != null) {
				transaction.rollback();
			}
			LogFactory.getLog(ActionHistoryDAO.class).error("Error occured while trying to save action history", e); //$NON-NLS-1$
		}
	}

	public static void saveHistory(Ticket ticket, String actionName, String description) {
		saveHistory(DataProvider.get().getCurrentUser(), ticket, actionName, description);
	}

	public static void saveHistory(User performer, Ticket ticket, String actionName, String description) {
		saveHistory(performer, ticket, null, actionName, description);
	}

	public static void saveHistory(User performer, String actionName, String description) {
		saveHistory(performer, null, null, actionName, description);
	}

	public static void saveHistory(String actionName, String description) {
		saveHistory(DataProvider.get().getCurrentUser(), null, null, actionName, description);
	}

	public static void saveHistory(User performer, String actionName, String description, Session session) {
		saveHistory(performer, null, null, actionName, description, session);
	}

	public static void saveHistory(String actionName, String description, Session session) {
		//TODO:
		saveHistory(null, null, null, actionName, description, session);
	}

	public static void saveHistory(User performer, Ticket ticket, PosTransaction transaction, String actionName, String description, Session session) {
		ActionHistory history = new ActionHistory();
		history.setActionName(actionName);
		history.setDescription(description);
		history.setPerformer(performer);
		history.setActionTime(StoreDAO.getServerTimestamp());
		history.setOutletId(DataProvider.get().getOutletId());

		if (ticket != null) {
			history.setTicketId(ticket.getId());
		}
		if (transaction != null) {
			history.setTransactionId(transaction.getId());
		}

		ActionHistoryDAO.getInstance().save(history, session);
	}

	public void addInventoryTransactionActionHistory(InventoryTransaction inventoryTransaction, User performer, Outlet fromOutlet,
			InventoryLocation fromLocation, Outlet toOutlet, InventoryLocation toLocation) {
		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();
			addInventoryTransactionActionHistory(inventoryTransaction, performer, fromOutlet, fromLocation, toOutlet, toLocation, session);
			transaction.commit();
		}
	}

	public void addInventoryTransactionActionHistory(InventoryTransaction inventoryTransaction, User performer, Outlet fromOutlet,
			InventoryLocation fromLocation, Outlet toOutlet, InventoryLocation toLocation, Session session) {
		ActionHistory history = new ActionHistory();
		history.setActionTime(new Date());

		String description = String.format("Stock reason: %s, Item Id: %s, Item name: %s, Quantity: %s , unit_id: %s", //$NON-NLS-1$
				inventoryTransaction.getReason() != null ? inventoryTransaction.getReason() : "", //$NON-NLS-1$
				inventoryTransaction.getMenuItem().getId(), inventoryTransaction.getItemName(),
				NumberUtil.trimDecilamIfNotNeeded(inventoryTransaction.getQuantity()), inventoryTransaction.getUnit());

		StringBuilder builder = new StringBuilder(description);
		if (fromOutlet != null) {
			builder.append(", from_outlet id: " + fromOutlet.getId()); //$NON-NLS-1$
			builder.append(", from_outlet name: " + fromOutlet.getName()); //$NON-NLS-1$
		}
		if (fromLocation != null) {
			builder.append(", from_location id: " + fromLocation.getId()); //$NON-NLS-1$
			builder.append(", from_location name: " + fromLocation.getName()); //$NON-NLS-1$
		}
		if (toOutlet != null) {
			builder.append(", to_outlet id: " + toOutlet.getId()); //$NON-NLS-1$
			builder.append(", to_outlet name: " + toOutlet.getName()); //$NON-NLS-1$
		}
		if (toLocation != null) {
			builder.append(", to_location id: " + toLocation.getId()); //$NON-NLS-1$
			builder.append(", to_location name: " + toLocation.getName()); //$NON-NLS-1$
		}

		history.setActionName("STOCK " + inventoryTransaction.getTransactionType().getName()); //$NON-NLS-1$
		history.setPerformer(performer);
		history.setDescription(builder.toString());
		history.setActionTime(StoreDAO.getServerTimestamp());
		history.setTicketId(inventoryTransaction.getTicketId());
		history.setOutletId(inventoryTransaction.getOutletId());
		history.setTransactionId(inventoryTransaction.getId());
		ActionHistoryDAO.getInstance().save(history, session);
	}

	private static void save(Map<String, ActionHistory> historyMap) {
		ActionHistoryDAO instance2 = getInstance();
		Transaction transaction = null;
		try (Session session = instance2.createNewSession()) {
			transaction = session.beginTransaction();
			for (ActionHistory actionHistory : historyMap.values()) {
				instance2.save(actionHistory, session);
			}
			transaction.commit();
		} catch (Exception e) {
			if (transaction != null) {
				transaction.rollback();
			}
		} finally {
			historyMap.clear();
		}
	}

	public void performActionHistorySaveOperation(Ticket ticket) {
		performActionHistorySaveOperation(ticket, false);
	}

	public void performActionHistorySaveOperation(Ticket ticket, boolean newOrder) {
		performSave(ticket, newOrder);
	}

	public void performSave(Ticket ticket, boolean newOrder) {
		if (ticket == null) {
			return;
		}
		try {
			StringBuilder sb = new StringBuilder();
			sb.append(POSConstants.CHECK + "#:" + ticket.getId()); //$NON-NLS-1$
			sb.append(", Server: " + ticket.getOwnerId() + "/" + ticket.getOwnerName()); //$NON-NLS-1$ //$NON-NLS-2$

			String prescriptionNo = ticket.getPrescriptionNo();
			if (StringUtils.isNotBlank(prescriptionNo)) {
				sb.append(", Prescription No: " + prescriptionNo); //$NON-NLS-1$ //$NON-NLS-2$
			}
			//ActionHistoryDAO.saveHistory(ActionHistory.SAVE_CHECK, sb.toString());
			ActionHistoryDAO.populateEventsDescription(ticket);
			ActionHistoryDAO.save(ticket.getEvents());
		} catch (Exception e) {
			PosLog.error(getReferenceClass(), e);
		}
	}

	public void performPurchaseOrderActionHistorySaveOperation(PurchaseOrder purchaseOrder, boolean newOrder) {
		performPurchaseOrderSave(purchaseOrder, newOrder);
	}

	public void performPurchaseOrderSave(PurchaseOrder purchaseOrder, boolean newOrder) {
		if (purchaseOrder == null) {
			return;
		}
		try {
			StringBuilder sb = new StringBuilder();
			sb.append("Purchase Order#:" + purchaseOrder.getId()); //$NON-NLS-1$
			String poNO = purchaseOrder.getOrderId();
			if (StringUtils.isNotBlank(poNO)) {
				sb.append(", PO No: " + poNO); //$NON-NLS-1$
			}
			ActionHistoryDAO.populatePurchaseEventsDescription(purchaseOrder);
			ActionHistoryDAO.save(purchaseOrder.getEvents());
		} catch (Exception e) {
			PosLog.error(getReferenceClass(), e);
		}
	}

	public static void createLogJournal(PurchaseOrder purchaseOrder, String actionName) {
		StringBuilder sb = new StringBuilder();
		sb.append("P.O id: " + purchaseOrder.getId()); //$NON-NLS-1$
		sb.append(", create date: " + purchaseOrder.getCreatedDateAsString()); //$NON-NLS-1$
		sb.append(", due date: " + purchaseOrder.getDueDateAsString()); //$NON-NLS-1$
		sb.append(", to outlet: " + purchaseOrder.getInventoryLocation()); //$NON-NLS-1$
		sb.append(", vendor: " + purchaseOrder.getVendor()); //$NON-NLS-1$

		StringBuilder sbItems = new StringBuilder();
		List<PurchaseOrderItem> items = purchaseOrder.getOrderItems();
		for (Iterator iterator = items.iterator(); iterator.hasNext();) {
			PurchaseOrderItem item = (PurchaseOrderItem) iterator.next();
			sbItems.append("{ item_id: " + item.getId()); //$NON-NLS-1$
			sbItems.append(", item_name: " + item.getName()); //$NON-NLS-1$
			sbItems.append(", order_qty: " + item.getItemQuantity()); //$NON-NLS-1$
			if (item.getQuantityReceived() > 0) {
				sbItems.append(", received_qty: " + item.getQuantityReceived()); //$NON-NLS-1$
			}
			sbItems.append("}"); //$NON-NLS-1$
			if (iterator.hasNext()) {
				sbItems.append(","); //$NON-NLS-1$
			}
		}
		sb.append(", items: " + sbItems.toString()); //$NON-NLS-1$

		ActionHistoryDAO.saveHistory(actionName, sb.toString());
	}

	private static void populateEventsDescription(Ticket ticket) {
		if (ticket == null) {
			return;
		}
		Map<String, ActionHistory> events = ticket.getEvents();
		if (events.isEmpty()) {
			return;
		}
		Collection<ActionHistory> histories = events.values();
		histories.forEach(history -> {
			history.setTicketId(ticket.getId());
			Object objectReference = history.getObjectReference();
			String description = history.getDescription();
			if (objectReference instanceof TicketDiscount) {
				TicketDiscount ticketDiscount = (TicketDiscount) objectReference;
				if (ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY) {
					description = String.format("Order id : %s, Loyalty points : %s", ticketDiscount.getTicket().getId(), ticketDiscount.getLoyaltyPoint()); //$NON-NLS-1$
				}
				else {
					description = String.format("Order id : %s, Discount id : %s, Coupon quantity : %s, %s", ticketDiscount.getTicket().getId(), //$NON-NLS-1$
							ticketDiscount.getDiscountId(), ticketDiscount.getCouponQuantity(), history.getDescription());
				}
			}
			else if (objectReference instanceof TicketItemDiscount) {
				TicketItemDiscount itemDiscount = (TicketItemDiscount) objectReference;
				TicketItem ticketItem = itemDiscount.getTicketItem();
				description = String.format("Order id : %s, Item id : %s, Discount id : %s, Coupon quantity : %s, %s, ", ticket != null ? ticket.getId() : "", //$NON-NLS-2$
						ticketItem != null ? ticketItem.getId() : "", itemDiscount.getDiscountId(), itemDiscount.getCouponQuantity(), history.getDescription());
				if (StringUtils.isBlank(ticketItem.getId())) {
					description += ", Menu item id: " + ticketItem.getMenuItemId();
					description += ", MenuItem name: " + ticketItem.getMenuItem().getName();
				}
			}
			else if (history.getActionName() == ActionHistory.TAX_EXEMPT && ticket.isTaxExempt()) {
				String exemptId = ticket.getProperty("tax_exempt_id");
				description = String.format("Order id : %s, Exempt id : %s", ticket.getId(), exemptId);
			}
			else if (history.getActionName() == ActionHistory.NEW_CHECK) {
				description = POSConstants.TICKET_ID + "#:" + ticket.getId(); //$NON-NLS-1$
			}
			history.setDescription(description);
		});
	}

	private static void populatePurchaseEventsDescription(PurchaseOrder purchaseOrder) {
		if (purchaseOrder == null) {
			return;
		}
		Map<String, ActionHistory> events = purchaseOrder.getEvents();
		if (events.isEmpty()) {
			return;
		}
		Collection<ActionHistory> histories = events.values();
		histories.forEach(history -> {
			//history.setTicketId(purchaseOrder.getId());
			String description = history.getDescription();
			if (history.getActionName() == ActionHistory.NEW_PURCHASE_ORDER) {
				description = "Purchase order Id" + "#:" + purchaseOrder.getId(); //$NON-NLS-1$
			}
			history.setDescription(description);
		});
	}

	public static void addDiscountRemovedActionHistory(Ticket ticket, TicketDiscount ticketDiscount) {
		if (ticket == null) {
			return;
		}
		String key = String.valueOf(ticketDiscount.hashCode());
		if (ticket.getEvents().containsKey(key)) {
			ticket.getEvents().remove(key);
		}
		else {
			String action = ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY ? ActionHistory.LOYALTY_REMOVED : ActionHistory.TICKET_DISCOUNT_REMOVED;
			Double afterAmount = ticket.getTotalAmount();
			Double discountAmount = ticket.getDiscountAmount();
			ticket.calculatePrice();
			String description = "Amount before: " + afterAmount;
			//			description += ", Discount id: " + ticketDiscount.getDiscountId();
			description += ", Discount Name: " + ticketDiscount.getName();
			description += ", Discount Value: " + ticketDiscount.getValue();
			description += ", Discount amount: " + discountAmount;
			description += ", Amount after: " + ticket.getTotalAmount();
			addActionHistory(ticket, key, action, description, ticketDiscount);
		}
	}

	public static void addDiscountAddedActionHistory(Ticket ticket, TicketDiscount ticketDiscount) {
		if (ticket == null) {
			return;
		}
		String key = String.valueOf(ticketDiscount.hashCode());
		String action = ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY ? ActionHistory.LOYALTY_ADDED : ActionHistory.TICKET_DISCOUNT_ADDED;
		Double afterAmount = ticket.getTotalAmount();
		ticket.calculatePrice();
		String description = "Amount before: " + afterAmount;
		//		description += ", Discount id: " + ticketDiscount.getDiscountId();
		description += ", Discount Name: " + ticketDiscount.getName();
		description += ", Discount Value: " + ticketDiscount.getValue();
		description += ", Ticket discount amount: " + ticket.getTicketDiscountAmount();
		description += ", Total discount amount: " + ticket.getDiscountAmount();
		description += ", Amount after: " + ticket.getTotalAmount();
		addActionHistory(ticket, key, action, description, ticketDiscount);
	}

	public static void addSetGratuityActionHistory(Ticket ticket, Gratuity gratuity) {
		if (ticket == null) {
			return;
		}
		if (ticket.getId() == null) {
			return;
		}
		String key, action;
		key = action = ActionHistory.GRATUITY_ADDED;
		String description = String.format("Order id : %s, Gratuity amount : %s", ticket.getId(), gratuity == null ? 0 : gratuity.getAmount()); //$NON-NLS-1$
		addActionHistory(ticket, key, action, description, gratuity);
	}

	public static void addSalesAreaAddedActionHistory(Ticket ticket, SalesArea salesArea) {
		if (ticket == null) {
			return;
		}
		if (ticket.getId() == null) {
			return;
		}
		if (salesArea == null) {
			return;
		}
		String key, action;
		key = action = ActionHistory.SALES_AREA_CHANGE;
		String description = String.format("Order id : %s, Sales area : %s", ticket.getId(), salesArea.getName()); //$NON-NLS-1$
		addActionHistory(ticket, key, action, description, salesArea);
	}

	public static void addTicketItemDiscountRemovedActionHistory(Ticket ticket, TicketItemDiscount itemDiscount) {
		addTicketItemDiscountRemovedActionHistory(ticket, itemDiscount, null);
	}

	public static void addTicketItemDiscountRemovedActionHistory(Ticket ticket, TicketItemDiscount itemDiscount, String description) {
		if (ticket == null) {
			return;
		}
		String key = String.valueOf(itemDiscount.hashCode());
		if (ticket.getEvents().containsKey(key)) {
			ticket.getEvents().remove(key);
		}
		else {
			addActionHistory(ticket, key, ActionHistory.ITEM_DISCOUNT_REMOVED, description, itemDiscount);
		}
	}

	public static void addTicketItemDiscountAddedActionHistory(Ticket ticket, TicketItemDiscount itemDiscount) {
		addTicketItemDiscountAddedActionHistory(ticket, itemDiscount, null);
	}

	public static void addTicketItemDiscountAddedActionHistory(Ticket ticket, TicketItemDiscount itemDiscount, String description) {
		if (ticket == null) {
			return;
		}
		String key = String.valueOf(itemDiscount.hashCode());
		addActionHistory(ticket, key, ActionHistory.ITEM_DISCOUNT_ADDED, description, itemDiscount);
	}

	public static void addItemVoidedActionHistory(Ticket ticket, TicketItem ticketItem, double voidQuantity) {
		if (ticket == null || ticket.getId() == null || ticketItem == null) {
			return;
		}
		String key = String.valueOf(ticketItem.hashCode());
		String action = ActionHistory.TICKET_ITEM_VOID;
		String description = String.format("Order id : %s, Item id : %s, Void quantity : %s", ticket.getId(), ticketItem.getId(), voidQuantity);
		addActionHistory(ticket, key, action, description, ticketItem);
	}

	public static void addActionHistory(Ticket ticket, String action, String description) {
		addActionHistory(ticket, action, action, description);
	}

	public static void addActionHistory(Ticket ticket, String key, String action, String description) {
		addActionHistory(ticket, key, action, description, null);
	}

	public static void addActionHistory(Ticket ticket, String key, String action, String description, Object objectReference) {
		if (ticket == null) {
			return;
		}
		ActionHistory history = ActionHistory.create(action, description, DataProvider.get().getCurrentUser(), DataProvider.get().getCurrentOutletId());
		history.setObjectReference(objectReference);
		ticket.getEvents().put(key, history);
	}

	public static void saveActionHistory(String action, String description, User user, String outletId) {
		try (Session session = ActionHistoryDAO.getInstance().createNewSession()) {
			saveActionHistory(action, description, user, outletId, session);
		}
	}

	public static void saveActionHistory(String action, String description, User user, String outletId, Session session) {
		ActionHistoryDAO.getInstance().save(ActionHistory.create(action, description, user, outletId), session);
	}

	public static void logJournalForInventoryTrans(InventoryTransaction it) {
		StringBuilder sb = new StringBuilder();
		sb.append("Inventory transaction id: " + it.getId());
		sb.append(", Order id: " + it.getTicketId());
		sb.append(", Menuitem id: " + it.getMenuItem().getId());
		sb.append(", Unit: " + it.getUnit());
		sb.append(", QTY: " + it.getQuantity());

		String actionName = "";
		if (it.getTransactionType() == InventoryTransactionType.OUT) {
			actionName = "Stock OUT";
		}
		else if (it.getTransactionType() == InventoryTransactionType.IN) {
			actionName = "Stock IN";
		}
		saveHistory(actionName, sb.toString());
	}

	public void findActionHistories(PaginationSupport model) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(ActionHistory.class);
			addDeletedFilter(criteria);

			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			criteria.addOrder(Order.desc(ActionHistory.PROP_ACTION_TIME));

			model.setRows(criteria.list());
		}
	}

	public List<ActionHistory> findActionHistoriesByActionType(String outletId, Date startDate, Date endDate, ActionHistory.ActionType actionType) {
		if (actionType == null) {
			return Collections.emptyList();
		}

		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(ActionHistory.class);
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(ActionHistory.PROP_OUTLET_ID, outletId));
			criteria.add(Restrictions.eq(ActionHistory.PROP_ACTION_TYPE, actionType.name()));
			if (startDate != null) {
				criteria.add(Restrictions.ge(ActionHistory.PROP_EVENT_TIME, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(ActionHistory.PROP_EVENT_TIME, endDate));
			}

			criteria.addOrder(Order.desc(ActionHistory.PROP_ACTION_TIME));

			return criteria.list();
		}

	}

	public ActionHistory addEMailActionHistory(Store store, User user, Outlet outlet, String shortDescription, String description, String recipient) {
		ActionHistory history = new ActionHistory();
		history.setActionName(ActionType.EMIL_SENT.getDisplayName());
		history.setActionType(ActionType.EMIL_SENT.name());
		history.setPerformer(user);
		history.setDescription(description);
		history.setShortDescription(shortDescription);
		Date serverTimestamp = StoreDAO.getServerTimestamp();
		history.setActionTime(serverTimestamp);
		history.setEventTime(serverTimestamp);
		history.setRecipient(recipient);
		if (outlet != null) {
			history.setOutletId(outlet.getId());
		}
		if (store != null) {
			history.setCharge(store.getEmailChargeRate());
		}
		return history;
	}

	public void saveSMSActionHistory(ActionType actionType, String shortDescription, String description, String recipient) {
		Outlet outlet = DataProvider.get().getOutlet();
		save(addSMSActionHistory(DataProvider.get().getStore(), DataProvider.get().getCurrentUser(), outlet, actionType, shortDescription, description,
				recipient));
	}

	public ActionHistory addSMSActionHistory(Store store, User user, Outlet outlet, ActionType actionType, String shortDescription, String description,
			String recipient) {
		ActionHistory history = new ActionHistory();
		history.setActionName(ActionHistory.SMS_SENT);
		history.setActionType(actionType.name());
		history.setPerformer(user);
		history.setDescription(description);
		history.setShortDescription(shortDescription);
		Date serverTimestamp = StoreDAO.getServerTimestamp();
		history.setActionTime(serverTimestamp);
		history.setEventTime(serverTimestamp);
		history.setRecipient(recipient);
		if (outlet != null) {
			history.setOutletId(outlet.getId());
		}

		if (store != null) {
			history.setCharge(store.getSmsChargeRate());
		}

		return history;
	}

	public List<ActionHistory> findEmailAndSmsBillingReports(String outletId, Date fromDate, Date toDate, String actionName) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(ActionHistory.class);
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(ActionHistory.PROP_ACTION_NAME, actionName).ignoreCase());

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(ActionHistory.PROP_OUTLET_ID, outletId));
			}

			if (fromDate != null) {
				criteria.add(Restrictions.ge(ActionHistory.PROP_ACTION_TIME, fromDate));
			}

			if (toDate != null) {
				criteria.add(Restrictions.le(ActionHistory.PROP_ACTION_TIME, toDate));
			}
			return criteria.list();
		}
	}

}