package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.Agent;
import com.floreantpos.model.Customer;
import com.floreantpos.swing.PaginationSupport;

public class AgentDAO extends BaseAgentDAO {

	public AgentDAO() {
	}

	@Override
	public List<Agent> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			return criteria.list();
		}
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Agent agent = (Agent) obj;
		if (agent == null) {
			throw new PosException("Agent not found");
		}

		agent.setDeleted(Boolean.TRUE);
		saveOrUpdate(agent, session);
		ActionHistoryDAO.saveHistory(ActionHistory.AGENT_DELETED,
				"Agent name: " + agent.getFirstName() + ", id: " + agent.getId() + ", phone: " + agent.getWorkPhoneNo() + ", email: " + agent.getEmail());
	}

	public void findByPhoneOrName(String searchString, PaginationSupport tableModel) {
		findByPhoneOrName(searchString, null, tableModel);
	}

	public void findByPhoneOrName(String searchString, String status, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Agent.class);

			addDeletedFilter(criteria);
			//			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_LOYALTY_NO, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			if (StringUtils.isNotEmpty(status)) {
				if (status.equals(POSConstants.ACTIVE)) {
					criteria.add(Restrictions.eq(Agent.PROP_ACTIVE, true));
				}
				else if (status.equals(POSConstants.DEACTIVE)) {
					criteria.add(Restrictions.eq(Agent.PROP_ACTIVE, false));
				}
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public Agent findByEmailOrPhone(String emailOrPhone) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, emailOrPhone));
			disjunction.add(Restrictions.ilike(Customer.PROP_EMAIL, emailOrPhone));
			criteria.add(disjunction);

			criteria.setMaxResults(1);
			Agent agent = (Agent) criteria.uniqueResult();
			return agent;
		}
	}

	public Agent findActiveAgent(String id) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Agent.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Agent.PROP_ID, id));
			criteria.add(Restrictions.eq(Agent.PROP_ACTIVE, true));
			return (Agent) criteria.uniqueResult();
		}
	}

}